/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.EOFException;
import java.io.IOException;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.Chore;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HServerAddress;
import org.apache.hadoop.hbase.HServerInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NotServingRegionException;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.catalog.CatalogTracker;
import org.apache.hadoop.hbase.catalog.MetaReader;
import org.apache.hadoop.hbase.catalog.RootLocationEditor;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.executor.EventHandler;
import org.apache.hadoop.hbase.executor.RegionTransitionData;
import org.apache.hadoop.hbase.ipc.ServerNotRunningException;
import org.apache.hadoop.hbase.master.BulkAssigner;
import org.apache.hadoop.hbase.master.LoadBalancer;
import org.apache.hadoop.hbase.master.ServerManager;
import org.apache.hadoop.hbase.master.handler.ClosedRegionHandler;
import org.apache.hadoop.hbase.master.handler.OpenedRegionHandler;
import org.apache.hadoop.hbase.master.handler.ServerShutdownHandler;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.ZKAssign;
import org.apache.hadoop.hbase.zookeeper.ZKTable;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperListener;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperWatcher;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;

public class AssignmentManager
extends ZooKeeperListener {
    private static final Log LOG = LogFactory.getLog(AssignmentManager.class);
    protected Server master;
    private ServerManager serverManager;
    private CatalogTracker catalogTracker;
    private TimeoutMonitor timeoutMonitor;
    private final int maximumAssignmentAttempts;
    final ConcurrentSkipListMap<String, RegionState> regionsInTransition = new ConcurrentSkipListMap();
    final NavigableMap<String, LoadBalancer.RegionPlan> regionPlans = new TreeMap<String, LoadBalancer.RegionPlan>();
    private final ZKTable zkTable;
    private final NavigableMap<HServerInfo, List<HRegionInfo>> servers = new TreeMap<HServerInfo, List<HRegionInfo>>();
    private final SortedMap<HRegionInfo, HServerInfo> regions = new TreeMap<HRegionInfo, HServerInfo>();
    private final org.apache.hadoop.hbase.executor.ExecutorService executorService;

    public AssignmentManager(Server master, ServerManager serverManager, CatalogTracker catalogTracker, org.apache.hadoop.hbase.executor.ExecutorService service) throws KeeperException {
        super(master.getZooKeeper());
        this.master = master;
        this.serverManager = serverManager;
        this.catalogTracker = catalogTracker;
        this.executorService = service;
        Configuration conf = master.getConfiguration();
        this.timeoutMonitor = new TimeoutMonitor(conf.getInt("hbase.master.assignment.timeoutmonitor.period", 10000), master, conf.getInt("hbase.master.assignment.timeoutmonitor.timeout", 180000));
        Threads.setDaemonThreadRunning(this.timeoutMonitor, master.getServerName() + ".timeoutMonitor");
        this.zkTable = new ZKTable(this.master.getZooKeeper());
        this.maximumAssignmentAttempts = this.master.getConfiguration().getInt("hbase.assignment.maximum.attempts", 10);
    }

    public ZKTable getZKTable() {
        return this.zkTable;
    }

    void cleanoutUnassigned() throws IOException, KeeperException {
        ZKAssign.deleteAllNodes(this.watcher);
        ZKUtil.listChildrenAndWatchForNewChildren(this.watcher, this.watcher.assignmentZNode);
    }

    void processFailover() throws KeeperException, IOException, InterruptedException {
        HServerInfo hsi = this.serverManager.getHServerInfo(this.catalogTracker.getMetaLocation());
        this.regionOnline(HRegionInfo.FIRST_META_REGIONINFO, hsi);
        hsi = this.serverManager.getHServerInfo(this.catalogTracker.getRootLocation());
        this.regionOnline(HRegionInfo.ROOT_REGIONINFO, hsi);
        Map<HServerInfo, List<Pair<HRegionInfo, Result>>> deadServers = this.rebuildUserRegions();
        this.processDeadServers(deadServers);
        List<String> nodes = ZKUtil.listChildrenAndWatchForNewChildren(this.watcher, this.watcher.assignmentZNode);
        if (nodes.isEmpty()) {
            LOG.info((Object)"No regions in transition in ZK to process on failover");
            return;
        }
        LOG.info((Object)("Failed-over master needs to process " + nodes.size() + " regions in transition"));
        for (String encodedRegionName : nodes) {
            this.processRegionInTransition(encodedRegionName, null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean processRegionInTransitionAndBlockUntilAssigned(HRegionInfo hri) throws InterruptedException, KeeperException, IOException {
        boolean intransistion = this.processRegionInTransition(hri.getEncodedName(), hri);
        if (!intransistion) {
            return intransistion;
        }
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            while (!this.master.isStopped() && this.regionsInTransition.containsKey(hri.getEncodedName())) {
                this.regionsInTransition.wait();
            }
        }
        return intransistion;
    }

    boolean processRegionInTransition(String encodedRegionName, HRegionInfo regionInfo) throws KeeperException, IOException {
        RegionTransitionData data = ZKAssign.getData(this.watcher, encodedRegionName);
        if (data == null) {
            return false;
        }
        HRegionInfo hri = regionInfo;
        if (hri == null) {
            Pair<HRegionInfo, HServerAddress> p = MetaReader.getRegion(this.catalogTracker, data.getRegionName());
            if (p == null) {
                return false;
            }
            hri = p.getFirst();
        }
        this.processRegionsInTransition(data, hri);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void processRegionsInTransition(RegionTransitionData data, HRegionInfo regionInfo) throws KeeperException {
        String encodedRegionName = regionInfo.getEncodedName();
        LOG.info((Object)("Processing region " + regionInfo.getRegionNameAsString() + " in state " + (Object)((Object)data.getEventType())));
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            switch (data.getEventType()) {
                case RS_ZK_REGION_CLOSING: {
                    this.regionsInTransition.put(encodedRegionName, new RegionState(regionInfo, RegionState.State.CLOSING, data.getStamp()));
                    break;
                }
                case RS_ZK_REGION_CLOSED: {
                    this.regionsInTransition.put(encodedRegionName, new RegionState(regionInfo, RegionState.State.CLOSED, data.getStamp()));
                    new ClosedRegionHandler(this.master, this, regionInfo).process();
                    break;
                }
                case M_ZK_REGION_OFFLINE: {
                    this.regionsInTransition.put(encodedRegionName, new RegionState(regionInfo, RegionState.State.OFFLINE, data.getStamp()));
                    new ClosedRegionHandler(this.master, this, regionInfo).process();
                    break;
                }
                case RS_ZK_REGION_OPENING: {
                    this.regionsInTransition.put(encodedRegionName, new RegionState(regionInfo, RegionState.State.OPENING, data.getStamp()));
                    break;
                }
                case RS_ZK_REGION_OPENED: {
                    this.regionsInTransition.put(encodedRegionName, new RegionState(regionInfo, RegionState.State.OPENING, data.getStamp()));
                    HServerInfo hsi = this.serverManager.getServerInfo(data.getServerName());
                    if (hsi == null) {
                        LOG.warn((Object)("Region in transition " + regionInfo.getEncodedName() + " references a server no longer up " + data.getServerName() + "; letting RIT timeout so will be assigned elsewhere"));
                        break;
                    }
                    new OpenedRegionHandler(this.master, this, regionInfo, hsi).process();
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleRegion(RegionTransitionData data) {
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            if (data == null || data.getServerName() == null) {
                LOG.warn((Object)("Unexpected NULL input " + data));
                return;
            }
            if (data.getServerName().equals("HBCKServerName")) {
                this.handleHBCK(data);
                return;
            }
            if (!this.serverManager.isServerOnline(data.getServerName()) && !this.master.getServerName().equals(data.getServerName())) {
                LOG.warn((Object)("Attempted to handle region transition for server but server is not online: " + data.getRegionName()));
                return;
            }
            String encodedName = HRegionInfo.encodeRegionName(data.getRegionName());
            String prettyPrintedRegionName = HRegionInfo.prettyPrint(encodedName);
            boolean lateEvent = data.getStamp() < System.currentTimeMillis() - 15000L;
            LOG.debug((Object)("Handling transition=" + (Object)((Object)data.getEventType()) + ", server=" + data.getServerName() + ", region=" + prettyPrintedRegionName + (lateEvent ? ", which is more than 15 seconds late" : "")));
            RegionState regionState = this.regionsInTransition.get(encodedName);
            switch (data.getEventType()) {
                case M_ZK_REGION_OFFLINE: {
                    break;
                }
                case RS_ZK_REGION_CLOSING: {
                    if (regionState == null || !regionState.isPendingClose() && !regionState.isClosing()) {
                        LOG.warn((Object)("Received CLOSING for region " + prettyPrintedRegionName + " from server " + data.getServerName() + " but region was in " + " the state " + regionState + " and not " + "in expected PENDING_CLOSE or CLOSING states"));
                        return;
                    }
                    regionState.update(RegionState.State.CLOSING, data.getStamp());
                    break;
                }
                case RS_ZK_REGION_CLOSED: {
                    if (regionState == null || !regionState.isPendingClose() && !regionState.isClosing()) {
                        LOG.warn((Object)("Received CLOSED for region " + prettyPrintedRegionName + " from server " + data.getServerName() + " but region was in " + " the state " + regionState + " and not " + "in expected PENDING_CLOSE or CLOSING states"));
                        return;
                    }
                    regionState.update(RegionState.State.CLOSED, data.getStamp());
                    this.executorService.submit(new ClosedRegionHandler(this.master, this, regionState.getRegion()));
                    break;
                }
                case RS_ZK_REGION_OPENING: {
                    if (regionState == null || !regionState.isPendingOpen() && !regionState.isOpening()) {
                        LOG.warn((Object)("Received OPENING for region " + prettyPrintedRegionName + " from server " + data.getServerName() + " but region was in " + " the state " + regionState + " and not " + "in expected PENDING_OPEN or OPENING states"));
                        return;
                    }
                    regionState.update(RegionState.State.OPENING, data.getStamp());
                    break;
                }
                case RS_ZK_REGION_OPENED: {
                    if (regionState == null || !regionState.isPendingOpen() && !regionState.isOpening()) {
                        LOG.warn((Object)("Received OPENED for region " + prettyPrintedRegionName + " from server " + data.getServerName() + " but region was in " + " the state " + regionState + " and not " + "in expected PENDING_OPEN or OPENING states"));
                        return;
                    }
                    regionState.update(RegionState.State.OPEN, data.getStamp());
                    this.executorService.submit(new OpenedRegionHandler(this.master, this, regionState.getRegion(), this.serverManager.getServerInfo(data.getServerName())));
                }
            }
        }
    }

    private void handleHBCK(RegionTransitionData data) {
        String encodedName = HRegionInfo.encodeRegionName(data.getRegionName());
        LOG.info((Object)("Handling HBCK triggered transition=" + (Object)((Object)data.getEventType()) + ", server=" + data.getServerName() + ", region=" + HRegionInfo.prettyPrint(encodedName)));
        RegionState regionState = this.regionsInTransition.get(encodedName);
        switch (data.getEventType()) {
            case M_ZK_REGION_OFFLINE: {
                HRegionInfo regionInfo = null;
                if (regionState != null) {
                    regionInfo = regionState.getRegion();
                } else {
                    try {
                        regionInfo = MetaReader.getRegion(this.catalogTracker, data.getRegionName()).getFirst();
                    }
                    catch (IOException e) {
                        LOG.info((Object)"Exception reading META doing HBCK repair operation", (Throwable)e);
                        return;
                    }
                }
                LOG.info((Object)("HBCK repair is triggering assignment of region=" + regionInfo.getRegionNameAsString()));
                this.assign(regionInfo, false);
                break;
            }
            default: {
                LOG.warn((Object)("Received unexpected region state from HBCK (" + (Object)((Object)data.getEventType()) + ")"));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void nodeCreated(String path) {
        if (path.startsWith(this.watcher.assignmentZNode)) {
            ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
            synchronized (concurrentSkipListMap) {
                try {
                    RegionTransitionData data = ZKAssign.getData(this.watcher, path);
                    if (data == null) {
                        return;
                    }
                    this.handleRegion(data);
                }
                catch (KeeperException e) {
                    this.master.abort("Unexpected ZK exception reading unassigned node data", e);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void nodeDataChanged(String path) {
        if (path.startsWith(this.watcher.assignmentZNode)) {
            ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
            synchronized (concurrentSkipListMap) {
                try {
                    RegionTransitionData data = ZKAssign.getData(this.watcher, path);
                    if (data == null) {
                        return;
                    }
                    this.handleRegion(data);
                }
                catch (KeeperException e) {
                    this.master.abort("Unexpected ZK exception reading unassigned node data", e);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void nodeChildrenChanged(String path) {
        if (path.equals(this.watcher.assignmentZNode)) {
            ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
            synchronized (concurrentSkipListMap) {
                try {
                    List<ZKUtil.NodeAndData> newNodes = ZKUtil.watchAndGetNewChildren(this.watcher, this.watcher.assignmentZNode);
                    for (ZKUtil.NodeAndData newNode : newNodes) {
                        LOG.debug((Object)("Handling new unassigned node: " + newNode));
                        this.handleRegion(RegionTransitionData.fromBytes(newNode.getData()));
                    }
                }
                catch (KeeperException e) {
                    this.master.abort("Unexpected ZK exception reading unassigned children", e);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void regionOnline(HRegionInfo regionInfo, HServerInfo serverInfo) {
        SortedMap<Object, Object> sortedMap = this.regionsInTransition;
        synchronized (sortedMap) {
            RegionState rs = this.regionsInTransition.remove(regionInfo.getEncodedName());
            if (rs != null) {
                this.regionsInTransition.notifyAll();
            }
        }
        sortedMap = this.regions;
        synchronized (sortedMap) {
            HServerInfo hsi = (HServerInfo)this.regions.get((Object)regionInfo);
            if (hsi != null) {
                LOG.warn((Object)("Overwriting " + regionInfo.getEncodedName() + " on " + hsi));
            }
            this.regions.put(regionInfo, serverInfo);
            this.addToServers(serverInfo, regionInfo);
            this.regions.notifyAll();
        }
        this.clearRegionPlan(regionInfo);
        this.updateTimers(serverInfo);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateTimers(HServerInfo hsi) {
        HashMap<String, LoadBalancer.RegionPlan> copy = new HashMap<String, LoadBalancer.RegionPlan>();
        NavigableMap<String, LoadBalancer.RegionPlan> navigableMap = this.regionPlans;
        synchronized (navigableMap) {
            copy.putAll(this.regionPlans);
        }
        for (Map.Entry e : copy.entrySet()) {
            if (!((LoadBalancer.RegionPlan)e.getValue()).getDestination().equals(hsi)) continue;
            RegionState rs = null;
            Object object = this.regionsInTransition;
            synchronized (object) {
                rs = this.regionsInTransition.get(e.getKey());
            }
            if (rs == null) continue;
            object = rs;
            synchronized (object) {
                rs.update(rs.getState());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void regionOffline(HRegionInfo regionInfo) {
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            if (this.regionsInTransition.remove(regionInfo.getEncodedName()) != null) {
                this.regionsInTransition.notifyAll();
            }
        }
        this.clearRegionPlan(regionInfo);
        this.setOffline(regionInfo);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setOffline(HRegionInfo regionInfo) {
        SortedMap<HRegionInfo, HServerInfo> sortedMap = this.regions;
        synchronized (sortedMap) {
            HServerInfo serverInfo = (HServerInfo)this.regions.remove((Object)regionInfo);
            if (serverInfo == null) {
                return;
            }
            List serverRegions = (List)this.servers.get(serverInfo);
            if (!serverRegions.remove((Object)regionInfo)) {
                LOG.warn((Object)("No " + (Object)((Object)regionInfo) + " on " + serverInfo));
            }
        }
    }

    public void offlineDisabledRegion(HRegionInfo regionInfo) {
        LOG.debug((Object)("Table being disabled so deleting ZK node and removing from regions in transition, skipping assignment of region " + regionInfo.getRegionNameAsString()));
        try {
            if (!ZKAssign.deleteClosedNode(this.watcher, regionInfo.getEncodedName())) {
                ZKAssign.deleteOfflineNode(this.watcher, regionInfo.getEncodedName());
            }
        }
        catch (KeeperException.NoNodeException nne) {
            LOG.debug((Object)("Tried to delete closed node for " + (Object)((Object)regionInfo) + " but it " + "does not exist so just offlining"));
        }
        catch (KeeperException e) {
            this.master.abort("Error deleting CLOSED node in ZK", e);
        }
        this.regionOffline(regionInfo);
    }

    public void assign(HRegionInfo region, boolean setOfflineInZK) {
        this.assign(region, setOfflineInZK, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void assign(HRegionInfo region, boolean setOfflineInZK, boolean forceNewPlan) {
        RegionState state;
        String tableName = region.getTableDesc().getNameAsString();
        boolean disabled = this.zkTable.isDisabledTable(tableName);
        if (disabled || this.zkTable.isDisablingTable(tableName)) {
            LOG.info((Object)("Table " + tableName + (disabled ? " disabled;" : " disabling;") + " skipping assign of " + region.getRegionNameAsString()));
            this.offlineDisabledRegion(region);
            return;
        }
        if (this.serverManager.isClusterShutdown()) {
            LOG.info((Object)("Cluster shutdown is set; skipping assign of " + region.getRegionNameAsString()));
            return;
        }
        RegionState regionState = state = this.addToRegionsInTransition(region);
        synchronized (regionState) {
            this.assign(state, setOfflineInZK, forceNewPlan);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void assign(HServerInfo destination, List<HRegionInfo> regions) {
        LOG.debug((Object)("Bulk assigning " + regions.size() + " region(s) to " + destination.getServerName()));
        ArrayList<RegionState> states = new ArrayList<RegionState>(regions.size());
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            for (HRegionInfo region : regions) {
                states.add(this.forceRegionStateToOffline(region));
            }
        }
        AtomicInteger counter = new AtomicInteger(0);
        CreateUnassignedAsyncCallback cb = new CreateUnassignedAsyncCallback(this.watcher, destination, counter);
        for (RegionState state : states) {
            if (this.asyncSetOfflineInZooKeeper(state, cb, state)) continue;
            return;
        }
        int total = regions.size();
        int oldCounter = 0;
        while (true) {
            int count;
            if (oldCounter != (count = counter.get())) {
                LOG.info((Object)(destination.getServerName() + " unassigned znodes=" + count + " of total=" + total));
                oldCounter = count;
            }
            if (count == total) break;
            Threads.sleep(1);
        }
        try {
            long maxWaitTime = System.currentTimeMillis() + this.master.getConfiguration().getLong("hbase.regionserver.rpc.startup.waittime", 60000L);
            while (!this.master.isStopped()) {
                try {
                    this.serverManager.sendRegionOpen(destination, regions);
                    break;
                }
                catch (ServerNotRunningException e) {
                    long now = System.currentTimeMillis();
                    if (now > maxWaitTime) {
                        throw e;
                    }
                    LOG.debug((Object)("Server is not yet up; waiting up to " + (maxWaitTime - now) + "ms"), (Throwable)e);
                    Thread.sleep(1000L);
                }
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        LOG.debug((Object)("Bulk assigning done for " + destination.getServerName()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private RegionState addToRegionsInTransition(HRegionInfo region) {
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            return this.forceRegionStateToOffline(region);
        }
    }

    private RegionState forceRegionStateToOffline(HRegionInfo region) {
        String encodedName = region.getEncodedName();
        RegionState state = this.regionsInTransition.get(encodedName);
        if (state == null) {
            state = new RegionState(region, RegionState.State.OFFLINE);
            this.regionsInTransition.put(encodedName, state);
        } else {
            LOG.debug((Object)("Forcing OFFLINE; was=" + state));
            state.update(RegionState.State.OFFLINE);
        }
        return state;
    }

    private void assign(RegionState state, boolean setOfflineInZK, boolean forceNewPlan) {
        for (int i = 0; i < this.maximumAssignmentAttempts; ++i) {
            if (setOfflineInZK && !this.setOfflineInZooKeeper(state)) {
                return;
            }
            if (this.master.isStopped()) {
                LOG.debug((Object)("Server stopped; skipping assign of " + state));
                return;
            }
            LoadBalancer.RegionPlan plan = this.getRegionPlan(state, forceNewPlan);
            if (plan == null) {
                return;
            }
            try {
                LOG.debug((Object)("Assigning region " + state.getRegion().getRegionNameAsString() + " to " + plan.getDestination().getServerName()));
                state.update(RegionState.State.PENDING_OPEN);
                this.serverManager.sendRegionOpen(plan.getDestination(), state.getRegion());
                break;
            }
            catch (Throwable t) {
                LOG.warn((Object)("Failed assignment of " + state.getRegion().getRegionNameAsString() + " to " + plan.getDestination() + ", trying to assign elsewhere instead; " + "retry=" + i), t);
                state.update(RegionState.State.OFFLINE);
                if (this.getRegionPlan(state, plan.getDestination(), true) != null) continue;
                LOG.warn((Object)("Unable to find a viable location to assign region " + state.getRegion().getRegionNameAsString()));
                return;
            }
        }
    }

    boolean setOfflineInZooKeeper(RegionState state) {
        if (!state.isClosed() && !state.isOffline()) {
            new RuntimeException("Unexpected state trying to OFFLINE; " + state);
            this.master.abort("Unexpected state trying to OFFLINE; " + state, new IllegalStateException());
            return false;
        }
        state.update(RegionState.State.OFFLINE);
        try {
            if (!ZKAssign.createOrForceNodeOffline(this.master.getZooKeeper(), state.getRegion(), this.master.getServerName())) {
                LOG.warn((Object)("Attempted to create/force node into OFFLINE state before completing assignment but failed to do so for " + state));
                return false;
            }
        }
        catch (KeeperException e) {
            this.master.abort("Unexpected ZK exception creating/setting node OFFLINE", e);
            return false;
        }
        return true;
    }

    boolean asyncSetOfflineInZooKeeper(RegionState state, AsyncCallback.StringCallback cb, Object ctx) {
        if (!state.isClosed() && !state.isOffline()) {
            new RuntimeException("Unexpected state trying to OFFLINE; " + state);
            this.master.abort("Unexpected state trying to OFFLINE; " + state, new IllegalStateException());
            return false;
        }
        state.update(RegionState.State.OFFLINE);
        try {
            ZKAssign.asyncCreateNodeOffline(this.master.getZooKeeper(), state.getRegion(), this.master.getServerName(), cb, ctx);
        }
        catch (KeeperException e) {
            this.master.abort("Unexpected ZK exception creating/setting node OFFLINE", e);
            return false;
        }
        return true;
    }

    LoadBalancer.RegionPlan getRegionPlan(RegionState state, boolean forceNewPlan) {
        return this.getRegionPlan(state, null, forceNewPlan);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    LoadBalancer.RegionPlan getRegionPlan(RegionState state, HServerInfo serverToExclude, boolean forceNewPlan) {
        String encodedName = state.getRegion().getEncodedName();
        List<HServerInfo> servers = this.serverManager.getOnlineServersList();
        if (serverToExclude != null) {
            servers.remove(serverToExclude);
        }
        if (servers.isEmpty()) {
            return null;
        }
        LoadBalancer.RegionPlan randomPlan = new LoadBalancer.RegionPlan(state.getRegion(), null, LoadBalancer.randomAssignment(servers));
        boolean newPlan = false;
        LoadBalancer.RegionPlan existingPlan = null;
        NavigableMap<String, LoadBalancer.RegionPlan> navigableMap = this.regionPlans;
        synchronized (navigableMap) {
            existingPlan = (LoadBalancer.RegionPlan)this.regionPlans.get(encodedName);
            if (forceNewPlan || existingPlan == null || existingPlan.getDestination() == null || existingPlan.getDestination().equals(serverToExclude)) {
                newPlan = true;
                this.regionPlans.put(encodedName, randomPlan);
            }
        }
        if (newPlan) {
            LOG.debug((Object)("No previous transition plan was found (or we are ignoring an existing plan) for " + state.getRegion().getRegionNameAsString() + " so generated a random one; " + randomPlan + "; " + this.serverManager.countOfRegionServers() + " (online=" + this.serverManager.getOnlineServers().size() + ", exclude=" + serverToExclude + ") available servers"));
            return randomPlan;
        }
        LOG.debug((Object)("Using pre-existing plan for region " + state.getRegion().getRegionNameAsString() + "; plan=" + existingPlan));
        return existingPlan;
    }

    public void unassign(HRegionInfo region) {
        this.unassign(region, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void unassign(HRegionInfo region, boolean force) {
        RegionState state;
        LOG.debug((Object)("Starting unassignment of region " + region.getRegionNameAsString() + " (offlining)"));
        SortedMap<HRegionInfo, HServerInfo> sortedMap = this.regions;
        synchronized (sortedMap) {
            if (!this.regions.containsKey((Object)region)) {
                LOG.debug((Object)("Attempted to unassign region " + region.getRegionNameAsString() + " but it is not " + "currently assigned anywhere"));
                return;
            }
        }
        String encodedName = region.getEncodedName();
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            state = this.regionsInTransition.get(encodedName);
            if (state == null) {
                state = new RegionState(region, RegionState.State.PENDING_CLOSE);
                this.regionsInTransition.put(encodedName, state);
            } else if (force && state.isPendingClose()) {
                LOG.debug((Object)("Attempting to unassign region " + region.getRegionNameAsString() + " which is already pending close " + "but forcing an additional close"));
                state.update(RegionState.State.PENDING_CLOSE);
            } else {
                LOG.debug((Object)("Attempting to unassign region " + region.getRegionNameAsString() + " but it is " + "already in transition (" + (Object)((Object)state.getState()) + ")"));
                return;
            }
        }
        HServerInfo server = null;
        SortedMap<HRegionInfo, HServerInfo> sortedMap2 = this.regions;
        synchronized (sortedMap2) {
            server = (HServerInfo)this.regions.get((Object)region);
        }
        try {
            if (this.serverManager.sendRegionClose(server, state.getRegion())) {
                LOG.debug((Object)("Sent CLOSE to " + server + " for region " + region.getRegionNameAsString()));
                return;
            }
            LOG.debug((Object)("Server " + server + " region CLOSE RPC returned false for " + region.getEncodedName()));
        }
        catch (NotServingRegionException nsre) {
            LOG.info((Object)("Server " + server + " returned " + nsre + " for " + region.getEncodedName()));
            return;
        }
        catch (ConnectException e) {
            LOG.info((Object)("Failed connect to " + server + ", message=" + e.getMessage() + ", region=" + region.getEncodedName()));
        }
        catch (SocketTimeoutException e) {
            LOG.info((Object)("Server " + server + " returned " + e.getMessage() + " for " + region.getEncodedName()));
        }
        catch (EOFException e) {
            LOG.info((Object)("Server " + server + " returned " + e.getMessage() + " for " + region.getEncodedName()));
        }
        catch (RemoteException re) {
            IOException ioe = re.unwrapRemoteException();
            if (ioe instanceof NotServingRegionException) {
                LOG.debug((Object)("Server " + server + " returned " + ioe + " for " + region.getEncodedName()));
            } else if (ioe instanceof EOFException) {
                LOG.debug((Object)("Server " + server + " returned " + ioe + " for " + region.getEncodedName()));
            } else {
                this.master.abort("Remote unexpected exception", ioe);
            }
        }
        catch (Throwable t) {
            this.master.abort("Remote unexpected exception", t);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void waitForAssignment(HRegionInfo regionInfo) throws InterruptedException {
        SortedMap<HRegionInfo, HServerInfo> sortedMap = this.regions;
        synchronized (sortedMap) {
            while (!this.regions.containsKey((Object)regionInfo)) {
                this.regions.wait();
            }
        }
    }

    public void assignRoot() throws KeeperException {
        RootLocationEditor.deleteRootLocation(this.master.getZooKeeper());
        this.assign(HRegionInfo.ROOT_REGIONINFO, true);
    }

    public void assignMeta() {
        this.assign(HRegionInfo.FIRST_META_REGIONINFO, true);
    }

    public void assignAllUserRegions() throws IOException, InterruptedException {
        List<HServerInfo> servers = this.serverManager.getOnlineServersList();
        Map<HRegionInfo, HServerAddress> allRegions = MetaReader.fullScan(this.catalogTracker, this.zkTable.getDisabledTables(), true);
        if (allRegions == null || allRegions.isEmpty()) {
            return;
        }
        boolean retainAssignment = this.master.getConfiguration().getBoolean("hbase.master.startup.retainassign", true);
        Map<HServerInfo, List<HRegionInfo>> bulkPlan = null;
        bulkPlan = retainAssignment ? LoadBalancer.retainAssignment(allRegions, servers) : LoadBalancer.roundRobinAssignment(new ArrayList<HRegionInfo>(allRegions.keySet()), servers);
        LOG.info((Object)("Bulk assigning " + allRegions.size() + " region(s) across " + servers.size() + " server(s), retainAssignment=" + retainAssignment));
        StartupBulkAssigner ba = new StartupBulkAssigner(this.master, bulkPlan, this);
        ba.bulkAssign();
        LOG.info((Object)"Bulk assigning done");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean waitUntilNoRegionsInTransition(long timeout) throws InterruptedException {
        long startTime = System.currentTimeMillis();
        long remaining = timeout;
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            while (this.regionsInTransition.size() > 0 && !this.master.isStopped() && remaining > 0L) {
                this.regionsInTransition.wait(remaining);
                remaining = timeout - (System.currentTimeMillis() - startTime);
            }
        }
        return this.regionsInTransition.isEmpty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean waitUntilNoRegionsInTransition(long timeout, Set<HRegionInfo> regions) throws InterruptedException {
        long startTime = System.currentTimeMillis();
        long remaining = timeout;
        boolean stillInTransition = true;
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            while (this.regionsInTransition.size() > 0 && !this.master.isStopped() && remaining > 0L && stillInTransition) {
                int count = 0;
                for (RegionState rs : this.regionsInTransition.values()) {
                    if (!regions.contains((Object)rs.getRegion())) continue;
                    ++count;
                    break;
                }
                if (count == 0) {
                    stillInTransition = false;
                    break;
                }
                this.regionsInTransition.wait(remaining);
                remaining = timeout - (System.currentTimeMillis() - startTime);
            }
        }
        return stillInTransition;
    }

    private Map<HServerInfo, List<Pair<HRegionInfo, Result>>> rebuildUserRegions() throws IOException {
        List<Result> results = MetaReader.fullScanOfResults(this.catalogTracker);
        TreeMap<HServerInfo, List<Pair<HRegionInfo, Result>>> offlineServers = new TreeMap<HServerInfo, List<Pair<HRegionInfo, Result>>>();
        for (Result result : results) {
            Pair<HRegionInfo, HServerInfo> region = MetaReader.metaRowToRegionPairWithInfo(result);
            if (region == null) continue;
            HServerInfo regionLocation = region.getSecond();
            HRegionInfo regionInfo = region.getFirst();
            if (regionLocation == null) {
                this.regions.put(regionInfo, null);
                continue;
            }
            if (!this.serverManager.isServerOnline(regionLocation.getServerName())) {
                ArrayList<Pair<HRegionInfo, Result>> offlineRegions = (ArrayList<Pair<HRegionInfo, Result>>)offlineServers.get(regionLocation);
                if (offlineRegions == null) {
                    offlineRegions = new ArrayList<Pair<HRegionInfo, Result>>(1);
                    offlineServers.put(regionLocation, offlineRegions);
                }
                offlineRegions.add(new Pair<HRegionInfo, Result>(regionInfo, result));
                continue;
            }
            this.regions.put(regionInfo, regionLocation);
            this.addToServers(regionLocation, regionInfo);
        }
        return offlineServers;
    }

    private void processDeadServers(Map<HServerInfo, List<Pair<HRegionInfo, Result>>> deadServers) throws IOException, KeeperException {
        for (Map.Entry<HServerInfo, List<Pair<HRegionInfo, Result>>> deadServer : deadServers.entrySet()) {
            List<Pair<HRegionInfo, Result>> regions = deadServer.getValue();
            for (Pair<HRegionInfo, Result> region : regions) {
                HRegionInfo regionInfo = region.getFirst();
                Result result = region.getSecond();
                try {
                    ZKAssign.createOrForceNodeOffline(this.watcher, regionInfo, this.master.getServerName());
                }
                catch (KeeperException.NoNodeException nne) {
                    // empty catch block
                }
                ServerShutdownHandler.processDeadRegion(regionInfo, result, this, this.catalogTracker);
            }
        }
    }

    private void addToServers(HServerInfo hsi, HRegionInfo hri) {
        ArrayList<HRegionInfo> hris = (ArrayList<HRegionInfo>)this.servers.get(hsi);
        if (hris == null) {
            hris = new ArrayList<HRegionInfo>();
            this.servers.put(hsi, hris);
        }
        hris.add(hri);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NavigableMap<String, RegionState> getRegionsInTransition() {
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            return new TreeMap<String, RegionState>((SortedMap<String, RegionState>)this.regionsInTransition);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isRegionsInTransition() {
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            return !this.regionsInTransition.isEmpty();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public RegionState isRegionInTransition(HRegionInfo hri) {
        ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = this.regionsInTransition;
        synchronized (concurrentSkipListMap) {
            return this.regionsInTransition.get(hri.getEncodedName());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clearRegionFromTransition(HRegionInfo hri) {
        SortedMap<Object, Object> sortedMap = this.regionsInTransition;
        synchronized (sortedMap) {
            this.regionsInTransition.remove(hri.getEncodedName());
        }
        sortedMap = this.regions;
        synchronized (sortedMap) {
            this.regions.remove((Object)hri);
            block6: for (List regions : this.servers.values()) {
                for (int i = 0; i < regions.size(); ++i) {
                    if (!((HRegionInfo)((Object)regions.get(i))).equals((Object)hri)) continue;
                    regions.remove(i);
                    continue block6;
                }
            }
        }
        this.clearRegionPlan(hri);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void clearRegionPlan(HRegionInfo region) {
        NavigableMap<String, LoadBalancer.RegionPlan> navigableMap = this.regionPlans;
        synchronized (navigableMap) {
            this.regionPlans.remove(region.getEncodedName());
        }
    }

    public void waitOnRegionToClearRegionsInTransition(HRegionInfo hri) throws IOException {
        if (this.isRegionInTransition(hri) == null) {
            return;
        }
        RegionState rs = null;
        while (!this.master.isStopped() && (rs = this.isRegionInTransition(hri)) != null) {
            Threads.sleep(1000);
            LOG.info((Object)("Waiting on " + rs + " to clear regions-in-transition"));
        }
        if (this.master.isStopped()) {
            LOG.info((Object)"Giving up wait on regions in transition because stoppable.isStopped is set");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<HRegionInfo> getRegionsOfTable(byte[] tableName) {
        ArrayList<HRegionInfo> tableRegions = new ArrayList<HRegionInfo>();
        HRegionInfo boundary = new HRegionInfo(new HTableDescriptor(tableName), null, null);
        SortedMap<HRegionInfo, HServerInfo> sortedMap = this.regions;
        synchronized (sortedMap) {
            for (HRegionInfo regionInfo : this.regions.tailMap(boundary).keySet()) {
                if (!Bytes.equals(regionInfo.getTableDesc().getName(), tableName)) break;
                tableRegions.add(regionInfo);
            }
        }
        return tableRegions;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<RegionState> processServerShutdown(HServerInfo hsi) {
        NavigableMap<String, LoadBalancer.RegionPlan> navigableMap = this.regionPlans;
        synchronized (navigableMap) {
            Iterator i = this.regionPlans.entrySet().iterator();
            while (i.hasNext()) {
                Map.Entry e = i.next();
                if (!((LoadBalancer.RegionPlan)e.getValue()).getDestination().equals(hsi)) continue;
                i.remove();
            }
        }
        TreeSet deadRegions = null;
        ArrayList<RegionState> rits = new ArrayList<RegionState>();
        SortedMap<HRegionInfo, HServerInfo> sortedMap = this.regions;
        synchronized (sortedMap) {
            List assignedRegions = (List)this.servers.remove(hsi);
            if (assignedRegions == null || assignedRegions.isEmpty()) {
                return rits;
            }
            deadRegions = new TreeSet(assignedRegions);
            for (HRegionInfo region : deadRegions) {
                this.regions.remove((Object)region);
            }
        }
        sortedMap = this.regionsInTransition;
        synchronized (sortedMap) {
            for (RegionState region : this.regionsInTransition.values()) {
                if (!deadRegions.remove((Object)region.getRegion())) continue;
                rits.add(region);
            }
        }
        return rits;
    }

    public void handleSplitReport(HServerInfo hsi, HRegionInfo parent, HRegionInfo a, HRegionInfo b) {
        this.regionOffline(parent);
        try {
            RegionTransitionData node = ZKAssign.getDataNoWatch(this.watcher, parent.getEncodedName(), null);
            if (node != null) {
                if (node.getEventType().equals((Object)EventHandler.EventType.RS_ZK_REGION_CLOSING)) {
                    ZKAssign.deleteClosingNode(this.watcher, parent);
                } else {
                    LOG.warn((Object)("Split report has RIT node (shouldnt have one): " + (Object)((Object)parent) + " node: " + node));
                }
            }
        }
        catch (KeeperException e) {
            LOG.warn((Object)"Exception while validating RIT during split report", (Throwable)e);
        }
        this.regionOnline(a, hsi);
        this.regionOnline(b, hsi);
        if (this.zkTable.isDisablingOrDisabledTable(parent.getTableDesc().getNameAsString())) {
            this.unassign(a);
            this.unassign(b);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Map<HServerInfo, List<HRegionInfo>> getAssignments() {
        HashMap<HServerInfo, List<HRegionInfo>> result = null;
        SortedMap<HRegionInfo, HServerInfo> sortedMap = this.regions;
        synchronized (sortedMap) {
            result = new HashMap<HServerInfo, List<HRegionInfo>>(this.servers.size());
            for (Map.Entry e : this.servers.entrySet()) {
                ArrayList shallowCopy = new ArrayList((Collection)e.getValue());
                HServerInfo clone = new HServerInfo((HServerInfo)e.getKey());
                clone.getLoad().setNumberOfRegions(((List)e.getValue()).size());
                result.put(clone, shallowCopy);
            }
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Pair<HRegionInfo, HServerInfo> getAssignment(byte[] encodedRegionName) {
        String name = Bytes.toString(encodedRegionName);
        SortedMap<HRegionInfo, HServerInfo> sortedMap = this.regions;
        synchronized (sortedMap) {
            for (Map.Entry<HRegionInfo, HServerInfo> e : this.regions.entrySet()) {
                if (!e.getKey().getEncodedName().equals(name)) continue;
                return new Pair<HRegionInfo, HServerInfo>(e.getKey(), e.getValue());
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void balance(LoadBalancer.RegionPlan plan) {
        NavigableMap<String, LoadBalancer.RegionPlan> navigableMap = this.regionPlans;
        synchronized (navigableMap) {
            this.regionPlans.put(plan.getRegionName(), plan);
        }
        this.unassign(plan.getRegionInfo());
    }

    void bulkAssignUserRegions(HRegionInfo[] regions, List<HServerInfo> servers, boolean sync) throws IOException {
        Map<HServerInfo, List<HRegionInfo>> bulkPlan = LoadBalancer.roundRobinAssignment(Arrays.asList(regions), servers);
        LOG.info((Object)("Bulk assigning " + regions.length + " region(s) " + "round-robin across " + servers.size() + " server(s)"));
        GeneralBulkAssigner ba = new GeneralBulkAssigner(this.master, bulkPlan, this);
        try {
            ((BulkAssigner)ba).bulkAssign(sync);
        }
        catch (InterruptedException e) {
            throw new IOException("InterruptedException bulk assigning", e);
        }
        LOG.info((Object)"Bulk assigning done");
    }

    public void stop() {
        this.timeoutMonitor.interrupt();
    }

    public static class RegionState
    implements Writable {
        private HRegionInfo region;
        private State state;
        private long stamp;

        public RegionState() {
        }

        RegionState(HRegionInfo region, State state) {
            this(region, state, System.currentTimeMillis());
        }

        RegionState(HRegionInfo region, State state, long stamp) {
            this.region = region;
            this.state = state;
            this.stamp = stamp;
        }

        public void update(State state, long stamp) {
            this.state = state;
            this.stamp = stamp;
        }

        public void update(State state) {
            this.state = state;
            this.stamp = System.currentTimeMillis();
        }

        public State getState() {
            return this.state;
        }

        public long getStamp() {
            return this.stamp;
        }

        public HRegionInfo getRegion() {
            return this.region;
        }

        public boolean isClosing() {
            return this.state == State.CLOSING;
        }

        public boolean isClosed() {
            return this.state == State.CLOSED;
        }

        public boolean isPendingClose() {
            return this.state == State.PENDING_CLOSE;
        }

        public boolean isOpening() {
            return this.state == State.OPENING;
        }

        public boolean isOpened() {
            return this.state == State.OPEN;
        }

        public boolean isPendingOpen() {
            return this.state == State.PENDING_OPEN;
        }

        public boolean isOffline() {
            return this.state == State.OFFLINE;
        }

        public String toString() {
            return this.region.getRegionNameAsString() + " state=" + (Object)((Object)this.state) + ", ts=" + this.stamp;
        }

        public void readFields(DataInput in) throws IOException {
            this.region = new HRegionInfo();
            this.region.readFields(in);
            this.state = State.valueOf(in.readUTF());
            this.stamp = in.readLong();
        }

        public void write(DataOutput out) throws IOException {
            this.region.write(out);
            out.writeUTF(this.state.name());
            out.writeLong(this.stamp);
        }

        public static enum State {
            OFFLINE,
            PENDING_OPEN,
            OPENING,
            OPEN,
            PENDING_CLOSE,
            CLOSING,
            CLOSED;

        }
    }

    public class TimeoutMonitor
    extends Chore {
        private final int timeout;
        private boolean bulkAssign;

        public TimeoutMonitor(int period, Stoppable stopper, int timeout) {
            super("AssignmentTimeoutMonitor", period, stopper);
            this.bulkAssign = false;
            this.timeout = timeout;
        }

        public boolean bulkAssign(boolean bulkAssign) {
            boolean result = this.bulkAssign;
            this.bulkAssign = bulkAssign;
            return result;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        protected void chore() {
            if (this.bulkAssign) {
                return;
            }
            ArrayList<HRegionInfo> unassigns = new ArrayList<HRegionInfo>();
            HashMap<HRegionInfo, Boolean> assigns = new HashMap<HRegionInfo, Boolean>();
            ConcurrentSkipListMap<String, RegionState> concurrentSkipListMap = AssignmentManager.this.regionsInTransition;
            synchronized (concurrentSkipListMap) {
                long l = System.currentTimeMillis();
                block22: for (RegionState regionState : AssignmentManager.this.regionsInTransition.values()) {
                    if (regionState.getStamp() + (long)this.timeout > l) continue;
                    HRegionInfo regionInfo = regionState.getRegion();
                    LOG.info((Object)("Regions in transition timed out:  " + regionState));
                    switch (regionState.getState()) {
                        case CLOSED: {
                            LOG.info((Object)("Region " + regionInfo.getEncodedName() + " has been CLOSED for too long, waiting on queued " + "ClosedRegionHandler to run or server shutdown"));
                            RegionState regionState2 = regionState;
                            synchronized (regionState2) {
                                regionState.update(regionState.getState());
                                break;
                            }
                        }
                        case OFFLINE: {
                            LOG.info((Object)("Region has been OFFLINE for too long, reassigning " + regionInfo.getRegionNameAsString() + " to a random server"));
                            assigns.put(regionState.getRegion(), Boolean.FALSE);
                            break;
                        }
                        case PENDING_OPEN: {
                            LOG.info((Object)("Region has been PENDING_OPEN for too long, reassigning region=" + regionInfo.getRegionNameAsString()));
                            assigns.put(regionState.getRegion(), Boolean.TRUE);
                            break;
                        }
                        case OPENING: {
                            LOG.info((Object)("Region has been OPENING for too long, reassigning region=" + regionInfo.getRegionNameAsString()));
                            try {
                                String node = ZKAssign.getNodeName(AssignmentManager.this.watcher, regionInfo.getEncodedName());
                                Stat stat = new Stat();
                                RegionTransitionData data = ZKAssign.getDataNoWatch(AssignmentManager.this.watcher, node, stat);
                                if (data == null) {
                                    LOG.warn((Object)("Data is null, node " + node + " no longer exists"));
                                    break;
                                }
                                if (data.getEventType() == EventHandler.EventType.RS_ZK_REGION_OPENED) {
                                    LOG.debug((Object)"Region has transitioned to OPENED, allowing watched event handlers to process");
                                    break;
                                }
                                if (data.getEventType() != EventHandler.EventType.RS_ZK_REGION_OPENING) {
                                    LOG.warn((Object)("While timing out a region in state OPENING, found ZK node in unexpected state: " + (Object)((Object)data.getEventType())));
                                    break;
                                }
                                try {
                                    data = new RegionTransitionData(EventHandler.EventType.M_ZK_REGION_OFFLINE, regionInfo.getRegionName(), AssignmentManager.this.master.getServerName());
                                    if (!ZKUtil.setData(AssignmentManager.this.watcher, node, data.getBytes(), stat.getVersion())) continue block22;
                                    ZKUtil.getDataAndWatch(AssignmentManager.this.watcher, node);
                                    LOG.info((Object)("Successfully transitioned region=" + regionInfo.getRegionNameAsString() + " into OFFLINE" + " and forcing a new assignment"));
                                    assigns.put(regionState.getRegion(), Boolean.TRUE);
                                }
                                catch (KeeperException.NoNodeException nne) {
                                }
                            }
                            catch (KeeperException ke) {
                                LOG.error((Object)"Unexpected ZK exception timing out CLOSING region", (Throwable)ke);
                            }
                            break;
                        }
                        case OPEN: {
                            LOG.error((Object)"Region has been OPEN for too long, we don't know where region was opened so can't do anything");
                            break;
                        }
                        case PENDING_CLOSE: {
                            LOG.info((Object)("Region has been PENDING_CLOSE for too long, running forced unassign again on region=" + regionInfo.getRegionNameAsString()));
                            try {
                                if (ZKUtil.watchAndCheckExists(AssignmentManager.this.watcher, ZKAssign.getNodeName(AssignmentManager.this.watcher, regionInfo.getEncodedName()))) continue block22;
                                unassigns.add(regionInfo);
                            }
                            catch (KeeperException.NoNodeException e) {
                                LOG.debug((Object)"Node no longer existed so not forcing another unassignment");
                            }
                            catch (KeeperException e) {
                                LOG.warn((Object)"Unexpected ZK exception timing out a region close", (Throwable)e);
                            }
                            break;
                        }
                        case CLOSING: {
                            LOG.info((Object)"Region has been CLOSING for too long, this should eventually complete or the server will expire, doing nothing");
                        }
                    }
                }
            }
            for (HRegionInfo hRegionInfo : unassigns) {
                AssignmentManager.this.unassign(hRegionInfo, true);
            }
            for (Map.Entry entry : assigns.entrySet()) {
                AssignmentManager.this.assign((HRegionInfo)((Object)entry.getKey()), false, (boolean)((Boolean)entry.getValue()));
            }
        }
    }

    static class SingleServerBulkAssigner
    implements Runnable {
        private final HServerInfo regionserver;
        private final List<HRegionInfo> regions;
        private final AssignmentManager assignmentManager;

        SingleServerBulkAssigner(HServerInfo regionserver, List<HRegionInfo> regions, AssignmentManager am, boolean startUp) {
            this.regionserver = regionserver;
            this.regions = regions;
            this.assignmentManager = am;
        }

        @Override
        public void run() {
            this.assignmentManager.assign(this.regionserver, this.regions);
        }
    }

    static class GeneralBulkAssigner
    extends StartupBulkAssigner {
        GeneralBulkAssigner(Server server, Map<HServerInfo, List<HRegionInfo>> bulkPlan, AssignmentManager am) {
            super(server, bulkPlan, am);
        }

        @Override
        protected Thread.UncaughtExceptionHandler getUncaughtExceptionHandler() {
            return new Thread.UncaughtExceptionHandler(){

                @Override
                public void uncaughtException(Thread t, Throwable e) {
                    LOG.warn((Object)("Assigning regions in " + t.getName()), e);
                }
            };
        }
    }

    static class StartupBulkAssigner
    extends BulkAssigner {
        final Map<HServerInfo, List<HRegionInfo>> bulkPlan;
        final AssignmentManager assignmentManager;

        StartupBulkAssigner(Server server, Map<HServerInfo, List<HRegionInfo>> bulkPlan, AssignmentManager am) {
            super(server);
            this.bulkPlan = bulkPlan;
            this.assignmentManager = am;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public boolean bulkAssign(boolean sync) throws InterruptedException {
            this.assignmentManager.timeoutMonitor.bulkAssign(true);
            try {
                boolean bl = super.bulkAssign(sync);
                return bl;
            }
            finally {
                this.assignmentManager.timeoutMonitor.bulkAssign(false);
            }
        }

        @Override
        protected String getThreadNamePrefix() {
            return this.server.getServerName() + "-StartupBulkAssigner";
        }

        @Override
        protected void populatePool(ExecutorService pool) {
            for (Map.Entry<HServerInfo, List<HRegionInfo>> e : this.bulkPlan.entrySet()) {
                pool.execute(new SingleServerBulkAssigner(e.getKey(), e.getValue(), this.assignmentManager, true));
            }
        }

        @Override
        protected boolean waitUntilDone(long timeout) throws InterruptedException {
            HashSet<HRegionInfo> regionSet = new HashSet<HRegionInfo>();
            for (List<HRegionInfo> regionList : this.bulkPlan.values()) {
                regionSet.addAll(regionList);
            }
            return this.assignmentManager.waitUntilNoRegionsInTransition(timeout, regionSet);
        }

        @Override
        protected long getTimeoutOnRIT() {
            long perRegionOpenTimeGuesstimate = this.server.getConfiguration().getLong("hbase.bulk.assignment.perregion.open.time", 1000L);
            int regionsPerServer = this.bulkPlan.entrySet().iterator().next().getValue().size();
            long timeout = perRegionOpenTimeGuesstimate * (long)regionsPerServer;
            LOG.debug((Object)("Timeout-on-RIT=" + timeout));
            return timeout;
        }
    }

    static class ExistsUnassignedAsyncCallback
    implements AsyncCallback.StatCallback {
        private final Log LOG = LogFactory.getLog(ExistsUnassignedAsyncCallback.class);
        private final AtomicInteger counter;

        ExistsUnassignedAsyncCallback(AtomicInteger counter) {
            this.counter = counter;
        }

        public void processResult(int rc, String path, Object ctx, Stat stat) {
            if (rc != 0) {
                this.LOG.warn((Object)("rc != 0 for " + path + " -- retryable connectionloss -- " + "FIX see http://wiki.apache.org/hadoop/ZooKeeper/FAQ#A2"));
                return;
            }
            RegionState state = (RegionState)ctx;
            this.LOG.debug((Object)("rs=" + state));
            state.update(RegionState.State.PENDING_OPEN);
            this.counter.addAndGet(1);
        }
    }

    static class CreateUnassignedAsyncCallback
    implements AsyncCallback.StringCallback {
        private final Log LOG = LogFactory.getLog(CreateUnassignedAsyncCallback.class);
        private final ZooKeeperWatcher zkw;
        private final HServerInfo destination;
        private final AtomicInteger counter;

        CreateUnassignedAsyncCallback(ZooKeeperWatcher zkw, HServerInfo destination, AtomicInteger counter) {
            this.zkw = zkw;
            this.destination = destination;
            this.counter = counter;
        }

        public void processResult(int rc, String path, Object ctx, String name) {
            if (rc != 0) {
                this.LOG.warn((Object)("rc != 0 for " + path + " -- retryable connectionloss -- " + "FIX see http://wiki.apache.org/hadoop/ZooKeeper/FAQ#A2"));
                this.zkw.abort("Connectionloss writing unassigned at " + path + ", rc=" + rc, null);
                return;
            }
            this.LOG.debug((Object)("rs=" + (RegionState)ctx + ", server=" + this.destination.getServerName()));
            this.zkw.getZooKeeper().exists(path, (Watcher)this.zkw, (AsyncCallback.StatCallback)new ExistsUnassignedAsyncCallback(this.counter), ctx);
        }
    }
}

