/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.ClusterStatus;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HServerAddress;
import org.apache.hadoop.hbase.HServerInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.MasterNotRunningException;
import org.apache.hadoop.hbase.ZooKeeperConnectionException;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HConnection;
import org.apache.hadoop.hbase.client.HConnectionManager;
import org.apache.hadoop.hbase.client.MetaScanner;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.ipc.HRegionInterface;
import org.apache.hadoop.hbase.regionserver.wal.HLog;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.HBaseFsckRepair;
import org.apache.hadoop.hbase.util.Writables;
import org.apache.hadoop.hbase.zookeeper.ZKTable;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperWatcher;
import org.apache.zookeeper.KeeperException;

public class HBaseFsck {
    public static final long DEFAULT_TIME_LAG = 60000L;
    public static final long DEFAULT_SLEEP_BEFORE_RERUN = 10000L;
    private static final int MAX_NUM_THREADS = 50;
    private static final long THREADS_KEEP_ALIVE_SECONDS = 60L;
    private static final Log LOG = LogFactory.getLog((String)HBaseFsck.class.getName());
    private Configuration conf;
    private ClusterStatus status;
    private HConnection connection;
    private TreeMap<String, HbckInfo> regionInfo = new TreeMap();
    private TreeMap<String, TInfo> tablesInfo = new TreeMap();
    private TreeSet<byte[]> disabledTables = new TreeSet<byte[]>(Bytes.BYTES_COMPARATOR);
    ErrorReporter errors = new PrintingErrorReporter();
    private static boolean details = false;
    private long timelag = 60000L;
    private boolean fix = false;
    private boolean rerun = false;
    private static boolean summary = false;
    private Set<Result> emptyRegionInfoQualifiers = new HashSet<Result>();
    private int numThreads = 50;
    ThreadPoolExecutor executor;

    public HBaseFsck(Configuration conf) throws MasterNotRunningException, ZooKeeperConnectionException, IOException {
        this.conf = conf;
        HBaseAdmin admin = new HBaseAdmin(conf);
        this.status = admin.getMaster().getClusterStatus();
        this.connection = admin.getConnection();
        this.numThreads = conf.getInt("hbasefsck.numthreads", this.numThreads);
        this.executor = new ThreadPoolExecutor(0, this.numThreads, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());
    }

    int doWork() throws IOException, KeeperException, InterruptedException {
        this.errors.print("Version: " + this.status.getHBaseVersion());
        this.regionInfo.clear();
        this.tablesInfo.clear();
        this.emptyRegionInfoQualifiers.clear();
        this.disabledTables.clear();
        this.errors.clear();
        if (!this.recordRootRegion()) {
            this.errors.reportError("Encountered fatal error. Exiting...");
            return -1;
        }
        this.getMetaEntries();
        if (!this.checkMetaEntries()) {
            this.errors.reportError("Encountered fatal error. Exiting...");
            return -1;
        }
        AtomicInteger numSkipped = new AtomicInteger(0);
        HTableDescriptor[] allTables = this.getTables(numSkipped);
        this.errors.print("Number of Tables: " + allTables.length);
        if (details) {
            if (numSkipped.get() > 0) {
                this.errors.detail("Number of Tables in flux: " + numSkipped.get());
            }
            for (HTableDescriptor td : allTables) {
                String tableName = td.getNameAsString();
                this.errors.detail("  Table: " + tableName + "\t" + (td.isReadOnly() ? "ro" : "rw") + "\t" + (td.isRootRegion() ? "ROOT" : (td.isMetaRegion() ? "META" : "    ")) + "\t" + " families: " + td.getFamilies().size());
            }
        }
        Collection<HServerInfo> regionServers = this.status.getServerInfo();
        this.errors.print("Number of live region servers: " + regionServers.size());
        if (details) {
            for (HServerInfo rsinfo : regionServers) {
                this.errors.print("  " + rsinfo.getServerName());
            }
        }
        Collection<String> deadRegionServers = this.status.getDeadServerNames();
        this.errors.print("Number of dead region servers: " + deadRegionServers.size());
        if (details) {
            for (String name : deadRegionServers) {
                this.errors.print("  " + name);
            }
        }
        this.processRegionServers(regionServers);
        this.checkHdfs();
        this.errors.print("Number of empty REGIONINFO_QUALIFIER rows in .META.: " + this.emptyRegionInfoQualifiers.size());
        if (details) {
            for (Result r : this.emptyRegionInfoQualifiers) {
                this.errors.print("  " + r);
            }
        }
        this.loadDisabledTables();
        this.checkConsistency();
        this.checkIntegrity();
        this.printTableSummary();
        return this.errors.summarize();
    }

    public ErrorReporter getErrors() {
        return this.errors;
    }

    private void loadDisabledTables() throws ZooKeeperConnectionException, IOException, KeeperException {
        ZooKeeperWatcher zkw = HConnectionManager.getConnection(this.conf).getZooKeeperWatcher();
        for (String tableName : ZKTable.getDisabledOrDisablingTables(zkw)) {
            this.disabledTables.add(Bytes.toBytes(tableName));
        }
    }

    private boolean isTableDisabled(HRegionInfo regionInfo) {
        return this.disabledTables.contains(regionInfo.getTableDesc().getName());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void checkHdfs() throws IOException, InterruptedException {
        FileStatus[] files;
        Path rootDir = new Path(this.conf.get("hbase.rootdir"));
        FileSystem fs = rootDir.getFileSystem(this.conf);
        ArrayList tableDirs = Lists.newArrayList();
        boolean foundVersionFile = false;
        for (FileStatus file : files = fs.listStatus(rootDir)) {
            if (file.getPath().getName().equals("hbase.version")) {
                foundVersionFile = true;
                continue;
            }
            tableDirs.add(file);
        }
        if (!foundVersionFile) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.NO_VERSION_FILE, "Version file does not exist in root dir " + rootDir);
        }
        WorkItemHdfsDir[] dirs = new WorkItemHdfsDir[tableDirs.size()];
        int num = 0;
        for (FileStatus tableDir : tableDirs) {
            dirs[num] = new WorkItemHdfsDir(this, fs, this.errors, tableDir);
            this.executor.execute(dirs[num]);
            ++num;
        }
        for (int i = 0; i < num; ++i) {
            WorkItemHdfsDir workItemHdfsDir = dirs[i];
            synchronized (workItemHdfsDir) {
                while (!dirs[i].isDone()) {
                    dirs[i].wait();
                }
                continue;
            }
        }
    }

    boolean recordRootRegion() throws IOException {
        HRegionLocation rootLocation = this.connection.locateRegion(HConstants.ROOT_TABLE_NAME, HConstants.EMPTY_START_ROW);
        if (rootLocation == null || rootLocation.getRegionInfo() == null || rootLocation.getServerAddress() == null) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.NULL_ROOT_REGION, "Root Region or some of its attributes are null.");
            return false;
        }
        MetaEntry m = new MetaEntry(rootLocation.getRegionInfo(), rootLocation.getServerAddress(), null, System.currentTimeMillis());
        HbckInfo hbInfo = new HbckInfo(m);
        this.regionInfo.put(rootLocation.getRegionInfo().getEncodedName(), hbInfo);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void processRegionServers(Collection<HServerInfo> regionServerList) throws IOException, InterruptedException {
        WorkItemRegion[] work = new WorkItemRegion[regionServerList.size()];
        int num = 0;
        for (HServerInfo rsinfo : regionServerList) {
            work[num] = new WorkItemRegion(this, rsinfo, this.errors, this.connection);
            this.executor.execute(work[num]);
            ++num;
        }
        for (int i = 0; i < num; ++i) {
            WorkItemRegion workItemRegion = work[i];
            synchronized (workItemRegion) {
                while (!work[i].isDone()) {
                    work[i].wait();
                }
                continue;
            }
        }
    }

    void checkConsistency() throws IOException, KeeperException, InterruptedException {
        for (Map.Entry<String, HbckInfo> e : this.regionInfo.entrySet()) {
            this.doConsistencyCheck(e.getKey(), e.getValue());
        }
    }

    void doConsistencyCheck(String key, HbckInfo hbi) throws IOException, KeeperException, InterruptedException {
        boolean recentlyModified;
        boolean deploymentMatchesMeta;
        String descriptiveName = hbi.toString();
        boolean inMeta = hbi.metaEntry != null;
        boolean inHdfs = hbi.foundRegionDir != null;
        boolean hasMetaAssignment = inMeta && hbi.metaEntry.regionServer != null;
        boolean isDeployed = !hbi.deployedOn.isEmpty();
        boolean isMultiplyDeployed = hbi.deployedOn.size() > 1;
        boolean bl = deploymentMatchesMeta = hasMetaAssignment && isDeployed && !isMultiplyDeployed && hbi.metaEntry.regionServer.equals(hbi.deployedOn.get(0));
        boolean splitParent = hbi.metaEntry == null ? false : hbi.metaEntry.isSplit() && hbi.metaEntry.isOffline();
        boolean shouldBeDeployed = inMeta && !this.isTableDisabled(hbi.metaEntry);
        boolean bl2 = recentlyModified = hbi.foundRegionDir != null && hbi.foundRegionDir.getModificationTime() + this.timelag > System.currentTimeMillis();
        if (hbi.onlyEdits) {
            return;
        }
        if (inMeta && inHdfs && isDeployed && deploymentMatchesMeta && shouldBeDeployed) {
            return;
        }
        if (inMeta && !isDeployed && splitParent) {
            LOG.debug((Object)("Region " + descriptiveName + " offline, split, parent, ignoring."));
            return;
        }
        if (inMeta && !shouldBeDeployed && !isDeployed) {
            LOG.debug((Object)("Region " + descriptiveName + " offline, ignoring."));
            return;
        }
        if (recentlyModified) {
            LOG.warn((Object)("Region " + descriptiveName + " was recently modified -- skipping"));
            return;
        }
        if (!(inMeta || inHdfs || isDeployed)) {
            assert (false) : "Entry for region with no data";
        } else if (!inMeta && !inHdfs && isDeployed) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.NOT_IN_META_HDFS, "Region " + descriptiveName + ", key=" + key + ", not on HDFS or in META but " + "deployed on " + Joiner.on((String)", ").join(hbi.deployedOn));
        } else if (!inMeta && inHdfs && !isDeployed) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.NOT_IN_META_OR_DEPLOYED, "Region " + descriptiveName + " on HDFS, but not listed in META " + "or deployed on any region server.");
        } else if (!inMeta && inHdfs && isDeployed) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.NOT_IN_META, "Region " + descriptiveName + " not in META, but deployed on " + Joiner.on((String)", ").join(hbi.deployedOn));
        } else if (inMeta && !inHdfs && !isDeployed) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.NOT_IN_HDFS_OR_DEPLOYED, "Region " + descriptiveName + " found in META, but not in HDFS " + "or deployed on any region server.");
        } else if (inMeta && !inHdfs && isDeployed) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.NOT_IN_HDFS, "Region " + descriptiveName + " found in META, but not in HDFS, " + "and deployed on " + Joiner.on((String)", ").join(hbi.deployedOn));
        } else if (inMeta && inHdfs && !isDeployed && shouldBeDeployed) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.NOT_DEPLOYED, "Region " + descriptiveName + " not deployed on any region server.");
            if (this.shouldFix()) {
                this.errors.print("Trying to fix unassigned region...");
                this.setShouldRerun();
                HBaseFsckRepair.fixUnassigned(this.conf, hbi.metaEntry);
            }
        } else if (inMeta && inHdfs && isDeployed && !shouldBeDeployed) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.SHOULD_NOT_BE_DEPLOYED, "Region " + descriptiveName + " should not be deployed according " + "to META, but is deployed on " + Joiner.on((String)", ").join(hbi.deployedOn));
        } else if (inMeta && inHdfs && isMultiplyDeployed) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.MULTI_DEPLOYED, "Region " + descriptiveName + " is listed in META on region server " + hbi.metaEntry.regionServer + " but is multiply assigned to region servers " + Joiner.on((String)", ").join(hbi.deployedOn));
            if (this.shouldFix()) {
                this.errors.print("Trying to fix assignment error...");
                this.setShouldRerun();
                HBaseFsckRepair.fixDupeAssignment(this.conf, hbi.metaEntry, hbi.deployedOn);
            }
        } else if (inMeta && inHdfs && isDeployed && !deploymentMatchesMeta) {
            this.errors.reportError(ErrorReporter.ERROR_CODE.SERVER_DOES_NOT_MATCH_META, "Region " + descriptiveName + " listed in META on region server " + hbi.metaEntry.regionServer + " but found on region server " + hbi.deployedOn.get(0));
            if (this.shouldFix()) {
                this.errors.print("Trying to fix assignment error...");
                this.setShouldRerun();
                HBaseFsckRepair.fixDupeAssignment(this.conf, hbi.metaEntry, hbi.deployedOn);
            }
        } else {
            this.errors.reportError(ErrorReporter.ERROR_CODE.UNKNOWN, "Region " + descriptiveName + " is in an unforeseen state:" + " inMeta=" + inMeta + " inHdfs=" + inHdfs + " isDeployed=" + isDeployed + " isMultiplyDeployed=" + isMultiplyDeployed + " deploymentMatchesMeta=" + deploymentMatchesMeta + " shouldBeDeployed=" + shouldBeDeployed);
        }
    }

    void checkIntegrity() {
        for (HbckInfo hbi : this.regionInfo.values()) {
            if (hbi.metaEntry == null || hbi.metaEntry.regionServer == null || hbi.onlyEdits || hbi.deployedOn.size() == 0) continue;
            String tableName = hbi.metaEntry.getTableDesc().getNameAsString();
            TInfo modTInfo = this.tablesInfo.get(tableName);
            if (modTInfo == null) {
                modTInfo = new TInfo(tableName);
            }
            for (HServerAddress server : hbi.deployedOn) {
                modTInfo.addServer(server);
            }
            modTInfo.addRegionInfo(hbi);
            this.tablesInfo.put(tableName, modTInfo);
        }
        for (TInfo tInfo : this.tablesInfo.values()) {
            if (tInfo.checkRegionChain()) continue;
            this.errors.report("Found inconsistency in table " + tInfo.getName());
        }
    }

    HTableDescriptor[] getTables(AtomicInteger numSkipped) {
        TreeSet<HTableDescriptor> uniqueTables = new TreeSet<HTableDescriptor>();
        long now = System.currentTimeMillis();
        for (HbckInfo hbi : this.regionInfo.values()) {
            MetaEntry info = hbi.metaEntry;
            if (info == null || info.getStartKey().length != 0 || info.isMetaRegion()) continue;
            if (info.modTime + this.timelag < now) {
                uniqueTables.add(info.getTableDesc());
                continue;
            }
            numSkipped.incrementAndGet();
        }
        return uniqueTables.toArray(new HTableDescriptor[uniqueTables.size()]);
    }

    private synchronized HbckInfo getOrCreateInfo(String name) {
        HbckInfo hbi = this.regionInfo.get(name);
        if (hbi == null) {
            hbi = new HbckInfo(null);
            this.regionInfo.put(name, hbi);
        }
        return hbi;
    }

    boolean checkMetaEntries() throws IOException, KeeperException, InterruptedException {
        ArrayList metaRegions = Lists.newArrayList();
        for (HbckInfo value : this.regionInfo.values()) {
            if (!value.metaEntry.isMetaTable()) continue;
            metaRegions.add(value);
        }
        if (metaRegions.size() != 1) {
            HRegionLocation rootLocation = this.connection.locateRegion(HConstants.ROOT_TABLE_NAME, HConstants.EMPTY_START_ROW);
            HbckInfo root = this.regionInfo.get(rootLocation.getRegionInfo().getEncodedName());
            if (metaRegions.size() == 0) {
                this.errors.reportError(ErrorReporter.ERROR_CODE.NO_META_REGION, ".META. is not found on any region.");
                if (this.shouldFix()) {
                    this.errors.print("Trying to fix a problem with .META...");
                    this.setShouldRerun();
                    HBaseFsckRepair.fixUnassigned(this.conf, root.metaEntry);
                }
            } else if (metaRegions.size() > 1) {
                this.errors.reportError(ErrorReporter.ERROR_CODE.MULTI_META_REGION, ".META. is found on more than one region.");
                if (this.shouldFix()) {
                    this.errors.print("Trying to fix a problem with .META...");
                    this.setShouldRerun();
                    ArrayList deployedOn = Lists.newArrayList();
                    for (HbckInfo mRegion : metaRegions) {
                        deployedOn.add(mRegion.metaEntry.regionServer);
                    }
                    HBaseFsckRepair.fixDupeAssignment(this.conf, root.metaEntry, deployedOn);
                }
            }
            return false;
        }
        return true;
    }

    void getMetaEntries() throws IOException {
        MetaScanner.MetaScannerVisitor visitor = new MetaScanner.MetaScannerVisitor(){
            int countRecord = 1;
            final Comparator<KeyValue> comp = new Comparator<KeyValue>(){

                @Override
                public int compare(KeyValue k1, KeyValue k2) {
                    return (int)(k1.getTimestamp() - k2.getTimestamp());
                }
            };

            @Override
            public boolean processRow(Result result) throws IOException {
                try {
                    long ts = Collections.max(result.list(), this.comp).getTimestamp();
                    byte[] value = result.getValue(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER);
                    if (value == null || value.length == 0) {
                        HBaseFsck.this.emptyRegionInfoQualifiers.add(result);
                        return true;
                    }
                    HRegionInfo info = Writables.getHRegionInfo(value);
                    HServerAddress server = null;
                    byte[] startCode = null;
                    value = result.getValue(HConstants.CATALOG_FAMILY, HConstants.SERVER_QUALIFIER);
                    if (value != null && value.length > 0) {
                        String address = Bytes.toString(value);
                        server = new HServerAddress(address);
                    }
                    if ((value = result.getValue(HConstants.CATALOG_FAMILY, HConstants.STARTCODE_QUALIFIER)) != null) {
                        startCode = value;
                    }
                    MetaEntry m = new MetaEntry(info, server, startCode, ts);
                    HbckInfo hbInfo = new HbckInfo(m);
                    HbckInfo previous = HBaseFsck.this.regionInfo.put(info.getEncodedName(), hbInfo);
                    if (previous != null) {
                        throw new IOException("Two entries in META are same " + previous);
                    }
                    if (this.countRecord % 100 == 0) {
                        HBaseFsck.this.errors.progress();
                    }
                    ++this.countRecord;
                    return true;
                }
                catch (RuntimeException e) {
                    LOG.error((Object)("Result=" + result));
                    throw e;
                }
            }
        };
        MetaScanner.metaScan(this.conf, visitor, null, null, Integer.MAX_VALUE, HConstants.ROOT_TABLE_NAME);
        MetaScanner.metaScan(this.conf, visitor);
        this.errors.print("");
    }

    private void printTableSummary() {
        System.out.println("Summary:");
        for (TInfo tInfo : this.tablesInfo.values()) {
            if (this.errors.tableHasErrors(tInfo)) {
                System.out.println("Table " + tInfo.getName() + " is inconsistent.");
            } else {
                System.out.println("  " + tInfo.getName() + " is okay.");
            }
            System.out.println("    Number of regions: " + tInfo.getNumRegions());
            System.out.print("    Deployed on: ");
            for (HServerAddress server : tInfo.deployedOn) {
                System.out.print(" " + server.toString());
            }
            System.out.println();
        }
    }

    void displayFullReport() {
        details = true;
    }

    void setSummary() {
        summary = true;
    }

    void setShouldRerun() {
        this.rerun = true;
    }

    boolean shouldRerun() {
        return this.rerun;
    }

    void setFixErrors(boolean shouldFix) {
        this.fix = shouldFix;
    }

    boolean shouldFix() {
        return this.fix;
    }

    void setTimeLag(long seconds) {
        this.timelag = seconds * 1000L;
    }

    protected static void printUsageAndExit() {
        System.err.println("Usage: fsck [opts] ");
        System.err.println(" where [opts] are:");
        System.err.println("   -details Display full report of all regions.");
        System.err.println("   -timelag {timeInSeconds}  Process only regions that  have not experienced any metadata updates in the last  {{timeInSeconds} seconds.");
        System.err.println("   -fix Try to fix some of the errors.");
        System.err.println("   -sleepBeforeRerun {timeInSeconds} Sleep this many seconds before checking if the fix worked if run with -fix");
        System.err.println("   -summary Print only summary of the tables and status.");
        Runtime.getRuntime().exit(-2);
    }

    public static void main(String[] args) throws Exception {
        Configuration conf = HBaseConfiguration.create();
        conf.set("fs.defaultFS", conf.get("hbase.rootdir"));
        HBaseFsck fsck = new HBaseFsck(conf);
        long sleepBeforeRerun = 10000L;
        for (int i = 0; i < args.length; ++i) {
            String cmd = args[i];
            if (cmd.equals("-details")) {
                fsck.displayFullReport();
                continue;
            }
            if (cmd.equals("-timelag")) {
                if (i == args.length - 1) {
                    System.err.println("HBaseFsck: -timelag needs a value.");
                    HBaseFsck.printUsageAndExit();
                }
                try {
                    long timelag = Long.parseLong(args[i + 1]);
                    fsck.setTimeLag(timelag);
                }
                catch (NumberFormatException e) {
                    System.err.println("-timelag needs a numeric value.");
                    HBaseFsck.printUsageAndExit();
                }
                ++i;
                continue;
            }
            if (cmd.equals("-sleepBeforeRerun")) {
                if (i == args.length - 1) {
                    System.err.println("HBaseFsck: -sleepBeforeRerun needs a value.");
                    HBaseFsck.printUsageAndExit();
                }
                try {
                    sleepBeforeRerun = Long.parseLong(args[i + 1]);
                }
                catch (NumberFormatException e) {
                    System.err.println("-sleepBeforeRerun needs a numeric value.");
                    HBaseFsck.printUsageAndExit();
                }
                ++i;
                continue;
            }
            if (cmd.equals("-fix")) {
                fsck.setFixErrors(true);
                continue;
            }
            if (cmd.equals("-summary")) {
                fsck.setSummary();
                continue;
            }
            String str = "Unknown command line option : " + cmd;
            LOG.info((Object)str);
            System.out.println(str);
            HBaseFsck.printUsageAndExit();
        }
        int code = fsck.doWork();
        if (fsck.shouldRerun()) {
            try {
                LOG.info((Object)("Sleeping " + sleepBeforeRerun + "ms before re-checking after fix..."));
                Thread.sleep(sleepBeforeRerun);
            }
            catch (InterruptedException ie) {
                Runtime.getRuntime().exit(code);
            }
            fsck.setFixErrors(false);
            fsck.errors.resetErrors();
            code = fsck.doWork();
        }
        Runtime.getRuntime().exit(code);
    }

    static class WorkItemHdfsDir
    implements Runnable {
        private HBaseFsck hbck;
        private FileStatus tableDir;
        private ErrorReporter errors;
        private FileSystem fs;
        private boolean done;

        WorkItemHdfsDir(HBaseFsck hbck, FileSystem fs, ErrorReporter errors, FileStatus status) {
            this.hbck = hbck;
            this.fs = fs;
            this.tableDir = status;
            this.errors = errors;
            this.done = false;
        }

        synchronized boolean isDone() {
            return this.done;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public synchronized void run() {
            try {
                FileStatus[] regionDirs;
                String tableName = this.tableDir.getPath().getName();
                if (tableName.startsWith(".") && !tableName.equals(Bytes.toString(HConstants.META_TABLE_NAME))) {
                    return;
                }
                for (FileStatus regionDir : regionDirs = this.fs.listStatus(this.tableDir.getPath())) {
                    HbckInfo hbi;
                    String encodedName = regionDir.getPath().getName();
                    if (!encodedName.toLowerCase().matches("[0-9a-f]+")) continue;
                    HbckInfo hbckInfo = hbi = this.hbck.getOrCreateInfo(encodedName);
                    synchronized (hbckInfo) {
                        if (hbi.foundRegionDir != null) {
                            this.errors.print("Directory " + encodedName + " duplicate??" + hbi.foundRegionDir);
                        }
                        hbi.foundRegionDir = regionDir;
                        hbi.onlyEdits = true;
                        FileStatus[] subDirs = this.fs.listStatus(regionDir.getPath());
                        Path ePath = HLog.getRegionDirRecoveredEditsDir(regionDir.getPath());
                        for (FileStatus subDir : subDirs) {
                            String sdName = subDir.getPath().getName();
                            if (sdName.startsWith(".") || sdName.equals(ePath.getName())) continue;
                            hbi.onlyEdits = false;
                            break;
                        }
                    }
                }
            }
            catch (IOException e) {
                this.errors.reportError(ErrorReporter.ERROR_CODE.RS_CONNECT_FAILURE, "Table Directory: " + this.tableDir.getPath().getName() + " Unable to fetch region information. " + e);
            }
            finally {
                this.done = true;
                this.notifyAll();
            }
        }
    }

    static class WorkItemRegion
    implements Runnable {
        private HBaseFsck hbck;
        private HServerInfo rsinfo;
        private ErrorReporter errors;
        private HConnection connection;
        private boolean done;

        WorkItemRegion(HBaseFsck hbck, HServerInfo info, ErrorReporter errors, HConnection connection) {
            this.hbck = hbck;
            this.rsinfo = info;
            this.errors = errors;
            this.connection = connection;
            this.done = false;
        }

        synchronized boolean isDone() {
            return this.done;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public synchronized void run() {
            this.errors.progress();
            try {
                HRegionInterface server = this.connection.getHRegionConnection(this.rsinfo.getServerAddress());
                List<HRegionInfo> regions = server.getOnlineRegions();
                if (details) {
                    this.errors.detail("RegionServer: " + this.rsinfo.getServerName() + " number of regions: " + regions.size());
                    for (HRegionInfo rinfo : regions) {
                        this.errors.detail("  " + rinfo.getRegionNameAsString() + " id: " + rinfo.getRegionId() + " encoded_name: " + rinfo.getEncodedName() + " start: " + Bytes.toStringBinary(rinfo.getStartKey()) + " end: " + Bytes.toStringBinary(rinfo.getEndKey()));
                    }
                }
                for (HRegionInfo r : regions) {
                    HbckInfo hbi = this.hbck.getOrCreateInfo(r.getEncodedName());
                    hbi.addServer(this.rsinfo.getServerAddress());
                }
            }
            catch (IOException e) {
                this.errors.reportError(ErrorReporter.ERROR_CODE.RS_CONNECT_FAILURE, "RegionServer: " + this.rsinfo.getServerName() + " Unable to fetch region information. " + e);
            }
            finally {
                this.done = true;
                this.notifyAll();
            }
        }
    }

    private static class PrintingErrorReporter
    implements ErrorReporter {
        public int errorCount = 0;
        private int showProgress;
        Set<TInfo> errorTables = new HashSet<TInfo>();
        private ArrayList<ErrorReporter.ERROR_CODE> errorList = new ArrayList();

        private PrintingErrorReporter() {
        }

        @Override
        public void clear() {
            this.errorTables.clear();
            this.errorList.clear();
            this.errorCount = 0;
        }

        @Override
        public synchronized void reportError(ErrorReporter.ERROR_CODE errorCode, String message) {
            this.errorList.add(errorCode);
            if (!summary) {
                System.out.println("ERROR: " + message);
            }
            ++this.errorCount;
            this.showProgress = 0;
        }

        @Override
        public synchronized void reportError(ErrorReporter.ERROR_CODE errorCode, String message, TInfo table, HbckInfo info) {
            this.errorTables.add(table);
            String reference = "(region " + info.metaEntry.getRegionNameAsString() + ")";
            this.reportError(errorCode, reference + " " + message);
        }

        @Override
        public synchronized void reportError(ErrorReporter.ERROR_CODE errorCode, String message, TInfo table, HbckInfo info1, HbckInfo info2) {
            this.errorTables.add(table);
            String reference = "(regions " + info1.metaEntry.getRegionNameAsString() + " and " + info2.metaEntry.getRegionNameAsString() + ")";
            this.reportError(errorCode, reference + " " + message);
        }

        @Override
        public synchronized void reportError(String message) {
            this.reportError(ErrorReporter.ERROR_CODE.UNKNOWN, message);
        }

        @Override
        public synchronized void report(String message) {
            if (!summary) {
                System.out.println("ERROR: " + message);
            }
            this.showProgress = 0;
        }

        @Override
        public synchronized int summarize() {
            System.out.println(Integer.toString(this.errorCount) + " inconsistencies detected.");
            if (this.errorCount == 0) {
                System.out.println("Status: OK");
                return 0;
            }
            System.out.println("Status: INCONSISTENT");
            return -1;
        }

        @Override
        public ArrayList<ErrorReporter.ERROR_CODE> getErrorList() {
            return this.errorList;
        }

        @Override
        public synchronized void print(String message) {
            if (!summary) {
                System.out.println(message);
            }
        }

        @Override
        public boolean tableHasErrors(TInfo table) {
            return this.errorTables.contains(table);
        }

        @Override
        public void resetErrors() {
            this.errorCount = 0;
        }

        @Override
        public synchronized void detail(String message) {
            if (details) {
                System.out.println(message);
            }
            this.showProgress = 0;
        }

        @Override
        public synchronized void progress() {
            if (this.showProgress++ == 10) {
                if (!summary) {
                    System.out.print(".");
                }
                this.showProgress = 0;
            }
        }
    }

    static interface ErrorReporter {
        public void clear();

        public void report(String var1);

        public void reportError(String var1);

        public void reportError(ERROR_CODE var1, String var2);

        public void reportError(ERROR_CODE var1, String var2, TInfo var3, HbckInfo var4);

        public void reportError(ERROR_CODE var1, String var2, TInfo var3, HbckInfo var4, HbckInfo var5);

        public int summarize();

        public void detail(String var1);

        public ArrayList<ERROR_CODE> getErrorList();

        public void progress();

        public void print(String var1);

        public void resetErrors();

        public boolean tableHasErrors(TInfo var1);

        public static enum ERROR_CODE {
            UNKNOWN,
            NO_META_REGION,
            NULL_ROOT_REGION,
            NO_VERSION_FILE,
            NOT_IN_META_HDFS,
            NOT_IN_META,
            NOT_IN_META_OR_DEPLOYED,
            NOT_IN_HDFS_OR_DEPLOYED,
            NOT_IN_HDFS,
            SERVER_DOES_NOT_MATCH_META,
            NOT_DEPLOYED,
            MULTI_DEPLOYED,
            SHOULD_NOT_BE_DEPLOYED,
            MULTI_META_REGION,
            RS_CONNECT_FAILURE,
            FIRST_REGION_STARTKEY_NOT_EMPTY,
            DUPE_STARTKEYS,
            HOLE_IN_REGION_CHAIN,
            OVERLAP_IN_REGION_CHAIN,
            REGION_CYCLE;

        }
    }

    static class HbckInfo
    implements Comparable {
        boolean onlyEdits = false;
        MetaEntry metaEntry = null;
        FileStatus foundRegionDir = null;
        List<HServerAddress> deployedOn = Lists.newArrayList();

        HbckInfo(MetaEntry metaEntry) {
            this.metaEntry = metaEntry;
        }

        public synchronized void addServer(HServerAddress server) {
            this.deployedOn.add(server);
        }

        public synchronized String toString() {
            if (this.metaEntry != null) {
                return this.metaEntry.getRegionNameAsString();
            }
            if (this.foundRegionDir != null) {
                return this.foundRegionDir.getPath().toString();
            }
            return "UNKNOWN_REGION on " + Joiner.on((String)", ").join(this.deployedOn);
        }

        public int compareTo(Object o) {
            HbckInfo other = (HbckInfo)o;
            int startComparison = Bytes.compareTo(this.metaEntry.getStartKey(), other.metaEntry.getStartKey());
            if (startComparison != 0) {
                return startComparison;
            }
            return Bytes.compareTo(this.metaEntry.getEndKey(), other.metaEntry.getEndKey());
        }
    }

    private static class MetaEntry
    extends HRegionInfo {
        HServerAddress regionServer;
        long modTime;

        public MetaEntry(HRegionInfo rinfo, HServerAddress regionServer, byte[] startCode, long modTime) {
            super(rinfo);
            this.regionServer = regionServer;
            this.modTime = modTime;
        }
    }

    private class TInfo {
        String tableName;
        TreeSet<HServerAddress> deployedOn;
        List<HbckInfo> regions = new ArrayList<HbckInfo>();

        TInfo(String name) {
            this.tableName = name;
            this.deployedOn = new TreeSet();
        }

        public void addRegionInfo(HbckInfo r) {
            this.regions.add(r);
        }

        public void addServer(HServerAddress server) {
            this.deployedOn.add(server);
        }

        public String getName() {
            return this.tableName;
        }

        public int getNumRegions() {
            return this.regions.size();
        }

        public boolean checkRegionChain() {
            Collections.sort(this.regions);
            HbckInfo last = null;
            for (HbckInfo r : this.regions) {
                if (last == null) {
                    if (!Bytes.equals(r.metaEntry.getStartKey(), HConstants.EMPTY_BYTE_ARRAY)) {
                        HBaseFsck.this.errors.reportError(ErrorReporter.ERROR_CODE.FIRST_REGION_STARTKEY_NOT_EMPTY, "First region should start with an empty key.", this, r);
                    }
                } else {
                    int cmpRegionKeys;
                    if (!Bytes.equals(r.metaEntry.getEndKey(), HConstants.EMPTY_BYTE_ARRAY) && (cmpRegionKeys = Bytes.compareTo(r.metaEntry.getStartKey(), r.metaEntry.getEndKey())) > 0) {
                        HBaseFsck.this.errors.reportError(ErrorReporter.ERROR_CODE.REGION_CYCLE, String.format("The endkey for this region comes before the startkey, startkey=%s, endkey=%s", Bytes.toString(r.metaEntry.getStartKey()), Bytes.toString(r.metaEntry.getEndKey())), this, r, last);
                    }
                    if (Bytes.equals(r.metaEntry.getStartKey(), last.metaEntry.getStartKey())) {
                        HBaseFsck.this.errors.reportError(ErrorReporter.ERROR_CODE.DUPE_STARTKEYS, "Two regions have the same startkey: " + Bytes.toString(r.metaEntry.getStartKey()), this, r, last);
                    } else {
                        int cmp = Bytes.compareTo(r.metaEntry.getStartKey(), last.metaEntry.getEndKey());
                        if (cmp > 0) {
                            HBaseFsck.this.errors.reportError(ErrorReporter.ERROR_CODE.HOLE_IN_REGION_CHAIN, "There is a hole in the region chain.", this, r, last);
                        } else if (cmp < 0) {
                            HBaseFsck.this.errors.reportError(ErrorReporter.ERROR_CODE.OVERLAP_IN_REGION_CHAIN, "There is an overlap in the region chain.", this, r, last);
                        }
                    }
                }
                last = r;
            }
            return HBaseFsck.this.errors.getErrorList().size() == 0;
        }
    }
}

