/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.mob;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.Tag;
import org.apache.hadoop.hbase.backup.HFileArchiver;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.io.compress.Compression;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.mob.MobConstants;
import org.apache.hadoop.hbase.mob.MobFileName;
import org.apache.hadoop.hbase.regionserver.BloomType;
import org.apache.hadoop.hbase.regionserver.StoreFile;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSUtils;

@InterfaceAudience.Private
public class MobUtils {
    private static final Log LOG = LogFactory.getLog(MobUtils.class);
    private static final ThreadLocal<SimpleDateFormat> LOCAL_FORMAT = new ThreadLocal<SimpleDateFormat>(){

        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yyyyMMdd");
        }
    };

    public static String formatDate(Date date) {
        return LOCAL_FORMAT.get().format(date);
    }

    public static Date parseDate(String dateString) throws ParseException {
        return LOCAL_FORMAT.get().parse(dateString);
    }

    public static boolean isMobReferenceCell(Cell cell) {
        if (cell.getTagsLengthUnsigned() > 0) {
            Tag tag = Tag.getTag((byte[])cell.getTagsArray(), (int)cell.getTagsOffset(), (int)cell.getTagsLengthUnsigned(), (byte)5);
            return tag != null;
        }
        return false;
    }

    public static boolean hasMobReferenceTag(List<Tag> tags) {
        if (!tags.isEmpty()) {
            for (Tag tag : tags) {
                if (tag.getType() != 5) continue;
                return true;
            }
        }
        return false;
    }

    public static boolean isRawMobScan(Scan scan) {
        byte[] raw = scan.getAttribute("hbase.mob.scan.raw");
        try {
            return raw != null && Bytes.toBoolean((byte[])raw);
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static boolean isRefOnlyScan(Scan scan) {
        byte[] refOnly = scan.getAttribute("hbase.mob.scan.ref.only");
        try {
            return refOnly != null && Bytes.toBoolean((byte[])refOnly);
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static boolean isCacheMobBlocks(Scan scan) {
        byte[] cache = scan.getAttribute("hbase.mob.cache.blocks");
        try {
            return cache != null && Bytes.toBoolean((byte[])cache);
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static void setCacheMobBlocks(Scan scan, boolean cacheBlocks) {
        scan.setAttribute("hbase.mob.cache.blocks", Bytes.toBytes((boolean)cacheBlocks));
    }

    public static void cleanExpiredMobFiles(FileSystem fs, Configuration conf, TableName tableName, HColumnDescriptor columnDescriptor, CacheConfig cacheConfig, long current) throws IOException {
        long timeToLive = columnDescriptor.getTimeToLive();
        if (Integer.MAX_VALUE == timeToLive) {
            return;
        }
        Date expireDate = new Date(current - timeToLive * 1000L);
        expireDate = new Date(expireDate.getYear(), expireDate.getMonth(), expireDate.getDate());
        LOG.info((Object)("MOB HFiles older than " + expireDate.toGMTString() + " will be deleted!"));
        FileStatus[] stats = null;
        Path mobTableDir = FSUtils.getTableDir(MobUtils.getMobHome(conf), tableName);
        Path path = MobUtils.getMobFamilyPath(conf, tableName, columnDescriptor.getNameAsString());
        try {
            stats = fs.listStatus(path);
        }
        catch (FileNotFoundException e) {
            LOG.warn((Object)("Fail to find the mob file " + path), (Throwable)e);
        }
        if (null == stats) {
            return;
        }
        ArrayList<StoreFile> filesToClean = new ArrayList<StoreFile>();
        int deletedFileCount = 0;
        for (FileStatus file : stats) {
            String fileName = file.getPath().getName();
            try {
                MobFileName mobFileName = null;
                if (!HFileLink.isHFileLink(file.getPath())) {
                    mobFileName = MobFileName.create(fileName);
                } else {
                    HFileLink hfileLink = new HFileLink(conf, file.getPath());
                    mobFileName = MobFileName.create(hfileLink.getOriginPath().getName());
                }
                Date fileDate = MobUtils.parseDate(mobFileName.getDate());
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("Checking file " + fileName));
                }
                if (fileDate.getTime() >= expireDate.getTime()) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)(fileName + " is an expired file"));
                }
                filesToClean.add(new StoreFile(fs, file.getPath(), conf, cacheConfig, BloomType.NONE));
            }
            catch (Exception e) {
                LOG.error((Object)("Cannot parse the fileName " + fileName), (Throwable)e);
            }
        }
        if (!filesToClean.isEmpty()) {
            try {
                MobUtils.removeMobFiles(conf, fs, tableName, mobTableDir, columnDescriptor.getName(), filesToClean);
                deletedFileCount = filesToClean.size();
            }
            catch (IOException e) {
                LOG.error((Object)("Fail to delete the mob files " + filesToClean), (Throwable)e);
            }
        }
        LOG.info((Object)(deletedFileCount + " expired mob files are deleted"));
    }

    public static String getColumnFamilyZNodeName(String tableName, String familyName) {
        return tableName + ":" + familyName;
    }

    public static Path getMobHome(Configuration conf) {
        Path hbaseDir = new Path(conf.get("hbase.rootdir"));
        return new Path(hbaseDir, "mobdir");
    }

    public static Path getQualifiedMobRootDir(Configuration conf) throws IOException {
        Path hbaseDir = new Path(conf.get("hbase.rootdir"));
        Path mobRootDir = new Path(hbaseDir, "mobdir");
        FileSystem fs = mobRootDir.getFileSystem(conf);
        return mobRootDir.makeQualified(fs);
    }

    public static Path getMobRegionPath(Configuration conf, TableName tableName) {
        Path tablePath = FSUtils.getTableDir(MobUtils.getMobHome(conf), tableName);
        HRegionInfo regionInfo = MobUtils.getMobRegionInfo(tableName);
        return new Path(tablePath, regionInfo.getEncodedName());
    }

    public static Path getMobFamilyPath(Configuration conf, TableName tableName, String familyName) {
        return new Path(MobUtils.getMobRegionPath(conf, tableName), familyName);
    }

    public static Path getMobFamilyPath(Path regionPath, String familyName) {
        return new Path(regionPath, familyName);
    }

    public static HRegionInfo getMobRegionInfo(TableName tableName) {
        HRegionInfo info = new HRegionInfo(tableName, MobConstants.MOB_REGION_NAME_BYTES, HConstants.EMPTY_END_ROW, false, 0L);
        return info;
    }

    public static boolean isMobRegionInfo(HRegionInfo regionInfo) {
        return regionInfo == null ? false : MobUtils.getMobRegionInfo(regionInfo.getTable()).getEncodedName().equals(regionInfo.getEncodedName());
    }

    public static Path getCompactionWorkingPath(Path root, String jobName) {
        return new Path(root, jobName);
    }

    public static void removeMobFiles(Configuration conf, FileSystem fs, TableName tableName, Path tableDir, byte[] family, Collection<StoreFile> storeFiles) throws IOException {
        HFileArchiver.archiveStoreFiles(conf, fs, MobUtils.getMobRegionInfo(tableName), tableDir, family, storeFiles);
    }

    public static KeyValue createMobRefKeyValue(Cell cell, byte[] fileName, Tag tableNameTag) {
        ArrayList<Tag> tags = new ArrayList<Tag>();
        tags.add(MobConstants.MOB_REF_TAG);
        tags.add(tableNameTag);
        tags.addAll(Tag.asList((byte[])cell.getTagsArray(), (int)cell.getTagsOffset(), (int)cell.getTagsLength()));
        int valueLength = cell.getValueLength();
        byte[] refValue = Bytes.add((byte[])Bytes.toBytes((int)valueLength), (byte[])fileName);
        KeyValue reference = new KeyValue(cell.getRowArray(), cell.getRowOffset(), (int)cell.getRowLength(), cell.getFamilyArray(), cell.getFamilyOffset(), (int)cell.getFamilyLength(), cell.getQualifierArray(), cell.getQualifierOffset(), cell.getQualifierLength(), cell.getTimestamp(), KeyValue.Type.Put, refValue, 0, refValue.length, tags);
        reference.setMvccVersion(cell.getMvccVersion());
        return reference;
    }

    public static StoreFile.Writer createWriter(Configuration conf, FileSystem fs, HColumnDescriptor family, String date, Path basePath, long maxKeyCount, Compression.Algorithm compression, String startKey, CacheConfig cacheConfig) throws IOException {
        MobFileName mobFileName = MobFileName.create(startKey, date, UUID.randomUUID().toString().replaceAll("-", ""));
        HFileContext hFileContext = new HFileContextBuilder().withCompression(compression).withIncludesMvcc(false).withIncludesTags(true).withChecksumType(HFile.DEFAULT_CHECKSUM_TYPE).withBytesPerCheckSum(16384).withBlockSize(family.getBlocksize()).withHBaseCheckSum(true).withDataBlockEncoding(family.getDataBlockEncoding()).build();
        StoreFile.Writer w = new StoreFile.WriterBuilder(conf, cacheConfig, fs).withFilePath(new Path(basePath, mobFileName.getFileName())).withComparator(KeyValue.COMPARATOR).withBloomType(BloomType.NONE).withMaxKeyCount(maxKeyCount).withFileContext(hFileContext).build();
        return w;
    }

    public static void commitFile(Configuration conf, FileSystem fs, Path sourceFile, Path targetPath, CacheConfig cacheConfig) throws IOException {
        if (sourceFile == null) {
            return;
        }
        Path dstPath = new Path(targetPath, sourceFile.getName());
        MobUtils.validateMobFile(conf, fs, sourceFile, cacheConfig);
        String msg = "Renaming flushed file from " + sourceFile + " to " + dstPath;
        LOG.info((Object)msg);
        Path parent = dstPath.getParent();
        if (!fs.exists(parent)) {
            fs.mkdirs(parent);
        }
        if (!fs.rename(sourceFile, dstPath)) {
            throw new IOException("Failed rename of " + sourceFile + " to " + dstPath);
        }
    }

    private static void validateMobFile(Configuration conf, FileSystem fs, Path path, CacheConfig cacheConfig) throws IOException {
        StoreFile storeFile = null;
        try {
            storeFile = new StoreFile(fs, path, conf, cacheConfig, BloomType.NONE);
            storeFile.createReader();
        }
        catch (IOException e) {
            LOG.error((Object)("Fail to open mob file[" + path + "], keep it in temp directory."), (Throwable)e);
            throw e;
        }
        finally {
            if (storeFile != null) {
                storeFile.closeReader(false);
            }
        }
    }

    public static boolean hasValidMobRefCellValue(Cell cell) {
        return cell.getValueLength() > 4;
    }

    public static int getMobValueLength(Cell cell) {
        return Bytes.toInt((byte[])cell.getValueArray(), (int)cell.getValueOffset(), (int)4);
    }

    public static String getMobFileName(Cell cell) {
        return Bytes.toString((byte[])cell.getValueArray(), (int)(cell.getValueOffset() + 4), (int)(cell.getValueLength() - 4));
    }
}

