/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation.decider;

import java.util.Map;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.ClusterInfoService;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.InternalClusterInfoService;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.routing.allocation.decider.AllocationDecider;
import org.elasticsearch.cluster.routing.allocation.decider.Decision;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.RatioValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.node.settings.NodeSettingsService;

public class DiskThresholdDecider
extends AllocationDecider {
    public static final String NAME = "disk_threshold";
    private volatile Double freeDiskThresholdLow;
    private volatile Double freeDiskThresholdHigh;
    private volatile ByteSizeValue freeBytesThresholdLow;
    private volatile ByteSizeValue freeBytesThresholdHigh;
    private volatile boolean includeRelocations;
    private volatile boolean enabled;
    private volatile TimeValue rerouteInterval;
    public static final String CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED = "cluster.routing.allocation.disk.threshold_enabled";
    public static final String CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK = "cluster.routing.allocation.disk.watermark.low";
    public static final String CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK = "cluster.routing.allocation.disk.watermark.high";
    public static final String CLUSTER_ROUTING_ALLOCATION_INCLUDE_RELOCATIONS = "cluster.routing.allocation.disk.include_relocations";
    public static final String CLUSTER_ROUTING_ALLOCATION_REROUTE_INTERVAL = "cluster.routing.allocation.disk.reroute_interval";

    public DiskThresholdDecider(Settings settings) {
        this(settings, new NodeSettingsService(settings), ClusterInfoService.EMPTY, null);
    }

    @Inject
    public DiskThresholdDecider(Settings settings, NodeSettingsService nodeSettingsService, ClusterInfoService infoService, Client client) {
        super(settings);
        String lowWatermark = settings.get(CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK, "85%");
        String highWatermark = settings.get(CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK, "90%");
        if (!this.validWatermarkSetting(lowWatermark)) {
            throw new ElasticsearchParseException("Unable to parse low watermark: [" + lowWatermark + "]");
        }
        if (!this.validWatermarkSetting(highWatermark)) {
            throw new ElasticsearchParseException("Unable to parse high watermark: [" + highWatermark + "]");
        }
        this.freeDiskThresholdLow = 100.0 - this.thresholdPercentageFromWatermark(lowWatermark);
        this.freeDiskThresholdHigh = 100.0 - this.thresholdPercentageFromWatermark(highWatermark);
        this.freeBytesThresholdLow = this.thresholdBytesFromWatermark(lowWatermark);
        this.freeBytesThresholdHigh = this.thresholdBytesFromWatermark(highWatermark);
        this.includeRelocations = settings.getAsBoolean(CLUSTER_ROUTING_ALLOCATION_INCLUDE_RELOCATIONS, (Boolean)true);
        this.rerouteInterval = settings.getAsTime(CLUSTER_ROUTING_ALLOCATION_REROUTE_INTERVAL, TimeValue.timeValueSeconds(60L));
        this.enabled = settings.getAsBoolean(CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED, (Boolean)true);
        nodeSettingsService.addListener(new ApplySettings());
        infoService.addListener(new DiskListener(client));
    }

    ApplySettings newApplySettings() {
        return new ApplySettings();
    }

    public Double getFreeDiskThresholdLow() {
        return this.freeDiskThresholdLow;
    }

    public Double getFreeDiskThresholdHigh() {
        return this.freeDiskThresholdHigh;
    }

    public ByteSizeValue getFreeBytesThresholdLow() {
        return this.freeBytesThresholdLow;
    }

    public ByteSizeValue getFreeBytesThresholdHigh() {
        return this.freeBytesThresholdHigh;
    }

    public boolean isIncludeRelocations() {
        return this.includeRelocations;
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    public TimeValue getRerouteInterval() {
        return this.rerouteInterval;
    }

    public long sizeOfRelocatingShards(RoutingNode node, Map<String, Long> shardSizes, boolean subtractShardsMovingAway) {
        long totalSize = 0L;
        for (ShardRouting shardRouting : node.shardsWithState(ShardRoutingState.RELOCATING, ShardRoutingState.INITIALIZING)) {
            if (shardRouting.initializing() && shardRouting.relocatingNodeId() != null) {
                totalSize += this.getShardSize(shardRouting, shardSizes);
                continue;
            }
            if (!subtractShardsMovingAway || !shardRouting.relocating()) continue;
            totalSize -= this.getShardSize(shardRouting, shardSizes);
        }
        return totalSize;
    }

    private long getShardSize(ShardRouting routing, Map<String, Long> shardSizes) {
        Long shardSize = shardSizes.get(InternalClusterInfoService.shardIdentifierFromRouting(routing));
        return shardSize == null ? 0L : shardSize;
    }

    @Override
    public Decision canAllocate(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        if (!this.enabled) {
            return allocation.decision(Decision.YES, NAME, "disk threshold decider disabled", new Object[0]);
        }
        if (allocation.nodes().size() <= 1) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Only a single node is present, allowing allocation", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "only a single node is present", new Object[0]);
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        if (clusterInfo == null) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Cluster info unavailable for disk threshold decider, allowing allocation.", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "cluster info unavailable", new Object[0]);
        }
        Map<String, DiskUsage> usages = clusterInfo.getNodeDiskUsages();
        Map<String, Long> shardSizes = clusterInfo.getShardSizes();
        if (usages.isEmpty()) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Unable to determine disk usages for disk-aware allocation, allowing allocation", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "disk usages unavailable", new Object[0]);
        }
        DiskUsage usage = usages.get(node.nodeId());
        if (usage == null) {
            usage = this.averageUsage(node, usages);
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Unable to determine disk usage for [{}], defaulting to average across nodes [{} total] [{} free] [{}% free]", node.nodeId(), usage.getTotalBytes(), usage.getFreeBytes(), usage.getFreeDiskAsPercentage());
            }
        }
        if (this.includeRelocations) {
            long relocatingShardsSize = this.sizeOfRelocatingShards(node, shardSizes, false);
            DiskUsage usageIncludingRelocations = new DiskUsage(node.nodeId(), node.node().name(), usage.getTotalBytes(), usage.getFreeBytes() - relocatingShardsSize);
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("usage without relocations: {}", usage);
                this.logger.trace("usage with relocations: [{} bytes] {}", relocatingShardsSize, usageIncludingRelocations);
            }
            usage = usageIncludingRelocations;
        }
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Node [{}] has {}% free disk", node.nodeId(), freeDiskPercentage);
        }
        boolean primaryHasBeenAllocated = allocation.routingTable().index(shardRouting.index()).shard(shardRouting.id()).primaryAllocatedPostApi();
        if (freeBytes < this.freeBytesThresholdLow.bytes()) {
            if (!shardRouting.primary() || shardRouting.primary() && primaryHasBeenAllocated) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("Less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation", this.freeBytesThresholdLow, freeBytes, node.nodeId());
                }
                return allocation.decision(Decision.NO, NAME, "less than required [%s] free on node, free: [%s]", this.freeBytesThresholdLow, new ByteSizeValue(freeBytes));
            }
            if (freeBytes > this.freeBytesThresholdHigh.bytes()) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("Less than the required {} free bytes threshold ({} bytes free) on node {}, but allowing allocation because primary has never been allocated", this.freeBytesThresholdLow, freeBytes, node.nodeId());
                }
                return allocation.decision(Decision.YES, NAME, "primary has never been allocated before", new Object[0]);
            }
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation even though primary has never been allocated", this.freeBytesThresholdHigh, freeBytes, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "less than required [%s] free on node, free: [%s]", this.freeBytesThresholdHigh, new ByteSizeValue(freeBytes));
        }
        if (freeDiskPercentage < this.freeDiskThresholdLow) {
            if (!shardRouting.primary() || shardRouting.primary() && primaryHasBeenAllocated) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("Less than the required {} free disk threshold ({} free) on node [{}], preventing allocation", Strings.format1Decimals(this.freeDiskThresholdLow, "%"), Strings.format1Decimals(freeDiskPercentage, "%"), node.nodeId());
                }
                return allocation.decision(Decision.NO, NAME, "less than required [%s%%] free disk on node, free: [%s%%]", this.freeDiskThresholdLow, freeDiskPercentage);
            }
            if (freeDiskPercentage > this.freeDiskThresholdHigh) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("Less than the required {} free disk threshold ({} free) on node [{}], but allowing allocation because primary has never been allocated", Strings.format1Decimals(this.freeDiskThresholdLow, "%"), Strings.format1Decimals(freeDiskPercentage, "%"), node.nodeId());
                }
                return allocation.decision(Decision.YES, NAME, "primary has never been allocated before", new Object[0]);
            }
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation even though primary has never been allocated", Strings.format1Decimals(this.freeDiskThresholdHigh, "%"), Strings.format1Decimals(freeDiskPercentage, "%"), node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "less than required [%s%%] free disk on node, free: [%s%%]", this.freeDiskThresholdLow, freeDiskPercentage);
        }
        Long shardSize = shardSizes.get(InternalClusterInfoService.shardIdentifierFromRouting(shardRouting));
        shardSize = shardSize == null ? 0L : shardSize;
        double freeSpaceAfterShard = this.freeDiskPercentageAfterShardAssigned(usage, shardSize);
        long freeBytesAfterShard = freeBytes - shardSize;
        if (freeBytesAfterShard < this.freeBytesThresholdHigh.bytes()) {
            this.logger.warn("After allocating, node [{}] would have less than the required {} free bytes threshold ({} bytes free), preventing allocation", node.nodeId(), this.freeBytesThresholdHigh, freeBytesAfterShard);
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s] free on node, free: [%s]", this.freeBytesThresholdLow, new ByteSizeValue(freeBytesAfterShard));
        }
        if (freeSpaceAfterShard < this.freeDiskThresholdHigh) {
            this.logger.warn("After allocating, node [{}] would have less than the required {} free disk threshold ({} free), preventing allocation", node.nodeId(), Strings.format1Decimals(this.freeDiskThresholdHigh, "%"), Strings.format1Decimals(freeSpaceAfterShard, "%"));
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s%%] free disk on node, free: [%s%%]", this.freeDiskThresholdLow, freeSpaceAfterShard);
        }
        return allocation.decision(Decision.YES, NAME, "enough disk for shard on node, free: [%s]", new ByteSizeValue(freeBytes));
    }

    @Override
    public Decision canRemain(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        if (!this.enabled) {
            return allocation.decision(Decision.YES, NAME, "disk threshold decider disabled", new Object[0]);
        }
        if (allocation.nodes().size() <= 1) {
            return allocation.decision(Decision.YES, NAME, "only a single node is present", new Object[0]);
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        if (clusterInfo == null) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Cluster info unavailable for disk threshold decider, allowing allocation.", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "cluster info unavailable", new Object[0]);
        }
        Map<String, DiskUsage> usages = clusterInfo.getNodeDiskUsages();
        if (usages.isEmpty()) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Unable to determine disk usages for disk-aware allocation, allowing allocation", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "disk usages unavailable", new Object[0]);
        }
        DiskUsage usage = usages.get(node.nodeId());
        if (usage == null) {
            usage = this.averageUsage(node, usages);
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Unable to determine disk usage for {}, defaulting to average across nodes [{} total] [{} free] [{}% free]", node.nodeId(), usage.getTotalBytes(), usage.getFreeBytes(), usage.getFreeDiskAsPercentage());
            }
        }
        if (this.includeRelocations) {
            Map<String, Long> shardSizes = clusterInfo.getShardSizes();
            long relocatingShardsSize = this.sizeOfRelocatingShards(node, shardSizes, true);
            DiskUsage usageIncludingRelocations = new DiskUsage(node.nodeId(), node.node().name(), usage.getTotalBytes(), usage.getFreeBytes() - relocatingShardsSize);
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("usage without relocations: {}", usage);
                this.logger.trace("usage with relocations: [{} bytes] {}", relocatingShardsSize, usageIncludingRelocations);
            }
            usage = usageIncludingRelocations;
        }
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("Node [{}] has {}% free disk ({} bytes)", node.nodeId(), freeDiskPercentage, freeBytes);
        }
        if (freeBytes < this.freeBytesThresholdHigh.bytes()) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Less than the required {} free bytes threshold ({} bytes free) on node {}, shard cannot remain", this.freeBytesThresholdHigh, freeBytes, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s] free on node, free: [%s]", this.freeBytesThresholdHigh, new ByteSizeValue(freeBytes));
        }
        if (freeDiskPercentage < this.freeDiskThresholdHigh) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Less than the required {}% free disk threshold ({}% free) on node {}, shard cannot remain", this.freeDiskThresholdHigh, freeDiskPercentage, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s%%] free disk on node, free: [%s%%]", this.freeDiskThresholdHigh, freeDiskPercentage);
        }
        return allocation.decision(Decision.YES, NAME, "enough disk for shard to remain on node, free: [%s]", new ByteSizeValue(freeBytes));
    }

    public DiskUsage averageUsage(RoutingNode node, Map<String, DiskUsage> usages) {
        if (usages.size() == 0) {
            return new DiskUsage(node.nodeId(), node.node().name(), 0L, 0L);
        }
        long totalBytes = 0L;
        long freeBytes = 0L;
        for (DiskUsage du : usages.values()) {
            totalBytes += du.getTotalBytes();
            freeBytes += du.getFreeBytes();
        }
        return new DiskUsage(node.nodeId(), node.node().name(), totalBytes / (long)usages.size(), freeBytes / (long)usages.size());
    }

    public double freeDiskPercentageAfterShardAssigned(DiskUsage usage, Long shardSize) {
        shardSize = shardSize == null ? 0L : shardSize;
        DiskUsage newUsage = new DiskUsage(usage.getNodeId(), usage.getNodeName(), usage.getTotalBytes(), usage.getFreeBytes() - shardSize);
        return newUsage.getFreeDiskAsPercentage();
    }

    public double thresholdPercentageFromWatermark(String watermark) {
        try {
            return RatioValue.parseRatioValue(watermark).getAsPercent();
        }
        catch (ElasticsearchParseException ex) {
            return 100.0;
        }
    }

    public ByteSizeValue thresholdBytesFromWatermark(String watermark) {
        try {
            return ByteSizeValue.parseBytesSizeValue(watermark);
        }
        catch (ElasticsearchParseException ex) {
            return ByteSizeValue.parseBytesSizeValue("0b");
        }
    }

    public boolean validWatermarkSetting(String watermark) {
        try {
            RatioValue.parseRatioValue(watermark);
            return true;
        }
        catch (ElasticsearchParseException e) {
            try {
                ByteSizeValue.parseBytesSizeValue(watermark);
                return true;
            }
            catch (ElasticsearchParseException ex) {
                return false;
            }
        }
    }

    class DiskListener
    implements ClusterInfoService.Listener {
        private final Client client;
        private long lastRun;

        DiskListener(Client client) {
            this.client = client;
        }

        private void warnAboutDiskIfNeeded(DiskUsage usage) {
            if (usage.getFreeBytes() < DiskThresholdDecider.this.freeBytesThresholdHigh.bytes()) {
                DiskThresholdDecider.this.logger.warn("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node", DiskThresholdDecider.this.freeBytesThresholdHigh, usage);
            } else if (usage.getFreeBytes() < DiskThresholdDecider.this.freeBytesThresholdLow.bytes()) {
                DiskThresholdDecider.this.logger.info("low disk watermark [{}] exceeded on {}, replicas will not be assigned to this node", DiskThresholdDecider.this.freeBytesThresholdLow, usage);
            }
            if (usage.getFreeDiskAsPercentage() < DiskThresholdDecider.this.freeDiskThresholdHigh) {
                DiskThresholdDecider.this.logger.warn("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node", Strings.format1Decimals(DiskThresholdDecider.this.freeDiskThresholdHigh, "%"), usage);
            } else if (usage.getFreeDiskAsPercentage() < DiskThresholdDecider.this.freeDiskThresholdLow) {
                DiskThresholdDecider.this.logger.info("low disk watermark [{}] exceeded on {}, replicas will not be assigned to this node", Strings.format1Decimals(DiskThresholdDecider.this.freeDiskThresholdLow, "%"), usage);
            }
        }

        @Override
        public void onNewInfo(ClusterInfo info) {
            Map<String, DiskUsage> usages = info.getNodeDiskUsages();
            if (usages != null) {
                boolean reroute = false;
                for (DiskUsage entry : usages.values()) {
                    this.warnAboutDiskIfNeeded(entry);
                    if (entry.getFreeBytes() >= DiskThresholdDecider.this.freeBytesThresholdHigh.bytes() && !(entry.getFreeDiskAsPercentage() < DiskThresholdDecider.this.freeDiskThresholdHigh)) continue;
                    if (System.currentTimeMillis() - this.lastRun > DiskThresholdDecider.this.rerouteInterval.millis()) {
                        this.lastRun = System.currentTimeMillis();
                        reroute = true;
                        continue;
                    }
                    DiskThresholdDecider.this.logger.debug("high disk watermark exceeded on {} but an automatic reroute has occurred in the last [{}], skipping reroute", entry, DiskThresholdDecider.this.rerouteInterval);
                }
                if (reroute) {
                    DiskThresholdDecider.this.logger.info("high disk watermark exceeded on one or more nodes, rerouting shards", new Object[0]);
                    this.client.admin().cluster().prepareReroute().execute();
                }
            }
        }
    }

    class ApplySettings
    implements NodeSettingsService.Listener {
        ApplySettings() {
        }

        @Override
        public void onRefreshSettings(Settings settings) {
            String newLowWatermark = settings.get(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK, null);
            String newHighWatermark = settings.get(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK, null);
            Boolean newRelocationsSetting = settings.getAsBoolean(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_INCLUDE_RELOCATIONS, null);
            Boolean newEnableSetting = settings.getAsBoolean(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED, null);
            TimeValue newRerouteInterval = settings.getAsTime(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_REROUTE_INTERVAL, null);
            if (newEnableSetting != null) {
                DiskThresholdDecider.this.logger.info("updating [{}] from [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED, DiskThresholdDecider.this.enabled, newEnableSetting);
                DiskThresholdDecider.this.enabled = newEnableSetting;
            }
            if (newRelocationsSetting != null) {
                DiskThresholdDecider.this.logger.info("updating [{}] from [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_INCLUDE_RELOCATIONS, DiskThresholdDecider.this.includeRelocations, newRelocationsSetting);
                DiskThresholdDecider.this.includeRelocations = newRelocationsSetting;
            }
            if (newLowWatermark != null) {
                if (!DiskThresholdDecider.this.validWatermarkSetting(newLowWatermark)) {
                    throw new ElasticsearchParseException("Unable to parse low watermark: [" + newLowWatermark + "]");
                }
                DiskThresholdDecider.this.logger.info("updating [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK, newLowWatermark);
                DiskThresholdDecider.this.freeDiskThresholdLow = 100.0 - DiskThresholdDecider.this.thresholdPercentageFromWatermark(newLowWatermark);
                DiskThresholdDecider.this.freeBytesThresholdLow = DiskThresholdDecider.this.thresholdBytesFromWatermark(newLowWatermark);
            }
            if (newHighWatermark != null) {
                if (!DiskThresholdDecider.this.validWatermarkSetting(newHighWatermark)) {
                    throw new ElasticsearchParseException("Unable to parse high watermark: [" + newHighWatermark + "]");
                }
                DiskThresholdDecider.this.logger.info("updating [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK, newHighWatermark);
                DiskThresholdDecider.this.freeDiskThresholdHigh = 100.0 - DiskThresholdDecider.this.thresholdPercentageFromWatermark(newHighWatermark);
                DiskThresholdDecider.this.freeBytesThresholdHigh = DiskThresholdDecider.this.thresholdBytesFromWatermark(newHighWatermark);
            }
            if (newRerouteInterval != null) {
                DiskThresholdDecider.this.logger.info("updating [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_REROUTE_INTERVAL, newRerouteInterval);
                DiskThresholdDecider.this.rerouteInterval = newRerouteInterval;
            }
        }
    }
}

