/*
 * Decompiled with CFR 0.152.
 */
package org.apache.oozie.util;

import java.util.AbstractQueue;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;

public class PriorityDelayQueue<E>
extends AbstractQueue<QueueElement<E>>
implements BlockingQueue<QueueElement<E>> {
    public static final long ANTI_STARVATION_INTERVAL = 500L;
    protected int priorities;
    protected DelayQueue<QueueElement<E>>[] queues;
    protected final transient ReentrantLock lock = new ReentrantLock();
    private transient long lastAntiStarvationCheck = 0L;
    private long maxWait;
    private int maxSize;
    protected AtomicInteger currentSize;

    public PriorityDelayQueue(int priorities, long maxWait, TimeUnit unit, int maxSize) {
        if (priorities < 1) {
            throw new IllegalArgumentException("priorities must be 1 or more");
        }
        if (maxWait < 0L) {
            throw new IllegalArgumentException("maxWait must be greater than 0");
        }
        if (maxSize < -1 || maxSize == 0) {
            throw new IllegalArgumentException("maxSize must be -1 or greater than 0");
        }
        this.priorities = priorities;
        this.queues = new DelayQueue[priorities];
        for (int i = 0; i < priorities; ++i) {
            this.queues[i] = new DelayQueue();
        }
        this.maxWait = unit.toMillis(maxWait);
        this.maxSize = maxSize;
        if (maxSize != -1) {
            this.currentSize = new AtomicInteger();
        }
    }

    public int getPriorities() {
        return this.priorities;
    }

    public long getMaxWait(TimeUnit unit) {
        return unit.convert(this.maxWait, TimeUnit.MILLISECONDS);
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Iterator<QueueElement<E>> iterator() {
        QueueElement[][] queueElements = new QueueElement[this.queues.length][];
        this.lock.lock();
        try {
            for (int i = 0; i < this.queues.length; ++i) {
                queueElements[i] = this.queues[i].toArray(new QueueElement[0]);
            }
        }
        finally {
            this.lock.unlock();
        }
        ArrayList<QueueElement> list = new ArrayList<QueueElement>();
        for (QueueElement[] elements : queueElements) {
            list.addAll(Arrays.asList(elements));
        }
        return list.iterator();
    }

    @Override
    public int size() {
        int size = 0;
        for (DelayQueue<QueueElement<E>> queue : this.queues) {
            size += queue.size();
        }
        return size;
    }

    public int[] sizes() {
        int[] sizes = new int[this.queues.length];
        for (int i = 0; i < this.queues.length; ++i) {
            sizes[i] = this.queues[i].size();
        }
        return sizes;
    }

    @Override
    public boolean add(QueueElement<E> queueElement) {
        return this.offer(queueElement, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean offer(QueueElement<E> queueElement, boolean ignoreSize) {
        boolean accepted;
        if (queueElement == null) {
            throw new NullPointerException("queueElement is NULL");
        }
        if (queueElement.getPriority() < 0 || queueElement.getPriority() >= this.priorities) {
            throw new IllegalArgumentException("priority out of range: " + queueElement);
        }
        if (queueElement.inQueue) {
            throw new IllegalStateException("queueElement already in a queue: " + queueElement);
        }
        if (!ignoreSize && this.currentSize != null && this.currentSize.get() >= this.maxSize) {
            return false;
        }
        this.lock.lock();
        try {
            accepted = this.queues[queueElement.getPriority()].offer(queueElement);
            this.debug("offer([{0}]), to P[{1}] delay[{2}ms] accepted[{3}]", queueElement.getElement().toString(), queueElement.getPriority(), queueElement.getDelay(TimeUnit.MILLISECONDS), accepted);
            if (accepted) {
                if (this.currentSize != null) {
                    this.currentSize.incrementAndGet();
                }
                queueElement.inQueue = true;
            }
        }
        finally {
            this.lock.unlock();
        }
        return accepted;
    }

    @Override
    public boolean offer(QueueElement<E> queueElement) {
        return this.offer(queueElement, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public QueueElement<E> poll() {
        this.lock.lock();
        try {
            int i;
            this.antiStarvation();
            QueueElement e = null;
            for (i = this.priorities; e == null && i > 0; --i) {
                e = (QueueElement)this.queues[i - 1].poll();
            }
            if (e != null) {
                if (this.currentSize != null) {
                    this.currentSize.decrementAndGet();
                }
                e.inQueue = false;
                this.debug("poll(): [{0}], from P[{1}]", e.getElement().toString(), i);
            }
            QueueElement queueElement = e;
            return queueElement;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public QueueElement<E> peek() {
        this.lock.lock();
        try {
            int i;
            this.antiStarvation();
            QueueElement e = null;
            QueueElement[] seeks = new QueueElement[this.priorities];
            boolean foundElement = false;
            for (i = this.priorities - 1; i > -1; --i) {
                e = (QueueElement)this.queues[i].peek();
                this.debug("peek(): considering [{0}] from P[{1}]", e, i);
                seeks[this.priorities - i - 1] = e;
                foundElement |= e != null;
            }
            if (foundElement) {
                e = null;
                for (i = 0; e == null && i < this.priorities; ++i) {
                    if (seeks[i] != null && seeks[i].getDelay(TimeUnit.MILLISECONDS) > 0L) {
                        this.debug("peek, ignoring [{0}]", seeks[i]);
                        continue;
                    }
                    e = seeks[i];
                }
                if (e != null) {
                    this.debug("peek(): choosing [{0}]", e);
                }
                if (e == null) {
                    for (int first = 0; e == null && first < this.priorities; ++first) {
                        e = seeks[first];
                    }
                    if (e != null) {
                        this.debug("peek(): initial choosing [{0}]", e);
                    }
                    for (int i2 = first; i2 < this.priorities; ++i2) {
                        QueueElement ee = seeks[i2];
                        if (ee == null || ee.getDelay(TimeUnit.MILLISECONDS) >= e.getDelay(TimeUnit.MILLISECONDS)) continue;
                        this.debug("peek(): choosing [{0}] over [{1}]", ee, e);
                        e = ee;
                    }
                }
            }
            if (e != null) {
                this.debug("peek(): [{0}], from P[{1}]", e.getElement().toString(), e.getPriority());
            } else {
                this.debug("peek(): NULL", new Object[0]);
            }
            QueueElement queueElement = e;
            return queueElement;
        }
        finally {
            this.lock.unlock();
        }
    }

    protected void antiStarvation() {
        long now = System.currentTimeMillis();
        if (now - this.lastAntiStarvationCheck > 500L) {
            for (int i = 0; i < this.queues.length - 1; ++i) {
                this.antiStarvation(this.queues[i], this.queues[i + 1], "from P[" + i + "] to P[" + (i + 1) + "]");
            }
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < this.queues.length; ++i) {
                sb.append("P[").append(i).append("]=").append(this.queues[i].size()).append(" ");
            }
            this.debug("sub-queue sizes: {0}", sb.toString());
            this.lastAntiStarvationCheck = System.currentTimeMillis();
        }
    }

    private void antiStarvation(DelayQueue<QueueElement<E>> lowerQ, DelayQueue<QueueElement<E>> higherQ, String msg) {
        int moved = 0;
        QueueElement e = (QueueElement)lowerQ.poll();
        while (e != null && e.getDelay(TimeUnit.MILLISECONDS) < -this.maxWait) {
            e.setDelay(0L, TimeUnit.MILLISECONDS);
            if (!higherQ.offer(e)) {
                throw new IllegalStateException("Could not move element to higher sub-queue, element rejected");
            }
            e.priority++;
            e = (QueueElement)lowerQ.poll();
            ++moved;
        }
        if (e != null && !lowerQ.offer(e)) {
            throw new IllegalStateException("Could not reinsert element to current sub-queue, element rejected");
        }
        this.debug("anti-starvation, moved {0} element(s) {1}", moved, msg);
    }

    protected void debug(String msgTemplate, Object ... msgArgs) {
    }

    @Override
    public void put(QueueElement<E> e) throws InterruptedException {
        while (!this.offer(e, true)) {
            Thread.sleep(10L);
        }
    }

    @Override
    public boolean offer(QueueElement<E> e, long timeout, TimeUnit unit) throws InterruptedException {
        return this.offer(e, true);
    }

    @Override
    public QueueElement<E> take() throws InterruptedException {
        Object e = this.poll();
        while (e == null) {
            Thread.sleep(10L);
            e = this.poll();
        }
        return e;
    }

    @Override
    public QueueElement<E> poll(long timeout, TimeUnit unit) throws InterruptedException {
        Object e = this.poll();
        long time = System.currentTimeMillis() + unit.toMillis(timeout);
        while (e == null && time > System.currentTimeMillis()) {
            Thread.sleep(10L);
            e = this.poll();
        }
        return this.poll();
    }

    @Override
    public int remainingCapacity() {
        return this.maxSize == -1 ? -1 : this.maxSize - this.size();
    }

    @Override
    public int drainTo(Collection<? super QueueElement<E>> c) {
        int count = 0;
        for (DelayQueue<QueueElement<? super QueueElement<E>>> delayQueue : this.queues) {
            count += delayQueue.drainTo(c);
        }
        return count;
    }

    @Override
    public int drainTo(Collection<? super QueueElement<E>> c, int maxElements) {
        int left = maxElements;
        int count = 0;
        for (DelayQueue<QueueElement<? super QueueElement<E>>> delayQueue : this.queues) {
            int drained = delayQueue.drainTo(c, left);
            count += drained;
            left -= drained;
        }
        return count;
    }

    @Override
    public void clear() {
        for (DelayQueue<QueueElement<E>> q : this.queues) {
            q.clear();
        }
    }

    public static class QueueElement<E>
    implements Delayed {
        private E element;
        private int priority;
        private long baseTime;
        boolean inQueue;

        public QueueElement(E element, int priority, long delay, TimeUnit unit) {
            if (element == null) {
                throw new IllegalArgumentException("element cannot be null");
            }
            if (priority < 0) {
                throw new IllegalArgumentException("priority cannot be negative, [" + element + "]");
            }
            if (delay < 0L) {
                throw new IllegalArgumentException("delay cannot be negative");
            }
            this.element = element;
            this.priority = priority;
            this.setDelay(delay, unit);
        }

        public QueueElement(E element) {
            this(element, 0, 0L, TimeUnit.MILLISECONDS);
        }

        public E getElement() {
            return this.element;
        }

        public int getPriority() {
            return this.priority;
        }

        public void setDelay(long delay, TimeUnit unit) {
            this.baseTime = System.currentTimeMillis() + unit.toMillis(delay);
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert(this.baseTime - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            long diff = this.getDelay(TimeUnit.MILLISECONDS) - o.getDelay(TimeUnit.MILLISECONDS);
            if (diff > 0L) {
                return 1;
            }
            if (diff < 0L) {
                return -1;
            }
            return 0;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("[").append(this.element).append("] priority=").append(this.priority).append(" delay=").append(this.getDelay(TimeUnit.MILLISECONDS));
            return sb.toString();
        }
    }
}

