/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import com.google.common.base.Preconditions;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.hbase.io.DoubleOutputStream;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.BoundedRangeFileInputStream;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializer;
import org.apache.hadoop.hbase.io.hfile.Compression;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.hbase.util.Writables;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.compress.Compressor;
import org.apache.hadoop.io.compress.Decompressor;

public class HFileBlock
implements Cacheable {
    public static final int HEADER_SIZE = 24;
    public static final byte[] DUMMY_HEADER = new byte[24];
    public static final int BYTE_BUFFER_HEAP_SIZE = (int)ClassSize.estimateBase(ByteBuffer.wrap(new byte[0], 0, 0).getClass(), false);
    static final int EXTRA_SERIALIZATION_SPACE = 12;
    private static final CacheableDeserializer<Cacheable> blockDeserializer = new CacheableDeserializer<Cacheable>(){

        @Override
        public HFileBlock deserialize(ByteBuffer buf) throws IOException {
            ByteBuffer newByteBuffer = ByteBuffer.allocate(buf.limit() - 12);
            buf.limit(buf.limit() - 12).rewind();
            newByteBuffer.put(buf);
            HFileBlock ourBuffer = new HFileBlock(newByteBuffer);
            buf.position(buf.limit());
            buf.limit(buf.limit() + 12);
            ourBuffer.offset = buf.getLong();
            ourBuffer.nextBlockOnDiskSizeWithHeader = buf.getInt();
            return ourBuffer;
        }
    };
    private BlockType blockType;
    private final int onDiskSizeWithoutHeader;
    private final int uncompressedSizeWithoutHeader;
    private final long prevBlockOffset;
    private ByteBuffer buf;
    private long offset = -1L;
    private int nextBlockOnDiskSizeWithHeader = -1;

    public HFileBlock(BlockType blockType, int onDiskSizeWithoutHeader, int uncompressedSizeWithoutHeader, long prevBlockOffset, ByteBuffer buf, boolean fillHeader, long offset) {
        this.blockType = blockType;
        this.onDiskSizeWithoutHeader = onDiskSizeWithoutHeader;
        this.uncompressedSizeWithoutHeader = uncompressedSizeWithoutHeader;
        this.prevBlockOffset = prevBlockOffset;
        this.buf = buf;
        if (fillHeader) {
            this.overwriteHeader();
        }
        this.offset = offset;
    }

    private HFileBlock(ByteBuffer b) throws IOException {
        b.rewind();
        this.blockType = BlockType.read(b);
        this.onDiskSizeWithoutHeader = b.getInt();
        this.uncompressedSizeWithoutHeader = b.getInt();
        this.prevBlockOffset = b.getLong();
        this.buf = b;
        this.buf.rewind();
    }

    public BlockType getBlockType() {
        return this.blockType;
    }

    public int getOnDiskSizeWithHeader() {
        return this.onDiskSizeWithoutHeader + 24;
    }

    public int getOnDiskSizeWithoutHeader() {
        return this.onDiskSizeWithoutHeader;
    }

    public int getUncompressedSizeWithoutHeader() {
        return this.uncompressedSizeWithoutHeader;
    }

    public long getPrevBlockOffset() {
        return this.prevBlockOffset;
    }

    private void overwriteHeader() {
        this.buf.rewind();
        this.blockType.write(this.buf);
        this.buf.putInt(this.onDiskSizeWithoutHeader);
        this.buf.putInt(this.uncompressedSizeWithoutHeader);
        this.buf.putLong(this.prevBlockOffset);
    }

    public ByteBuffer getBufferWithoutHeader() {
        return ByteBuffer.wrap(this.buf.array(), this.buf.arrayOffset() + 24, this.buf.limit() - 24).slice();
    }

    public ByteBuffer getBufferReadOnly() {
        return this.buf;
    }

    public ByteBuffer getBufferWithHeader() {
        ByteBuffer dupBuf = this.buf.duplicate();
        dupBuf.rewind();
        return dupBuf;
    }

    public void readInto(Writable w) throws IOException {
        Preconditions.checkNotNull((Object)w);
        if (Writables.getWritable(this.buf.array(), this.buf.arrayOffset() + 24, this.buf.limit() - 24, w) == null) {
            throw new IOException("Failed to deserialize block " + this + " into a " + w.getClass().getSimpleName());
        }
    }

    private void sanityCheckAssertion(long valueFromBuf, long valueFromField, String fieldName) throws IOException {
        if (valueFromBuf != valueFromField) {
            throw new AssertionError((Object)(fieldName + " in the buffer (" + valueFromBuf + ") is different from that in the field (" + valueFromField + ")"));
        }
    }

    void sanityCheck() throws IOException {
        this.buf.rewind();
        BlockType blockTypeFromBuf = BlockType.read(this.buf);
        if (blockTypeFromBuf != this.blockType) {
            throw new IOException("Block type stored in the buffer: " + (Object)((Object)blockTypeFromBuf) + ", block type field: " + (Object)((Object)this.blockType));
        }
        this.sanityCheckAssertion(this.buf.getInt(), this.onDiskSizeWithoutHeader, "onDiskSizeWithoutHeader");
        this.sanityCheckAssertion(this.buf.getInt(), this.uncompressedSizeWithoutHeader, "uncompressedSizeWithoutHeader");
        this.sanityCheckAssertion(this.buf.getLong(), this.prevBlockOffset, "prevBlocKOffset");
        int expectedBufLimit = this.uncompressedSizeWithoutHeader + 24;
        if (this.buf.limit() != expectedBufLimit) {
            throw new AssertionError((Object)("Expected buffer limit " + expectedBufLimit + ", got " + this.buf.limit()));
        }
        if (this.buf.capacity() != this.uncompressedSizeWithoutHeader + 24 && this.buf.capacity() != this.uncompressedSizeWithoutHeader + 48) {
            throw new AssertionError((Object)("Invalid buffer capacity: " + this.buf.capacity() + ", expected " + (this.uncompressedSizeWithoutHeader + 24) + " or " + (this.uncompressedSizeWithoutHeader + 48)));
        }
    }

    public String toString() {
        return "blockType=" + (Object)((Object)this.blockType) + ", onDiskSizeWithoutHeader=" + this.onDiskSizeWithoutHeader + ", uncompressedSizeWithoutHeader=" + this.uncompressedSizeWithoutHeader + ", prevBlockOffset=" + this.prevBlockOffset + ", dataBeginsWith=" + Bytes.toStringBinary(this.buf.array(), this.buf.arrayOffset() + 24, Math.min(32, this.buf.limit() - this.buf.arrayOffset() - 24)) + ", fileOffset=" + this.offset;
    }

    private void validateOnDiskSizeWithoutHeader(int expectedOnDiskSizeWithoutHeader) throws IOException {
        if (this.onDiskSizeWithoutHeader != expectedOnDiskSizeWithoutHeader) {
            String blockInfoMsg = "Block offset: " + this.offset + ", data starts with: " + Bytes.toStringBinary(this.buf.array(), this.buf.arrayOffset(), this.buf.arrayOffset() + Math.min(32, this.buf.limit()));
            throw new IOException("On-disk size without header provided is " + expectedOnDiskSizeWithoutHeader + ", but block " + "header contains " + this.onDiskSizeWithoutHeader + ". " + blockInfoMsg);
        }
    }

    private void allocateBuffer(boolean extraBytes) {
        int capacityNeeded = 24 + this.uncompressedSizeWithoutHeader + (extraBytes ? 24 : 0);
        ByteBuffer newBuf = ByteBuffer.allocate(capacityNeeded);
        System.arraycopy(this.buf.array(), this.buf.arrayOffset(), newBuf.array(), newBuf.arrayOffset(), 24);
        this.buf = newBuf;
        this.buf.limit(24 + this.uncompressedSizeWithoutHeader);
    }

    public void assumeUncompressed() throws IOException {
        if (this.onDiskSizeWithoutHeader != this.uncompressedSizeWithoutHeader) {
            throw new IOException("Using no compression but onDiskSizeWithoutHeader=" + this.onDiskSizeWithoutHeader + ", " + "uncompressedSizeWithoutHeader=" + this.uncompressedSizeWithoutHeader);
        }
    }

    public void expectType(BlockType expectedType) throws IOException {
        if (this.blockType != expectedType) {
            throw new IOException("Invalid block type: expected=" + (Object)((Object)expectedType) + ", actual=" + (Object)((Object)this.blockType));
        }
    }

    public long getOffset() {
        if (this.offset < 0L) {
            throw new IllegalStateException("HFile block offset not initialized properly");
        }
        return this.offset;
    }

    public DataInputStream getByteStream() {
        return new DataInputStream(new ByteArrayInputStream(this.buf.array(), this.buf.arrayOffset() + 24, this.buf.limit() - 24));
    }

    @Override
    public long heapSize() {
        if (this.buf != null) {
            return ClassSize.align(ClassSize.OBJECT + 2 * ClassSize.REFERENCE + 12 + 16 + BYTE_BUFFER_HEAP_SIZE) + ClassSize.align(this.buf.capacity());
        }
        return ClassSize.align(ClassSize.OBJECT + 2 * ClassSize.REFERENCE + 12 + 16 + BYTE_BUFFER_HEAP_SIZE);
    }

    public static boolean readWithExtra(InputStream in, byte[] buf, int bufOffset, int necessaryLen, int extraLen) throws IOException {
        int bytesRemaining;
        int ret;
        for (bytesRemaining = necessaryLen + extraLen; bytesRemaining > 0 && ((ret = in.read(buf, bufOffset, bytesRemaining)) != -1 || bytesRemaining > extraLen); bytesRemaining -= ret) {
            if (ret < 0) {
                throw new IOException("Premature EOF from inputStream (read returned " + ret + ", was trying to read " + necessaryLen + " necessary bytes and " + extraLen + " extra bytes, " + "successfully read " + (necessaryLen + extraLen - bytesRemaining));
            }
            bufOffset += ret;
        }
        return bytesRemaining <= 0;
    }

    public int getNextBlockOnDiskSizeWithHeader() {
        return this.nextBlockOnDiskSizeWithHeader;
    }

    @Override
    public int getSerializedLength() {
        if (this.buf != null) {
            return this.buf.limit() + 12;
        }
        return 0;
    }

    @Override
    public void serialize(ByteBuffer destination) {
        destination.put(this.buf.duplicate());
        destination.putLong(this.offset);
        destination.putInt(this.nextBlockOnDiskSizeWithHeader);
        destination.rewind();
    }

    @Override
    public CacheableDeserializer<Cacheable> getDeserializer() {
        return blockDeserializer;
    }

    public boolean equals(Object comparison) {
        if (this == comparison) {
            return true;
        }
        if (comparison == null) {
            return false;
        }
        if (comparison.getClass() != this.getClass()) {
            return false;
        }
        HFileBlock castedComparison = (HFileBlock)comparison;
        if (castedComparison.blockType != this.blockType) {
            return false;
        }
        if (castedComparison.nextBlockOnDiskSizeWithHeader != this.nextBlockOnDiskSizeWithHeader) {
            return false;
        }
        if (castedComparison.offset != this.offset) {
            return false;
        }
        if (castedComparison.onDiskSizeWithoutHeader != this.onDiskSizeWithoutHeader) {
            return false;
        }
        if (castedComparison.prevBlockOffset != this.prevBlockOffset) {
            return false;
        }
        if (castedComparison.uncompressedSizeWithoutHeader != this.uncompressedSizeWithoutHeader) {
            return false;
        }
        if (this.buf.compareTo(castedComparison.buf) != 0) {
            return false;
        }
        if (this.buf.position() != castedComparison.buf.position()) {
            return false;
        }
        return this.buf.limit() == castedComparison.buf.limit();
    }

    public static class FSReaderV2
    extends AbstractFSReader {
        private ThreadLocal<PrefetchedHeader> prefetchedHeaderForThread = new ThreadLocal<PrefetchedHeader>(){

            @Override
            public PrefetchedHeader initialValue() {
                return new PrefetchedHeader();
            }
        };

        public FSReaderV2(FSDataInputStream istream, Compression.Algorithm compressAlgo, long fileSize) {
            super(istream, compressAlgo, fileSize);
        }

        @Override
        public HFileBlock readBlockData(long offset, long onDiskSizeWithHeaderL, int uncompressedSize, boolean pread) throws IOException {
            HFileBlock b;
            if (offset < 0L) {
                throw new IOException("Invalid offset=" + offset + " trying to read " + "block (onDiskSize=" + onDiskSizeWithHeaderL + ", uncompressedSize=" + uncompressedSize + ")");
            }
            if (uncompressedSize != -1) {
                throw new IOException("Version 2 block reader API does not need the uncompressed size parameter");
            }
            if (onDiskSizeWithHeaderL < 24L && onDiskSizeWithHeaderL != -1L || onDiskSizeWithHeaderL >= Integer.MAX_VALUE) {
                throw new IOException("Invalid onDisksize=" + onDiskSizeWithHeaderL + ": expected to be at least " + 24 + " and at most " + Integer.MAX_VALUE + ", or -1 (offset=" + offset + ", uncompressedSize=" + uncompressedSize + ")");
            }
            int onDiskSizeWithHeader = (int)onDiskSizeWithHeaderL;
            if (onDiskSizeWithHeader > 0) {
                int preReadHeaderSize;
                int onDiskSizeWithoutHeader = onDiskSizeWithHeader - 24;
                assert (onDiskSizeWithoutHeader >= 0);
                PrefetchedHeader prefetchedHeader = this.prefetchedHeaderForThread.get();
                byte[] header = prefetchedHeader.offset == offset ? prefetchedHeader.header : null;
                int n = preReadHeaderSize = header == null ? 0 : 24;
                if (this.compressAlgo == Compression.Algorithm.NONE) {
                    ByteBuffer headerAndData = ByteBuffer.allocate(onDiskSizeWithHeader + 24);
                    headerAndData.limit(onDiskSizeWithHeader);
                    if (header != null) {
                        System.arraycopy(header, 0, headerAndData.array(), 0, 24);
                    }
                    int nextBlockOnDiskSizeWithHeader = this.readAtOffset(headerAndData.array(), headerAndData.arrayOffset() + preReadHeaderSize, onDiskSizeWithHeader - preReadHeaderSize, true, offset + (long)preReadHeaderSize, pread);
                    b = new HFileBlock(headerAndData);
                    b.assumeUncompressed();
                    b.validateOnDiskSizeWithoutHeader(onDiskSizeWithoutHeader);
                    b.nextBlockOnDiskSizeWithHeader = nextBlockOnDiskSizeWithHeader;
                    if (b.nextBlockOnDiskSizeWithHeader > 0) {
                        this.setNextBlockHeader(offset, b);
                    }
                } else {
                    byte[] onDiskBlock = new byte[onDiskSizeWithHeader + 24];
                    int nextBlockOnDiskSize = this.readAtOffset(onDiskBlock, preReadHeaderSize, onDiskSizeWithHeader - preReadHeaderSize, true, offset + (long)preReadHeaderSize, pread);
                    if (header == null) {
                        header = onDiskBlock;
                    }
                    try {
                        b = new HFileBlock(ByteBuffer.wrap(header, 0, 24));
                    }
                    catch (IOException ex) {
                        throw new IOException("Failed to read compressed block at " + offset + ", onDiskSizeWithoutHeader=" + onDiskSizeWithHeader + ", preReadHeaderSize=" + preReadHeaderSize + ", header.length=" + header.length + ", header bytes: " + Bytes.toStringBinary(header, 0, 24), ex);
                    }
                    b.validateOnDiskSizeWithoutHeader(onDiskSizeWithoutHeader);
                    b.nextBlockOnDiskSizeWithHeader = nextBlockOnDiskSize;
                    DataInputStream dis = new DataInputStream(new ByteArrayInputStream(onDiskBlock, 24, onDiskSizeWithoutHeader));
                    b.allocateBuffer(b.nextBlockOnDiskSizeWithHeader > 0);
                    this.decompress(b.buf.array(), b.buf.arrayOffset() + 24, dis, onDiskSizeWithoutHeader, b.uncompressedSizeWithoutHeader);
                    if (nextBlockOnDiskSize > 0) {
                        System.arraycopy(onDiskBlock, onDiskSizeWithHeader, b.buf.array(), b.buf.arrayOffset() + 24 + b.uncompressedSizeWithoutHeader, 24);
                        this.setNextBlockHeader(offset, b);
                    }
                }
            } else {
                ByteBuffer headerBuf;
                PrefetchedHeader prefetchedHeader = this.prefetchedHeaderForThread.get();
                ByteBuffer byteBuffer = headerBuf = prefetchedHeader.offset == offset ? prefetchedHeader.buf : null;
                if (headerBuf == null) {
                    headerBuf = ByteBuffer.allocate(24);
                    this.readAtOffset(headerBuf.array(), headerBuf.arrayOffset(), 24, false, offset, pread);
                }
                b = new HFileBlock(headerBuf);
                b.allocateBuffer(true);
                if (this.compressAlgo == Compression.Algorithm.NONE) {
                    b.assumeUncompressed();
                    b.nextBlockOnDiskSizeWithHeader = this.readAtOffset(b.buf.array(), b.buf.arrayOffset() + 24, b.uncompressedSizeWithoutHeader, true, offset + 24L, pread);
                    if (b.nextBlockOnDiskSizeWithHeader > 0) {
                        this.setNextBlockHeader(offset, b);
                    }
                } else {
                    byte[] compressedBytes = new byte[b.getOnDiskSizeWithHeader() + 24];
                    b.nextBlockOnDiskSizeWithHeader = this.readAtOffset(compressedBytes, 24, b.onDiskSizeWithoutHeader, true, offset + 24L, pread);
                    DataInputStream dis = new DataInputStream(new ByteArrayInputStream(compressedBytes, 24, b.onDiskSizeWithoutHeader));
                    this.decompress(b.buf.array(), b.buf.arrayOffset() + 24, dis, b.onDiskSizeWithoutHeader, b.uncompressedSizeWithoutHeader);
                    if (b.nextBlockOnDiskSizeWithHeader > 0) {
                        int nextHeaderOffset = b.buf.arrayOffset() + 24 + b.uncompressedSizeWithoutHeader;
                        System.arraycopy(compressedBytes, compressedBytes.length - 24, b.buf.array(), nextHeaderOffset, 24);
                        this.setNextBlockHeader(offset, b);
                    }
                }
            }
            b.offset = offset;
            return b;
        }

        private void setNextBlockHeader(long offset, HFileBlock b) {
            PrefetchedHeader prefetchedHeader = this.prefetchedHeaderForThread.get();
            prefetchedHeader.offset = offset + (long)b.getOnDiskSizeWithHeader();
            int nextHeaderOffset = b.buf.arrayOffset() + 24 + b.uncompressedSizeWithoutHeader;
            System.arraycopy(b.buf.array(), nextHeaderOffset, prefetchedHeader.header, 0, 24);
        }
    }

    private static class PrefetchedHeader {
        long offset = -1L;
        byte[] header = new byte[24];
        ByteBuffer buf = ByteBuffer.wrap(this.header, 0, 24);

        private PrefetchedHeader() {
        }
    }

    public static class FSReaderV1
    extends AbstractFSReader {
        private static final int HEADER_DELTA = 16;

        public FSReaderV1(FSDataInputStream istream, Compression.Algorithm compressAlgo, long fileSize) {
            super(istream, compressAlgo, fileSize);
        }

        @Override
        public HFileBlock readBlockData(long offset, long onDiskSizeWithMagic, int uncompressedSizeWithMagic, boolean pread) throws IOException {
            int onDiskSizeWithoutHeader;
            if (uncompressedSizeWithMagic <= 0) {
                throw new IOException("Invalid uncompressedSize=" + uncompressedSizeWithMagic + " for a version 1 block");
            }
            if (onDiskSizeWithMagic <= 0L || onDiskSizeWithMagic >= Integer.MAX_VALUE) {
                throw new IOException("Invalid onDiskSize=" + onDiskSizeWithMagic + " (maximum allowed: " + Integer.MAX_VALUE + ")");
            }
            int onDiskSize = (int)onDiskSizeWithMagic;
            if (uncompressedSizeWithMagic < 8) {
                throw new IOException("Uncompressed size for a version 1 block is " + uncompressedSizeWithMagic + " but must be at least " + 8);
            }
            ByteBuffer buf = ByteBuffer.allocate(uncompressedSizeWithMagic + 16);
            if (this.compressAlgo == Compression.Algorithm.NONE) {
                if (onDiskSize != uncompressedSizeWithMagic) {
                    throw new IOException("onDiskSize=" + onDiskSize + " and uncompressedSize=" + uncompressedSizeWithMagic + " must be equal for version 1 with no compression");
                }
                this.readAtOffset(buf.array(), buf.arrayOffset() + 16, onDiskSize, false, offset, pread);
                onDiskSizeWithoutHeader = uncompressedSizeWithMagic - 8;
            } else {
                InputStream bufferedBoundedStream = this.createBufferedBoundedStream(offset, onDiskSize, pread);
                this.decompress(buf.array(), buf.arrayOffset() + 16, bufferedBoundedStream, onDiskSize, uncompressedSizeWithMagic);
                onDiskSizeWithoutHeader = onDiskSize;
            }
            BlockType newBlockType = BlockType.parse(buf.array(), buf.arrayOffset() + 16, 8);
            HFileBlock b = new HFileBlock(newBlockType, onDiskSizeWithoutHeader, uncompressedSizeWithMagic - 8, -1L, buf, true, offset);
            return b;
        }
    }

    public static abstract class AbstractFSReader
    implements FSReader {
        protected FSDataInputStream istream;
        protected Compression.Algorithm compressAlgo;
        protected long fileSize;
        public static final int DEFAULT_BUFFER_SIZE = 0x100000;

        public AbstractFSReader(FSDataInputStream istream, Compression.Algorithm compressAlgo, long fileSize) {
            this.istream = istream;
            this.compressAlgo = compressAlgo;
            this.fileSize = fileSize;
        }

        @Override
        public BlockIterator blockRange(final long startOffset, final long endOffset) {
            return new BlockIterator(){
                private long offset;
                {
                    this.offset = startOffset;
                }

                @Override
                public HFileBlock nextBlock() throws IOException {
                    if (this.offset >= endOffset) {
                        return null;
                    }
                    HFileBlock b = AbstractFSReader.this.readBlockData(this.offset, -1L, -1, false);
                    this.offset += (long)b.getOnDiskSizeWithHeader();
                    return b;
                }

                @Override
                public DataInputStream nextBlockAsStream(BlockType blockType) throws IOException {
                    HFileBlock blk = this.nextBlock();
                    if (blk.getBlockType() != blockType) {
                        throw new IOException("Expected block of type " + (Object)((Object)blockType) + " but found " + (Object)((Object)blk.getBlockType()));
                    }
                    return blk.getByteStream();
                }
            };
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected int readAtOffset(byte[] dest, int destOffset, int size, boolean peekIntoNextBlock, long fileOffset, boolean pread) throws IOException {
            if (peekIntoNextBlock && destOffset + size + 24 > dest.length) {
                throw new IOException("Attempted to read " + size + " bytes and " + 24 + " bytes of next header into a " + dest.length + "-byte array at offset " + destOffset);
            }
            if (pread) {
                int extraSize = peekIntoNextBlock ? 24 : 0;
                int ret = this.istream.read(fileOffset, dest, destOffset, size + extraSize);
                if (ret < size) {
                    throw new IOException("Positional read of " + size + " bytes " + "failed at offset " + fileOffset + " (returned " + ret + ")");
                }
                if (ret == size || ret < size + extraSize) {
                    return -1;
                }
            } else {
                FSDataInputStream fSDataInputStream = this.istream;
                synchronized (fSDataInputStream) {
                    this.istream.seek(fileOffset);
                    long realOffset = this.istream.getPos();
                    if (realOffset != fileOffset) {
                        throw new IOException("Tried to seek to " + fileOffset + " to " + "read " + size + " bytes, but pos=" + realOffset + " after seek");
                    }
                    if (!peekIntoNextBlock) {
                        IOUtils.readFully((InputStream)this.istream, (byte[])dest, (int)destOffset, (int)size);
                        return -1;
                    }
                    if (!HFileBlock.readWithExtra((InputStream)this.istream, dest, destOffset, size, 24)) {
                        return -1;
                    }
                }
            }
            assert (peekIntoNextBlock);
            return Bytes.toInt(dest, destOffset + size + 8) + 24;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void decompress(byte[] dest, int destOffset, InputStream bufferedBoundedStream, int compressedSize, int uncompressedSize) throws IOException {
            Decompressor decompressor = null;
            try {
                decompressor = this.compressAlgo.getDecompressor();
                InputStream is = this.compressAlgo.createDecompressionStream(bufferedBoundedStream, decompressor, 0);
                IOUtils.readFully((InputStream)is, (byte[])dest, (int)destOffset, (int)uncompressedSize);
                is.close();
            }
            finally {
                if (decompressor != null) {
                    this.compressAlgo.returnDecompressor(decompressor);
                }
            }
        }

        protected InputStream createBufferedBoundedStream(long offset, int size, boolean pread) {
            return new BufferedInputStream(new BoundedRangeFileInputStream(this.istream, offset, size, pread), Math.min(0x100000, size));
        }
    }

    public static interface FSReader {
        public HFileBlock readBlockData(long var1, long var3, int var5, boolean var6) throws IOException;

        public BlockIterator blockRange(long var1, long var3);
    }

    public static interface BlockIterator {
        public HFileBlock nextBlock() throws IOException;

        public DataInputStream nextBlockAsStream(BlockType var1) throws IOException;
    }

    public static interface BlockWritable {
        public BlockType getBlockType();

        public void writeToBlock(DataOutput var1) throws IOException;
    }

    public static class Writer {
        private State state = State.INIT;
        private final Compression.Algorithm compressAlgo;
        private ByteArrayOutputStream baosOnDisk;
        private ByteArrayOutputStream baosInMemory;
        private Compressor compressor;
        private BlockType blockType;
        private DataOutputStream userDataStream;
        private byte[] onDiskBytesWithHeader;
        private int uncompressedSizeWithoutHeader;
        private byte[] uncompressedBytesWithHeader;
        private long startOffset;
        private long[] prevOffsetByType;
        private boolean cacheOnWrite;
        private long prevOffset;

        public Writer(Compression.Algorithm compressionAlgorithm) {
            this.compressAlgo = compressionAlgorithm == null ? Compression.Algorithm.NONE : compressionAlgorithm;
            this.baosOnDisk = new ByteArrayOutputStream();
            if (this.compressAlgo != Compression.Algorithm.NONE) {
                this.compressor = compressionAlgorithm.getCompressor();
            }
            this.prevOffsetByType = new long[BlockType.values().length];
            for (int i = 0; i < this.prevOffsetByType.length; ++i) {
                this.prevOffsetByType[i] = -1L;
            }
        }

        public DataOutputStream startWriting(BlockType newBlockType, boolean cacheOnWrite) throws IOException {
            if (this.state == State.BLOCK_READY && this.startOffset != -1L) {
                this.prevOffsetByType[this.blockType.ordinal()] = this.startOffset;
            }
            this.cacheOnWrite = cacheOnWrite;
            this.startOffset = -1L;
            this.blockType = newBlockType;
            this.baosOnDisk.reset();
            this.baosOnDisk.write(DUMMY_HEADER);
            this.state = State.WRITING;
            if (this.compressAlgo == Compression.Algorithm.NONE) {
                this.userDataStream = new DataOutputStream(this.baosOnDisk);
            } else {
                OutputStream compressingOutputStream = this.compressAlgo.createCompressionStream(this.baosOnDisk, this.compressor, 0);
                if (cacheOnWrite) {
                    if (this.baosInMemory == null) {
                        this.baosInMemory = new ByteArrayOutputStream();
                    }
                    this.baosInMemory.reset();
                    this.baosInMemory.write(DUMMY_HEADER);
                    this.userDataStream = new DataOutputStream(new DoubleOutputStream(compressingOutputStream, this.baosInMemory));
                } else {
                    this.userDataStream = new DataOutputStream(compressingOutputStream);
                }
            }
            return this.userDataStream;
        }

        DataOutputStream getUserDataStream() {
            this.expectState(State.WRITING);
            return this.userDataStream;
        }

        private void ensureBlockReady() throws IOException {
            Preconditions.checkState((this.state != State.INIT ? 1 : 0) != 0, (Object)("Unexpected state: " + (Object)((Object)this.state)));
            if (this.state == State.BLOCK_READY) {
                return;
            }
            this.finishBlock();
            this.state = State.BLOCK_READY;
        }

        private void finishBlock() throws IOException {
            this.userDataStream.flush();
            this.uncompressedSizeWithoutHeader = this.userDataStream.size();
            this.onDiskBytesWithHeader = this.baosOnDisk.toByteArray();
            this.prevOffset = this.prevOffsetByType[this.blockType.ordinal()];
            this.putHeader(this.onDiskBytesWithHeader, 0);
            if (this.cacheOnWrite && this.compressAlgo != Compression.Algorithm.NONE) {
                this.uncompressedBytesWithHeader = this.baosInMemory.toByteArray();
                if (this.uncompressedSizeWithoutHeader != this.uncompressedBytesWithHeader.length - 24) {
                    throw new IOException("Uncompressed size mismatch: " + this.uncompressedSizeWithoutHeader + " vs. " + (this.uncompressedBytesWithHeader.length - 24));
                }
                this.putHeader(this.uncompressedBytesWithHeader, 0);
            }
        }

        private void putHeader(byte[] dest, int offset) {
            offset = this.blockType.put(dest, offset);
            offset = Bytes.putInt(dest, offset, this.onDiskBytesWithHeader.length - 24);
            offset = Bytes.putInt(dest, offset, this.uncompressedSizeWithoutHeader);
            Bytes.putLong(dest, offset, this.prevOffset);
        }

        public void writeHeaderAndData(FSDataOutputStream out) throws IOException {
            long offset = out.getPos();
            if (this.startOffset != -1L && offset != this.startOffset) {
                throw new IOException("A " + (Object)((Object)this.blockType) + " block written to a " + "stream twice, first at offset " + this.startOffset + ", then at " + offset);
            }
            this.startOffset = offset;
            this.writeHeaderAndData((DataOutputStream)out);
        }

        private void writeHeaderAndData(DataOutputStream out) throws IOException {
            this.ensureBlockReady();
            out.write(this.onDiskBytesWithHeader);
        }

        public byte[] getHeaderAndData() throws IOException {
            this.ensureBlockReady();
            return this.onDiskBytesWithHeader;
        }

        public void releaseCompressor() {
            if (this.compressor != null) {
                this.compressAlgo.returnCompressor(this.compressor);
                this.compressor = null;
            }
        }

        public int getOnDiskSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBytesWithHeader.length - 24;
        }

        public int getOnDiskSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.onDiskBytesWithHeader.length;
        }

        public int getUncompressedSizeWithoutHeader() {
            this.expectState(State.BLOCK_READY);
            return this.uncompressedSizeWithoutHeader;
        }

        public int getUncompressedSizeWithHeader() {
            this.expectState(State.BLOCK_READY);
            return this.uncompressedSizeWithoutHeader + 24;
        }

        public boolean isWriting() {
            return this.state == State.WRITING;
        }

        public int blockSizeWritten() {
            if (this.state != State.WRITING) {
                return 0;
            }
            return this.userDataStream.size();
        }

        private byte[] getUncompressedDataWithHeader() {
            this.expectState(State.BLOCK_READY);
            if (this.compressAlgo == Compression.Algorithm.NONE) {
                return this.onDiskBytesWithHeader;
            }
            if (!this.cacheOnWrite) {
                throw new IllegalStateException("Cache-on-write is turned off");
            }
            if (this.uncompressedBytesWithHeader == null) {
                throw new NullPointerException();
            }
            return this.uncompressedBytesWithHeader;
        }

        private void expectState(State expectedState) {
            if (this.state != expectedState) {
                throw new IllegalStateException("Expected state: " + (Object)((Object)expectedState) + ", actual state: " + (Object)((Object)this.state));
            }
        }

        public ByteBuffer getUncompressedBufferWithHeader() {
            byte[] b = this.getUncompressedDataWithHeader();
            return ByteBuffer.wrap(b, 0, b.length);
        }

        public void writeBlock(BlockWritable bw, FSDataOutputStream out) throws IOException {
            bw.writeToBlock(this.startWriting(bw.getBlockType(), false));
            this.writeHeaderAndData(out);
        }

        public HFileBlock getBlockForCaching() {
            return new HFileBlock(this.blockType, this.onDiskBytesWithHeader.length - 24, this.uncompressedSizeWithoutHeader, this.prevOffset, this.getUncompressedBufferWithHeader(), false, this.startOffset);
        }

        private static enum State {
            INIT,
            WRITING,
            BLOCK_READY;

        }
    }
}

