/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import java.io.DataInput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileScanner;
import org.apache.hadoop.hbase.regionserver.TimeRangeTracker;
import org.apache.hadoop.hbase.util.BloomFilter;
import org.apache.hadoop.hbase.util.BloomFilterFactory;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.Writables;
import org.apache.hadoop.io.Writable;

public class HFilePrettyPrinter {
    private static final Log LOG = LogFactory.getLog(HFilePrettyPrinter.class);
    private Options options = new Options();
    private boolean verbose;
    private boolean printValue;
    private boolean printKey;
    private boolean shouldPrintMeta;
    private boolean printBlocks;
    private boolean printStats;
    private boolean checkRow;
    private boolean checkFamily;
    private Configuration conf;
    private List<Path> files = new ArrayList<Path>();
    private int count;
    private static final String FOUR_SPACES = "    ";

    public HFilePrettyPrinter() {
        this.options.addOption("v", "verbose", false, "Verbose output; emits file and meta data delimiters");
        this.options.addOption("p", "printkv", false, "Print key/value pairs");
        this.options.addOption("e", "printkey", false, "Print keys");
        this.options.addOption("m", "printmeta", false, "Print meta data of file");
        this.options.addOption("b", "printblocks", false, "Print block index meta data");
        this.options.addOption("k", "checkrow", false, "Enable row order check; looks for out-of-order keys");
        this.options.addOption("a", "checkfamily", false, "Enable family check");
        this.options.addOption("f", "file", true, "File to scan. Pass full-path; e.g. hdfs://a:9000/hbase/.META./12/34");
        this.options.addOption("r", "region", true, "Region to scan. Pass region name; e.g. '.META.,,1'");
        this.options.addOption("s", "stats", false, "Print statistics");
    }

    public boolean parseOptions(String[] args) throws ParseException, IOException {
        if (args.length == 0) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("HFile", this.options, true);
            return false;
        }
        PosixParser parser = new PosixParser();
        CommandLine cmd = parser.parse(this.options, args);
        this.verbose = cmd.hasOption("v");
        this.printValue = cmd.hasOption("p");
        this.printKey = cmd.hasOption("e") || this.printValue;
        this.shouldPrintMeta = cmd.hasOption("m");
        this.printBlocks = cmd.hasOption("b");
        this.printStats = cmd.hasOption("s");
        this.checkRow = cmd.hasOption("k");
        this.checkFamily = cmd.hasOption("a");
        if (cmd.hasOption("f")) {
            this.files.add(new Path(cmd.getOptionValue("f")));
        }
        if (cmd.hasOption("r")) {
            String regionName = cmd.getOptionValue("r");
            byte[] rn = Bytes.toBytes(regionName);
            byte[][] hri = HRegionInfo.parseRegionName(rn);
            Path rootDir = FSUtils.getRootDir(this.conf);
            Path tableDir = new Path(rootDir, Bytes.toString(hri[0]));
            String enc = HRegionInfo.encodeRegionName(rn);
            Path regionDir = new Path(tableDir, enc);
            if (this.verbose) {
                System.out.println("region dir -> " + regionDir);
            }
            List<Path> regionFiles = HFile.getStoreFiles(FileSystem.get((Configuration)this.conf), regionDir);
            if (this.verbose) {
                System.out.println("Number of region files found -> " + regionFiles.size());
            }
            if (this.verbose) {
                int i = 1;
                for (Path p : regionFiles) {
                    if (!this.verbose) continue;
                    System.out.println("Found file[" + i++ + "] -> " + p);
                }
            }
            this.files.addAll(regionFiles);
        }
        return true;
    }

    public int run(String[] args) {
        this.conf = HBaseConfiguration.create();
        this.conf.set("fs.defaultFS", this.conf.get("hbase.rootdir"));
        this.conf.set("fs.default.name", this.conf.get("hbase.rootdir"));
        try {
            if (!this.parseOptions(args)) {
                return 1;
            }
        }
        catch (IOException ex) {
            LOG.error((Object)"Error parsing command-line options", (Throwable)ex);
            return 1;
        }
        catch (ParseException ex) {
            LOG.error((Object)"Error parsing command-line options", (Throwable)ex);
            return 1;
        }
        for (Path fileName : this.files) {
            try {
                this.processFile(fileName);
            }
            catch (IOException ex) {
                LOG.error((Object)("Error reading " + fileName), (Throwable)ex);
            }
        }
        if (this.verbose || this.printKey) {
            System.out.println("Scanned kv count -> " + this.count);
        }
        return 0;
    }

    private void processFile(Path file) throws IOException {
        FileSystem fs;
        if (this.verbose) {
            System.out.println("Scanning -> " + file);
        }
        if (!(fs = file.getFileSystem(this.conf)).exists(file)) {
            System.err.println("ERROR, file doesnt exist: " + file);
        }
        HFile.Reader reader = HFile.createReader(fs, file, new CacheConfig(this.conf));
        Map<byte[], byte[]> fileInfo = reader.loadFileInfo();
        KeyValueStatsCollector fileStats = null;
        if (this.verbose || this.printKey || this.checkRow || this.checkFamily || this.printStats) {
            HFileScanner scanner = reader.getScanner(false, false, false);
            fileStats = new KeyValueStatsCollector();
            if (scanner.seekTo()) {
                this.scanKeysValues(file, fileStats, scanner);
            }
        }
        if (this.shouldPrintMeta) {
            this.printMeta(reader, fileInfo);
        }
        if (this.printBlocks) {
            System.out.println("Block Index:");
            System.out.println(reader.getDataBlockIndexReader());
        }
        if (this.printStats) {
            fileStats.finish();
            System.out.println("Stats:\n" + fileStats);
        }
        reader.close();
    }

    private void scanKeysValues(Path file, KeyValueStatsCollector fileStats, HFileScanner scanner) throws IOException {
        KeyValue pkv = null;
        do {
            KeyValue kv = scanner.getKeyValue();
            if (this.printStats) {
                fileStats.collect(kv);
            }
            if (this.printKey) {
                System.out.print("K: " + kv);
                if (this.printValue) {
                    System.out.print(" V: " + Bytes.toStringBinary(kv.getValue()));
                }
                System.out.println();
            }
            if (this.checkRow && pkv != null && Bytes.compareTo(pkv.getRow(), kv.getRow()) > 0) {
                System.err.println("WARNING, previous row is greater then current row\n\tfilename -> " + file + "\n\tprevious -> " + Bytes.toStringBinary(pkv.getKey()) + "\n\tcurrent  -> " + Bytes.toStringBinary(kv.getKey()));
            }
            if (this.checkFamily) {
                String fam = Bytes.toString(kv.getFamily());
                if (!file.toString().contains(fam)) {
                    System.err.println("WARNING, filename does not match kv family,\n\tfilename -> " + file + "\n\tkeyvalue -> " + Bytes.toStringBinary(kv.getKey()));
                }
                if (pkv != null && !Bytes.equals(pkv.getFamily(), kv.getFamily())) {
                    System.err.println("WARNING, previous kv has different family compared to current key\n\tfilename -> " + file + "\n\tprevious -> " + Bytes.toStringBinary(pkv.getKey()) + "\n\tcurrent  -> " + Bytes.toStringBinary(kv.getKey()));
                }
            }
            pkv = kv;
            ++this.count;
        } while (scanner.next());
    }

    private static String asSeparateLines(String keyValueStr) {
        return keyValueStr.replaceAll(", ([a-zA-Z]+=)", ",\n    $1");
    }

    private void printMeta(HFile.Reader reader, Map<byte[], byte[]> fileInfo) throws IOException {
        System.out.println("Block index size as per heapsize: " + reader.indexSize());
        System.out.println(HFilePrettyPrinter.asSeparateLines(reader.toString()));
        System.out.println("Trailer:\n    " + HFilePrettyPrinter.asSeparateLines(reader.getTrailer().toString()));
        System.out.println("Fileinfo:");
        for (Map.Entry<byte[], byte[]> e : fileInfo.entrySet()) {
            System.out.print(FOUR_SPACES + Bytes.toString(e.getKey()) + " = ");
            if (Bytes.compareTo(e.getKey(), Bytes.toBytes("MAX_SEQ_ID_KEY")) == 0) {
                long seqid = Bytes.toLong(e.getValue());
                System.out.println(seqid);
                continue;
            }
            if (Bytes.compareTo(e.getKey(), Bytes.toBytes("TIMERANGE")) == 0) {
                TimeRangeTracker timeRangeTracker = new TimeRangeTracker();
                Writables.copyWritable(e.getValue(), (Writable)timeRangeTracker);
                System.out.println(timeRangeTracker.getMinimumTimestamp() + "...." + timeRangeTracker.getMaximumTimestamp());
                continue;
            }
            if (Bytes.compareTo(e.getKey(), HFile.FileInfo.AVG_KEY_LEN) == 0 || Bytes.compareTo(e.getKey(), HFile.FileInfo.AVG_VALUE_LEN) == 0) {
                System.out.println(Bytes.toInt(e.getValue()));
                continue;
            }
            System.out.println(Bytes.toStringBinary(e.getValue()));
        }
        System.out.println("Mid-key: " + Bytes.toStringBinary(reader.midkey()));
        DataInput bloomMeta = reader.getBloomFilterMetadata();
        BloomFilter bloomFilter = null;
        if (bloomMeta != null) {
            bloomFilter = BloomFilterFactory.createFromMeta(bloomMeta, reader);
        }
        System.out.println("Bloom filter:");
        if (bloomFilter != null) {
            System.out.println(FOUR_SPACES + bloomFilter.toString().replaceAll("; ", "\n    "));
        } else {
            System.out.println("    Not present");
        }
    }

    private static class KeyValueStatsCollector {
        LongStats keyLen = new LongStats();
        LongStats valLen = new LongStats();
        LongStats rowSizeBytes = new LongStats();
        LongStats rowSizeCols = new LongStats();
        long curRowBytes = 0L;
        long curRowCols = 0L;
        byte[] biggestRow = null;
        private KeyValue prevKV = null;
        private long maxRowBytes = 0L;

        private KeyValueStatsCollector() {
        }

        public void collect(KeyValue kv) {
            this.keyLen.collect(kv.getKeyLength());
            this.valLen.collect(kv.getValueLength());
            if (this.prevKV != null && KeyValue.COMPARATOR.compareRows(this.prevKV, kv) != 0) {
                this.collectRow();
            }
            this.curRowBytes += (long)kv.getLength();
            ++this.curRowCols;
            this.prevKV = kv;
        }

        private void collectRow() {
            this.rowSizeBytes.collect(this.curRowBytes);
            this.rowSizeCols.collect(this.curRowCols);
            if (this.curRowBytes > this.maxRowBytes && this.prevKV != null) {
                this.biggestRow = this.prevKV.getRow();
            }
            this.curRowBytes = 0L;
            this.curRowCols = 0L;
        }

        public void finish() {
            if (this.curRowCols > 0L) {
                this.collectRow();
            }
        }

        public String toString() {
            if (this.prevKV == null) {
                return "no data available for statistics";
            }
            return "Key length: " + this.keyLen + "\n" + "Val length: " + this.valLen + "\n" + "Row size (bytes): " + this.rowSizeBytes + "\n" + "Row size (columns): " + this.rowSizeCols + "\n" + "Key of biggest row: " + Bytes.toStringBinary(this.biggestRow);
        }
    }

    private static class LongStats {
        private long min = Long.MAX_VALUE;
        private long max = Long.MIN_VALUE;
        private long sum = 0L;
        private long count = 0L;

        private LongStats() {
        }

        void collect(long d) {
            if (d < this.min) {
                this.min = d;
            }
            if (d > this.max) {
                this.max = d;
            }
            this.sum += d;
            ++this.count;
        }

        public String toString() {
            return "count: " + this.count + "\tmin: " + this.min + "\tmax: " + this.max + "\tmean: " + (double)this.sum / (double)this.count;
        }
    }
}

