/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ClockOutOfSyncException;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HServerAddress;
import org.apache.hadoop.hbase.HServerLoad;
import org.apache.hadoop.hbase.PleaseHoldException;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.YouAreDeadException;
import org.apache.hadoop.hbase.ZooKeeperConnectionException;
import org.apache.hadoop.hbase.client.HConnection;
import org.apache.hadoop.hbase.client.HConnectionManager;
import org.apache.hadoop.hbase.ipc.HRegionInterface;
import org.apache.hadoop.hbase.master.DeadServer;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.handler.MetaServerShutdownHandler;
import org.apache.hadoop.hbase.master.handler.ServerShutdownHandler;
import org.apache.hadoop.hbase.monitoring.MonitoredTask;
import org.apache.hadoop.hbase.regionserver.RegionOpeningState;

public class ServerManager {
    private static final Log LOG = LogFactory.getLog(ServerManager.class);
    private volatile boolean clusterShutdown = false;
    private final Map<ServerName, HServerLoad> onlineServers = new ConcurrentHashMap<ServerName, HServerLoad>();
    private final Map<ServerName, HRegionInterface> serverConnections = new HashMap<ServerName, HRegionInterface>();
    private final ArrayList<ServerName> drainingServers = new ArrayList();
    private final Server master;
    private final MasterServices services;
    private final HConnection connection;
    private final DeadServer deadservers;
    private final long maxSkew;

    public ServerManager(Server master, MasterServices services) throws ZooKeeperConnectionException {
        this(master, services, true);
    }

    ServerManager(Server master, MasterServices services, boolean connect) throws ZooKeeperConnectionException {
        this.master = master;
        this.services = services;
        Configuration c = master.getConfiguration();
        this.maxSkew = c.getLong("hbase.master.maxclockskew", 30000L);
        this.deadservers = new DeadServer();
        this.connection = connect ? HConnectionManager.getConnection(c) : null;
    }

    ServerName regionServerStartup(InetAddress ia, int port, long serverStartcode, long serverCurrentTime) throws IOException {
        ServerName sn = new ServerName(ia.getHostName(), port, serverStartcode);
        this.checkClockSkew(sn, serverCurrentTime);
        this.checkIsDead(sn, "STARTUP");
        this.checkAlreadySameHostPort(sn);
        this.recordNewServer(sn, HServerLoad.EMPTY_HSERVERLOAD);
        return sn;
    }

    void regionServerReport(ServerName sn, HServerLoad hsl) throws YouAreDeadException, PleaseHoldException {
        this.checkIsDead(sn, "REPORT");
        if (!this.onlineServers.containsKey(sn)) {
            this.checkAlreadySameHostPort(sn);
            this.recordNewServer(sn, hsl);
        } else {
            this.onlineServers.put(sn, hsl);
        }
    }

    void checkAlreadySameHostPort(ServerName serverName) throws PleaseHoldException {
        ServerName existingServer = ServerName.findServerWithSameHostnamePort(this.getOnlineServersList(), serverName);
        if (existingServer != null) {
            String message = "Server serverName=" + serverName + " rejected; we already have " + existingServer.toString() + " registered with same hostname and port";
            LOG.info((Object)message);
            if (existingServer.getStartcode() < serverName.getStartcode()) {
                LOG.info((Object)("Triggering server recovery; existingServer " + existingServer + " looks stale, new server:" + serverName));
                this.expireServer(existingServer);
            }
            throw new PleaseHoldException(message);
        }
    }

    private void checkClockSkew(ServerName serverName, long serverCurrentTime) throws ClockOutOfSyncException {
        long skew = System.currentTimeMillis() - serverCurrentTime;
        if (skew > this.maxSkew) {
            String message = "Server " + serverName + " has been " + "rejected; Reported time is too far out of sync with master.  " + "Time difference of " + skew + "ms > max allowed of " + this.maxSkew + "ms";
            LOG.warn((Object)message);
            throw new ClockOutOfSyncException(message);
        }
    }

    private void checkIsDead(ServerName serverName, String what) throws YouAreDeadException {
        if (this.deadservers.isDeadServer(serverName)) {
            String message = "Server " + what + " rejected; currently processing " + serverName + " as dead server";
            LOG.debug((Object)message);
            throw new YouAreDeadException(message);
        }
        if (this.deadservers.cleanPreviousInstance(serverName)) {
            LOG.debug((Object)(what + ":" + " Server " + serverName + " came back up," + " removed it from the dead servers list"));
        }
    }

    void recordNewServer(ServerName serverName, HServerLoad hsl) {
        LOG.info((Object)("Registering server=" + serverName));
        this.onlineServers.put(serverName, hsl);
        this.serverConnections.remove(serverName);
    }

    public HServerLoad getLoad(ServerName serverName) {
        return this.onlineServers.get(serverName);
    }

    public HServerLoad getLoad(HServerAddress address) {
        ServerName sn = new ServerName(address.toString(), -1L);
        ServerName actual = ServerName.findServerWithSameHostnamePort(this.getOnlineServersList(), sn);
        return actual == null ? null : this.getLoad(actual);
    }

    public double getAverageLoad() {
        int totalLoad = 0;
        int numServers = 0;
        double averageLoad = 0.0;
        for (HServerLoad hsl : this.onlineServers.values()) {
            ++numServers;
            totalLoad += hsl.getNumberOfRegions();
        }
        averageLoad = (double)totalLoad / (double)numServers;
        return averageLoad;
    }

    int countOfRegionServers() {
        return this.onlineServers.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map<ServerName, HServerLoad> getOnlineServers() {
        Map<ServerName, HServerLoad> map = this.onlineServers;
        synchronized (map) {
            return Collections.unmodifiableMap(this.onlineServers);
        }
    }

    public Set<ServerName> getDeadServers() {
        return this.deadservers.clone();
    }

    public boolean areDeadServersInProgress() {
        return this.deadservers.areDeadServersInProgress();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    void letRegionServersShutdown() {
        Map<ServerName, HServerLoad> map = this.onlineServers;
        synchronized (map) {
            while (!this.onlineServers.isEmpty()) {
                StringBuilder sb = new StringBuilder();
                for (ServerName key : this.onlineServers.keySet()) {
                    if (sb.length() > 0) {
                        sb.append(", ");
                    }
                    sb.append(key);
                }
                LOG.info((Object)("Waiting on regionserver(s) to go down " + sb.toString()));
                try {
                    this.onlineServers.wait(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
            return;
        }
    }

    public synchronized void expireServer(ServerName serverName) {
        if (!this.onlineServers.containsKey(serverName)) {
            LOG.warn((Object)("Received expiration of " + serverName + " but server is not currently online"));
            return;
        }
        if (this.deadservers.contains(serverName)) {
            LOG.warn((Object)("Received expiration of " + serverName + " but server shutdown is already in progress"));
            return;
        }
        this.deadservers.add(serverName);
        this.onlineServers.remove(serverName);
        this.serverConnections.remove(serverName);
        if (this.clusterShutdown) {
            LOG.info((Object)("Cluster shutdown set; " + serverName + " expired; onlineServers=" + this.onlineServers.size()));
            if (this.onlineServers.isEmpty()) {
                this.master.stop("Cluster shutdown set; onlineServer=0");
            }
            return;
        }
        boolean carryingRoot = this.services.getAssignmentManager().isCarryingRoot(serverName);
        boolean carryingMeta = this.services.getAssignmentManager().isCarryingMeta(serverName);
        if (carryingRoot || carryingMeta) {
            this.services.getExecutorService().submit(new MetaServerShutdownHandler(this.master, this.services, this.deadservers, serverName, carryingRoot, carryingMeta));
        } else {
            this.services.getExecutorService().submit(new ServerShutdownHandler(this.master, this.services, this.deadservers, serverName, true));
        }
        LOG.debug((Object)("Added=" + serverName + " to dead servers, submitted shutdown handler to be executed, root=" + carryingRoot + ", meta=" + carryingMeta));
    }

    public boolean removeServerFromDrainList(ServerName sn) {
        if (!this.isServerOnline(sn)) {
            LOG.warn((Object)("Server " + sn + " is not currently online. " + "Removing from draining list anyway, as requested."));
        }
        return this.drainingServers.remove(sn);
    }

    public boolean addServerToDrainList(ServerName sn) {
        if (!this.isServerOnline(sn)) {
            LOG.warn((Object)("Server " + sn + " is not currently online. " + "Ignoring request to add it to draining list."));
            return false;
        }
        if (this.drainingServers.contains(sn)) {
            LOG.warn((Object)("Server " + sn + " is already in the draining server list." + "Ignoring request to add it again."));
            return false;
        }
        return this.drainingServers.add(sn);
    }

    public RegionOpeningState sendRegionOpen(ServerName server, HRegionInfo region, int versionOfOfflineNode) throws IOException {
        HRegionInterface hri = this.getServerConnection(server);
        if (hri == null) {
            LOG.warn((Object)("Attempting to send OPEN RPC to server " + server.toString() + " failed because no RPC connection found to this server"));
            return RegionOpeningState.FAILED_OPENING;
        }
        return versionOfOfflineNode == -1 ? hri.openRegion(region) : hri.openRegion(region, versionOfOfflineNode);
    }

    public void sendRegionOpen(ServerName server, List<HRegionInfo> regions) throws IOException {
        HRegionInterface hri = this.getServerConnection(server);
        if (hri == null) {
            LOG.warn((Object)("Attempting to send OPEN RPC to server " + server.toString() + " failed because no RPC connection found to this server"));
            return;
        }
        hri.openRegions(regions);
    }

    public boolean sendRegionClose(ServerName server, HRegionInfo region, int versionOfClosingNode) throws IOException {
        if (server == null) {
            throw new NullPointerException("Passed server is null");
        }
        HRegionInterface hri = this.getServerConnection(server);
        if (hri == null) {
            throw new IOException("Attempting to send CLOSE RPC to server " + server.toString() + " for region " + region.getRegionNameAsString() + " failed because no RPC connection found to this server");
        }
        return hri.closeRegion(region, versionOfClosingNode);
    }

    private HRegionInterface getServerConnection(ServerName sn) throws IOException {
        HRegionInterface hri = this.serverConnections.get(sn.toString());
        if (hri == null) {
            LOG.debug((Object)("New connection to " + sn.toString()));
            hri = this.connection.getHRegionConnection(sn.getHostname(), sn.getPort());
            this.serverConnections.put(sn, hri);
        }
        return hri;
    }

    public void waitForRegionServers(MonitoredTask status) throws InterruptedException {
        long interval = this.master.getConfiguration().getLong("hbase.master.wait.on.regionservers.interval", 1500L);
        long timeout = this.master.getConfiguration().getLong("hbase.master.wait.on.regionservers.timeout", 4500L);
        int minToStart = this.master.getConfiguration().getInt("hbase.master.wait.on.regionservers.mintostart", 1);
        int maxToStart = this.master.getConfiguration().getInt("hbase.master.wait.on.regionservers.maxtostart", Integer.MAX_VALUE);
        int count = 0;
        long slept = 0L;
        int oldcount = this.countOfRegionServers();
        while (!this.master.isStopped()) {
            Thread.sleep(interval);
            count = this.countOfRegionServers();
            if (count == oldcount && count >= minToStart && (slept += interval) >= timeout) {
                LOG.info((Object)("Finished waiting for regionserver count to settle; count=" + count + ", sleptFor=" + slept));
                break;
            }
            if (count >= maxToStart) {
                LOG.info((Object)("At least the max configured number of regionserver(s) have checked in: " + count));
                break;
            }
            String msg = count == 0 ? "Waiting on regionserver(s) to checkin" : "Waiting on regionserver(s) count to settle; currently=" + count;
            LOG.info((Object)msg);
            status.setStatus(msg);
            oldcount = count;
        }
    }

    public List<ServerName> getOnlineServersList() {
        return new ArrayList<ServerName>(this.onlineServers.keySet());
    }

    public List<ServerName> getDrainingServersList() {
        return new ArrayList<ServerName>(this.drainingServers);
    }

    public boolean isServerOnline(ServerName serverName) {
        return this.onlineServers.containsKey(serverName);
    }

    public void shutdownCluster() {
        this.clusterShutdown = true;
        this.master.stop("Cluster shutdown requested");
    }

    public boolean isClusterShutdown() {
        return this.clusterShutdown;
    }

    public void stop() {
        if (this.connection != null) {
            try {
                this.connection.close();
            }
            catch (IOException e) {
                LOG.error((Object)"Attempt to close connection to master failed", (Throwable)e);
            }
        }
    }
}

