/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.handler;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.catalog.CatalogTracker;
import org.apache.hadoop.hbase.catalog.MetaEditor;
import org.apache.hadoop.hbase.catalog.MetaReader;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.executor.EventHandler;
import org.apache.hadoop.hbase.master.AssignmentManager;
import org.apache.hadoop.hbase.master.DeadServer;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.zookeeper.KeeperException;

public class ServerShutdownHandler
extends EventHandler {
    private static final Log LOG = LogFactory.getLog(ServerShutdownHandler.class);
    private final ServerName serverName;
    private final MasterServices services;
    private final DeadServer deadServers;
    private final boolean shouldSplitHlog;

    public ServerShutdownHandler(Server server, MasterServices services, DeadServer deadServers, ServerName serverName, boolean shouldSplitHlog) {
        this(server, services, deadServers, serverName, EventHandler.EventType.M_SERVER_SHUTDOWN, shouldSplitHlog);
    }

    ServerShutdownHandler(Server server, MasterServices services, DeadServer deadServers, ServerName serverName, EventHandler.EventType type, boolean shouldSplitHlog) {
        super(server, type);
        this.serverName = serverName;
        this.server = server;
        this.services = services;
        this.deadServers = deadServers;
        if (!this.deadServers.contains(this.serverName)) {
            LOG.warn((Object)(this.serverName + " is NOT in deadservers; it should be!"));
        }
        this.shouldSplitHlog = shouldSplitHlog;
    }

    @Override
    public String getInformativeName() {
        if (this.serverName != null) {
            return this.getClass().getSimpleName() + " for " + this.serverName;
        }
        return super.getInformativeName();
    }

    private void verifyAndAssignRoot() throws InterruptedException, IOException, KeeperException {
        long timeout = this.server.getConfiguration().getLong("hbase.catalog.verification.timeout", 1000L);
        if (!this.server.getCatalogTracker().verifyRootRegionLocation(timeout)) {
            this.services.getAssignmentManager().assignRoot();
        }
    }

    private void verifyAndAssignRootWithRetries() throws IOException {
        int iTimes = this.server.getConfiguration().getInt("hbase.catalog.verification.retries", 10);
        long waitTime = this.server.getConfiguration().getLong("hbase.catalog.verification.timeout", 1000L);
        int iFlag = 0;
        while (true) {
            try {
                this.verifyAndAssignRoot();
            }
            catch (KeeperException e) {
                this.server.abort("In server shutdown processing, assigning root", e);
                throw new IOException("Aborting", e);
            }
            catch (Exception e) {
                if (iFlag >= iTimes) {
                    this.server.abort("verifyAndAssignRoot failed after" + iTimes + " times retries, aborting", e);
                    throw new IOException("Aborting", e);
                }
                try {
                    Thread.sleep(waitTime);
                }
                catch (InterruptedException e1) {
                    LOG.warn((Object)"Interrupted when is the thread sleep", (Throwable)e1);
                    Thread.currentThread().interrupt();
                    throw new IOException("Interrupted", e1);
                }
                ++iFlag;
                continue;
            }
            break;
        }
    }

    boolean isCarryingRoot() {
        return false;
    }

    boolean isCarryingMeta() {
        return false;
    }

    @Override
    public String toString() {
        String name = "UnknownServerName";
        if (this.server != null && this.server.getServerName() != null) {
            name = this.server.getServerName().toString();
        }
        return this.getClass().getSimpleName() + "-" + name + "-" + this.getSeqid();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void process() throws IOException {
        ServerName serverName = this.serverName;
        try {
            try {
                LOG.info((Object)("Splitting logs for " + serverName));
                this.services.getMasterFileSystem().splitLog(serverName);
            }
            catch (IOException ioe) {
                this.services.getExecutorService().submit(this);
                this.deadServers.add(serverName);
                throw new IOException("failed log splitting for " + serverName + ", will retry", ioe);
            }
            if (this.isCarryingRoot()) {
                LOG.info((Object)("Server " + serverName + " was carrying ROOT. Trying to assign."));
                this.services.getAssignmentManager().regionOffline(HRegionInfo.ROOT_REGIONINFO);
                this.verifyAndAssignRootWithRetries();
            }
            if (this.isCarryingMeta()) {
                LOG.info((Object)("Server " + serverName + " was carrying META. Trying to assign."));
                this.services.getAssignmentManager().regionOffline(HRegionInfo.FIRST_META_REGIONINFO);
                this.services.getAssignmentManager().assignMeta();
            }
            if (this.isCarryingRoot() || this.isCarryingMeta()) {
                this.services.getExecutorService().submit(new ServerShutdownHandler(this.server, this.services, this.deadServers, serverName, false));
                this.deadServers.add(serverName);
                return;
            }
            List<AssignmentManager.RegionState> regionsInTransition = this.services.getAssignmentManager().processServerShutdown(this.serverName);
            NavigableMap<HRegionInfo, Result> hris = null;
            while (!this.server.isStopped()) {
                try {
                    this.server.getCatalogTracker().waitForMeta();
                    hris = MetaReader.getServerUserRegions(this.server.getCatalogTracker(), this.serverName);
                    break;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new IOException("Interrupted", e);
                }
                catch (IOException ioe) {
                    LOG.info((Object)("Received exception accessing META during server shutdown of " + serverName + ", retrying META read"), (Throwable)ioe);
                }
            }
            for (AssignmentManager.RegionState regionState : regionsInTransition) {
                if (regionState.isClosing() || regionState.isPendingClose()) continue;
                LOG.debug((Object)("Removed " + regionState.getRegion().getRegionNameAsString() + " from list of regions to assign because in RIT" + " region state: " + (Object)((Object)regionState.getState())));
                if (hris == null) continue;
                hris.remove((Object)regionState.getRegion());
            }
            assert (regionsInTransition != null);
            LOG.info((Object)("Reassigning " + (hris == null ? 0 : hris.size()) + " region(s) that " + (serverName == null ? "null" : serverName) + " was carrying (skipping " + regionsInTransition.size() + " regions(s) that are already in transition)"));
            if (hris != null) {
                for (Map.Entry entry : hris.entrySet()) {
                    if (!ServerShutdownHandler.processDeadRegion((HRegionInfo)((Object)entry.getKey()), (Result)entry.getValue(), this.services.getAssignmentManager(), this.server.getCatalogTracker())) continue;
                    AssignmentManager.RegionState rit = this.services.getAssignmentManager().isRegionInTransition((HRegionInfo)((Object)entry.getKey()));
                    ServerName addressFromAM = this.services.getAssignmentManager().getRegionServerOfRegion((HRegionInfo)((Object)entry.getKey()));
                    if (rit != null && !rit.isClosing() && !rit.isPendingClose()) {
                        LOG.info((Object)("Skip assigning region " + rit.toString()));
                        continue;
                    }
                    if (addressFromAM != null && !addressFromAM.equals(this.serverName)) {
                        LOG.debug((Object)("Skip assigning region " + ((HRegionInfo)((Object)entry.getKey())).getRegionNameAsString() + " because it has been opened in " + addressFromAM.getServerName()));
                        continue;
                    }
                    this.services.getAssignmentManager().assign((HRegionInfo)((Object)entry.getKey()), true);
                }
            }
        }
        finally {
            this.deadServers.finish(serverName);
        }
        LOG.info((Object)("Finished processing of shutdown of " + serverName));
    }

    public static boolean processDeadRegion(HRegionInfo hri, Result result, AssignmentManager assignmentManager, CatalogTracker catalogTracker) throws IOException {
        boolean disabled = assignmentManager.getZKTable().isDisabledTable(hri.getTableNameAsString());
        if (disabled) {
            return false;
        }
        if (hri.isOffline() && hri.isSplit()) {
            LOG.debug((Object)("Offlined and split region " + hri.getRegionNameAsString() + "; checking daughter presence"));
            ServerShutdownHandler.fixupDaughters(result, assignmentManager, catalogTracker);
            return false;
        }
        return true;
    }

    public static int fixupDaughters(Result result, AssignmentManager assignmentManager, CatalogTracker catalogTracker) throws IOException {
        int fixedA = ServerShutdownHandler.fixupDaughter(result, HConstants.SPLITA_QUALIFIER, assignmentManager, catalogTracker);
        int fixedB = ServerShutdownHandler.fixupDaughter(result, HConstants.SPLITB_QUALIFIER, assignmentManager, catalogTracker);
        return fixedA + fixedB;
    }

    static int fixupDaughter(Result result, byte[] qualifier, AssignmentManager assignmentManager, CatalogTracker catalogTracker) throws IOException {
        HRegionInfo daughter = MetaReader.parseHRegionInfoFromCatalogResult(result, qualifier);
        if (daughter == null) {
            return 0;
        }
        if (ServerShutdownHandler.isDaughterMissing(catalogTracker, daughter)) {
            LOG.info((Object)("Fixup; missing daughter " + daughter.getRegionNameAsString()));
            MetaEditor.addDaughter(catalogTracker, daughter, null);
            assignmentManager.assign(daughter, true);
            return 1;
        }
        LOG.debug((Object)("Daughter " + daughter.getRegionNameAsString() + " present"));
        return 0;
    }

    private static boolean isDaughterMissing(CatalogTracker catalogTracker, HRegionInfo daughter) throws IOException {
        FindDaughterVisitor visitor = new FindDaughterVisitor(daughter);
        byte[] startrow = daughter.getRegionName();
        MetaReader.fullScan(catalogTracker, visitor, startrow);
        return !visitor.foundDaughter();
    }

    static class FindDaughterVisitor
    implements MetaReader.Visitor {
        private final HRegionInfo daughter;
        private boolean found = false;

        FindDaughterVisitor(HRegionInfo daughter) {
            this.daughter = daughter;
        }

        boolean foundDaughter() {
            return this.found;
        }

        @Override
        public boolean visit(Result r) throws IOException {
            HRegionInfo hri = MetaReader.parseHRegionInfoFromCatalogResult(r, HConstants.REGIONINFO_QUALIFIER);
            if (hri == null) {
                LOG.warn((Object)("No serialized HRegionInfo in " + r));
                return true;
            }
            byte[] value = r.getValue(HConstants.CATALOG_FAMILY, HConstants.SERVER_QUALIFIER);
            if (value == null) {
                return false;
            }
            if (!Bytes.equals(this.daughter.getTableName(), hri.getTableName())) {
                return false;
            }
            if (!Bytes.equals(this.daughter.getStartKey(), hri.getStartKey())) {
                return false;
            }
            this.found = true;
            return false;
        }
    }
}

