/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Ordering;
import java.io.DataInput;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.SortedSet;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HDFSBlocksDistribution;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.io.HalfStoreFileReader;
import org.apache.hadoop.hbase.io.Reference;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.Compression;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileScanner;
import org.apache.hadoop.hbase.io.hfile.HFileWriterV2;
import org.apache.hadoop.hbase.regionserver.StoreFileScanner;
import org.apache.hadoop.hbase.regionserver.TimeRangeTracker;
import org.apache.hadoop.hbase.util.BloomFilter;
import org.apache.hadoop.hbase.util.BloomFilterFactory;
import org.apache.hadoop.hbase.util.BloomFilterWriter;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.Writables;
import org.apache.hadoop.io.RawComparator;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.WritableUtils;

public class StoreFile {
    static final Log LOG = LogFactory.getLog((String)StoreFile.class.getName());
    public static final byte[] MAX_SEQ_ID_KEY = Bytes.toBytes("MAX_SEQ_ID_KEY");
    public static final byte[] MAJOR_COMPACTION_KEY = Bytes.toBytes("MAJOR_COMPACTION_KEY");
    public static final byte[] EXCLUDE_FROM_MINOR_COMPACTION_KEY = Bytes.toBytes("EXCLUDE_FROM_MINOR_COMPACTION");
    static final byte[] BLOOM_FILTER_TYPE_KEY = Bytes.toBytes("BLOOM_FILTER_TYPE");
    private static final byte[] LAST_BLOOM_KEY = Bytes.toBytes("LAST_BLOOM_KEY");
    public static final byte[] TIMERANGE_KEY = Bytes.toBytes("TIMERANGE");
    public static final int DEFAULT_BLOCKSIZE_SMALL = 8192;
    private final FileSystem fs;
    private final Path path;
    private Reference reference;
    private Path referencePath;
    private final CacheConfig cacheConf;
    private HDFSBlocksDistribution hdfsBlocksDistribution;
    private long sequenceid = -1L;
    private long maxMemstoreTS = -1L;
    private AtomicBoolean majorCompaction = null;
    private boolean excludeFromMinorCompaction = false;
    public static final byte[] BULKLOAD_TASK_KEY = Bytes.toBytes("BULKLOAD_SOURCE_TASK");
    public static final byte[] BULKLOAD_TIME_KEY = Bytes.toBytes("BULKLOAD_TIMESTAMP");
    private Map<byte[], byte[]> metadataMap;
    private static final Pattern REF_NAME_PARSER = Pattern.compile("^([0-9a-f]+)(?:\\.(.+))?$");
    private volatile Reader reader;
    private final BloomType cfBloomType;
    private long modificationTimeStamp = 0L;

    public long getMaxMemstoreTS() {
        return this.maxMemstoreTS;
    }

    public void setMaxMemstoreTS(long maxMemstoreTS) {
        this.maxMemstoreTS = maxMemstoreTS;
    }

    StoreFile(FileSystem fs, Path p, Configuration conf, CacheConfig cacheConf, BloomType cfBloomType) throws IOException {
        this.fs = fs;
        this.path = p;
        this.cacheConf = cacheConf;
        if (StoreFile.isReference(p)) {
            this.reference = Reference.read(fs, p);
            this.referencePath = StoreFile.getReferredToFile(this.path);
        }
        if (BloomFilterFactory.isBloomEnabled(conf)) {
            this.cfBloomType = cfBloomType;
        } else {
            LOG.info((Object)("Ignoring bloom filter check for file " + this.path + ": " + "cfBloomType=" + (Object)((Object)cfBloomType) + " (disabled in config)"));
            this.cfBloomType = BloomType.NONE;
        }
        FileStatus[] stats = FSUtils.listStatus(fs, p, null);
        this.modificationTimeStamp = stats != null && stats.length == 1 ? stats[0].getModificationTime() : 0L;
    }

    Path getPath() {
        return this.path;
    }

    byte[] getFamily() {
        return Bytes.toBytes(this.path.getParent().getName());
    }

    boolean isReference() {
        return this.reference != null;
    }

    public static boolean isReference(Path p) {
        return !p.getName().startsWith("_") && StoreFile.isReference(p, REF_NAME_PARSER.matcher(p.getName()));
    }

    public static boolean isReference(Path p, Matcher m) {
        if (m == null || !m.matches()) {
            LOG.warn((Object)("Failed match of store file name " + p.toString()));
            throw new RuntimeException("Failed match of store file name " + p.toString());
        }
        return m.groupCount() > 1 && m.group(2) != null;
    }

    static Path getReferredToFile(Path p) {
        Matcher m = REF_NAME_PARSER.matcher(p.getName());
        if (m == null || !m.matches()) {
            LOG.warn((Object)("Failed match of store file name " + p.toString()));
            throw new RuntimeException("Failed match of store file name " + p.toString());
        }
        String otherRegion = m.group(2);
        Path tableDir = p.getParent().getParent().getParent();
        String nameStrippedOfSuffix = m.group(1);
        return new Path(new Path(new Path(tableDir, otherRegion), p.getParent().getName()), nameStrippedOfSuffix);
    }

    boolean isMajorCompaction() {
        if (this.majorCompaction == null) {
            throw new NullPointerException("This has not been set yet");
        }
        return this.majorCompaction.get();
    }

    boolean excludeFromMinorCompaction() {
        return this.excludeFromMinorCompaction;
    }

    public long getMaxSequenceId() {
        return this.sequenceid;
    }

    public long getModificationTimeStamp() {
        return this.modificationTimeStamp;
    }

    public static long getMaxMemstoreTSInList(Collection<StoreFile> sfs) {
        long max = 0L;
        for (StoreFile sf : sfs) {
            if (sf.isBulkLoadResult()) continue;
            max = Math.max(max, sf.getMaxMemstoreTS());
        }
        return max;
    }

    public static long getMaxSequenceIdInList(Collection<StoreFile> sfs) {
        long max = 0L;
        for (StoreFile sf : sfs) {
            if (sf.isBulkLoadResult()) continue;
            max = Math.max(max, sf.getMaxSequenceId());
        }
        return max;
    }

    boolean isBulkLoadResult() {
        return this.metadataMap.containsKey(BULKLOAD_TIME_KEY);
    }

    public long getBulkLoadTimestamp() {
        return Bytes.toLong(this.metadataMap.get(BULKLOAD_TIME_KEY));
    }

    public HDFSBlocksDistribution getHDFSBlockDistribution() {
        return this.hdfsBlocksDistribution;
    }

    private static HDFSBlocksDistribution computeRefFileHDFSBlockDistribution(FileSystem fs, Reference reference, Path referencePath) throws IOException {
        if (referencePath == null) {
            return null;
        }
        FileStatus status = fs.getFileStatus(referencePath);
        long start = 0L;
        long length = 0L;
        if (Reference.isTopFileRegion(reference.getFileRegion())) {
            start = status.getLen() / 2L;
            length = status.getLen() - status.getLen() / 2L;
        } else {
            start = 0L;
            length = status.getLen() / 2L;
        }
        return FSUtils.computeHDFSBlocksDistribution(fs, status, start, length);
    }

    public static HDFSBlocksDistribution computeHDFSBlockDistribution(FileSystem fs, Path p) throws IOException {
        if (StoreFile.isReference(p)) {
            Reference reference = Reference.read(fs, p);
            Path referencePath = StoreFile.getReferredToFile(p);
            return StoreFile.computeRefFileHDFSBlockDistribution(fs, reference, referencePath);
        }
        FileStatus status = fs.getFileStatus(p);
        long length = status.getLen();
        return FSUtils.computeHDFSBlocksDistribution(fs, status, 0L, length);
    }

    private void computeHDFSBlockDistribution() throws IOException {
        if (this.isReference()) {
            this.hdfsBlocksDistribution = StoreFile.computeRefFileHDFSBlockDistribution(this.fs, this.reference, this.referencePath);
        } else {
            FileStatus status = this.fs.getFileStatus(this.path);
            long length = status.getLen();
            this.hdfsBlocksDistribution = FSUtils.computeHDFSBlocksDistribution(this.fs, status, 0L, length);
        }
    }

    private Reader open() throws IOException {
        if (this.reader != null) {
            throw new IllegalAccessError("Already open");
        }
        this.reader = this.isReference() ? new HalfStoreFileReader(this.fs, this.referencePath, this.cacheConf, this.reference) : new Reader(this.fs, this.path, this.cacheConf);
        this.computeHDFSBlockDistribution();
        this.metadataMap = Collections.unmodifiableMap(this.reader.loadFileInfo());
        byte[] b = this.metadataMap.get(MAX_SEQ_ID_KEY);
        if (b != null) {
            this.sequenceid = Bytes.toLong(b);
            if (this.isReference() && Reference.isTopFileRegion(this.reference.getFileRegion())) {
                ++this.sequenceid;
            }
        }
        this.reader.setSequenceID(this.sequenceid);
        b = this.metadataMap.get(HFileWriterV2.MAX_MEMSTORE_TS_KEY);
        if (b != null) {
            this.maxMemstoreTS = Bytes.toLong(b);
        }
        if ((b = this.metadataMap.get(MAJOR_COMPACTION_KEY)) != null) {
            boolean mc = Bytes.toBoolean(b);
            if (this.majorCompaction == null) {
                this.majorCompaction = new AtomicBoolean(mc);
            } else {
                this.majorCompaction.set(mc);
            }
        } else {
            this.majorCompaction = new AtomicBoolean(false);
        }
        b = this.metadataMap.get(EXCLUDE_FROM_MINOR_COMPACTION_KEY);
        this.excludeFromMinorCompaction = b != null && Bytes.toBoolean(b);
        BloomType hfileBloomType = this.reader.getBloomFilterType();
        if (this.cfBloomType != BloomType.NONE) {
            this.reader.loadBloomfilter();
            if (hfileBloomType != this.cfBloomType) {
                LOG.info((Object)("HFile Bloom filter type for " + this.reader.getHFileReader().getName() + ": " + (Object)((Object)hfileBloomType) + ", but " + (Object)((Object)this.cfBloomType) + " specified in column family " + "configuration"));
            }
        } else if (hfileBloomType != BloomType.NONE) {
            LOG.info((Object)("Bloom filter turned off by CF config for " + this.reader.getHFileReader().getName()));
        }
        try {
            byte[] timerangeBytes = this.metadataMap.get(TIMERANGE_KEY);
            if (timerangeBytes != null) {
                this.reader.timeRangeTracker = new TimeRangeTracker();
                Writables.copyWritable(timerangeBytes, (Writable)this.reader.timeRangeTracker);
            }
        }
        catch (IllegalArgumentException e) {
            LOG.error((Object)"Error reading timestamp range data from meta -- proceeding without", (Throwable)e);
            this.reader.timeRangeTracker = null;
        }
        return this.reader;
    }

    public Reader createReader() throws IOException {
        if (this.reader == null) {
            this.reader = this.open();
        }
        return this.reader;
    }

    public Reader getReader() {
        return this.reader;
    }

    public synchronized void closeReader(boolean evictOnClose) throws IOException {
        if (this.reader != null) {
            this.reader.close(evictOnClose);
            this.reader = null;
        }
    }

    public void deleteReader() throws IOException {
        this.closeReader(true);
        this.fs.delete(this.getPath(), true);
    }

    public String toString() {
        return this.path.toString() + (this.isReference() ? "-" + this.referencePath + "-" + this.reference.toString() : "");
    }

    public String toStringDetailed() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.path.toString());
        sb.append(", isReference=").append(this.isReference());
        sb.append(", isBulkLoadResult=").append(this.isBulkLoadResult());
        if (this.isBulkLoadResult()) {
            sb.append(", bulkLoadTS=").append(this.getBulkLoadTimestamp());
        } else {
            sb.append(", seqid=").append(this.getMaxSequenceId());
        }
        sb.append(", majorCompaction=").append(this.isMajorCompaction());
        return sb.toString();
    }

    public static Path rename(FileSystem fs, Path src, Path tgt) throws IOException {
        if (!fs.exists(src)) {
            throw new FileNotFoundException(src.toString());
        }
        if (!fs.rename(src, tgt)) {
            throw new IOException("Failed rename of " + src + " to " + tgt);
        }
        return tgt;
    }

    public static Writer createWriter(FileSystem fs, Path dir, int blocksize, Configuration conf, CacheConfig cacheConf) throws IOException {
        return StoreFile.createWriter(fs, dir, blocksize, null, null, conf, cacheConf, BloomType.NONE, 0L);
    }

    public static Writer createWriter(FileSystem fs, Path dir, int blocksize, Compression.Algorithm algorithm, KeyValue.KVComparator c, Configuration conf, CacheConfig cacheConf, BloomType bloomType, long maxKeyCount) throws IOException {
        if (!fs.exists(dir)) {
            fs.mkdirs(dir);
        }
        Path path = StoreFile.getUniqueFile(fs, dir);
        if (!BloomFilterFactory.isBloomEnabled(conf)) {
            bloomType = BloomType.NONE;
        }
        return new Writer(fs, path, blocksize, algorithm == null ? HFile.DEFAULT_COMPRESSION_ALGORITHM : algorithm, conf, cacheConf, c == null ? KeyValue.COMPARATOR : c, bloomType, maxKeyCount);
    }

    public static Path getUniqueFile(FileSystem fs, Path dir) throws IOException {
        if (!fs.getFileStatus(dir).isDir()) {
            throw new IOException("Expecting " + dir.toString() + " to be a directory");
        }
        return StoreFile.getRandomFilename(fs, dir);
    }

    static Path getRandomFilename(FileSystem fs, Path dir) throws IOException {
        return StoreFile.getRandomFilename(fs, dir, null);
    }

    static Path getRandomFilename(FileSystem fs, Path dir, String suffix) throws IOException {
        return new Path(dir, UUID.randomUUID().toString().replaceAll("-", "") + (suffix == null ? "" : suffix));
    }

    static Path split(FileSystem fs, Path splitDir, StoreFile f, byte[] splitRow, Reference.Range range) throws IOException {
        Reference r = new Reference(splitRow, range);
        String parentRegionName = f.getPath().getParent().getParent().getName();
        Path p = new Path(splitDir, f.getPath().getName() + "." + parentRegionName);
        return r.write(fs, p);
    }

    static abstract class Comparators {
        static final Comparator<StoreFile> FLUSH_TIME = Ordering.compound((Iterable)ImmutableList.of((Object)Ordering.natural().onResultOf((Function)new GetBulkTime()), (Object)Ordering.natural().onResultOf((Function)new GetSeqId()), (Object)Ordering.natural().onResultOf((Function)new GetPathName())));
        static final Comparator<StoreFile> FILE_SIZE = Ordering.natural().reverse().onResultOf((Function)new Function<StoreFile, Long>(){

            public Long apply(StoreFile sf) {
                return sf.getReader().length();
            }
        });

        Comparators() {
        }

        private static class GetPathName
        implements Function<StoreFile, String> {
            private GetPathName() {
            }

            public String apply(StoreFile sf) {
                return sf.getPath().getName();
            }
        }

        private static class GetSeqId
        implements Function<StoreFile, Long> {
            private GetSeqId() {
            }

            public Long apply(StoreFile sf) {
                if (sf.isBulkLoadResult()) {
                    return -1L;
                }
                return sf.getMaxSequenceId();
            }
        }

        private static class GetBulkTime
        implements Function<StoreFile, Long> {
            private GetBulkTime() {
            }

            public Long apply(StoreFile sf) {
                if (!sf.isBulkLoadResult()) {
                    return Long.MAX_VALUE;
                }
                return sf.getBulkLoadTimestamp();
            }
        }
    }

    public static class Reader {
        static final Log LOG = LogFactory.getLog((String)Reader.class.getName());
        protected BloomFilter bloomFilter = null;
        protected BloomType bloomFilterType;
        private final HFile.Reader reader;
        protected TimeRangeTracker timeRangeTracker = null;
        protected long sequenceID = -1L;
        private byte[] lastBloomKey;

        public Reader(FileSystem fs, Path path, CacheConfig cacheConf) throws IOException {
            this.reader = HFile.createReader(fs, path, cacheConf);
            this.bloomFilterType = BloomType.NONE;
        }

        Reader() {
            this.reader = null;
        }

        public RawComparator<byte[]> getComparator() {
            return this.reader.getComparator();
        }

        public StoreFileScanner getStoreFileScanner(boolean cacheBlocks, boolean pread) {
            return this.getStoreFileScanner(cacheBlocks, pread, false);
        }

        public StoreFileScanner getStoreFileScanner(boolean cacheBlocks, boolean pread, boolean isCompaction) {
            return new StoreFileScanner(this, this.getScanner(cacheBlocks, pread, isCompaction), !isCompaction);
        }

        @Deprecated
        public HFileScanner getScanner(boolean cacheBlocks, boolean pread) {
            return this.getScanner(cacheBlocks, pread, false);
        }

        @Deprecated
        public HFileScanner getScanner(boolean cacheBlocks, boolean pread, boolean isCompaction) {
            return this.reader.getScanner(cacheBlocks, pread, isCompaction);
        }

        public void close(boolean evictOnClose) throws IOException {
            this.reader.close(evictOnClose);
        }

        public boolean shouldSeek(Scan scan, SortedSet<byte[]> columns) {
            return this.passesTimerangeFilter(scan) && this.passesBloomFilter(scan, columns);
        }

        private boolean passesTimerangeFilter(Scan scan) {
            if (this.timeRangeTracker == null) {
                return true;
            }
            return this.timeRangeTracker.includesTimeRange(scan.getTimeRange());
        }

        private boolean passesBloomFilter(Scan scan, SortedSet<byte[]> columns) {
            if (!scan.isGetScan()) {
                return true;
            }
            byte[] row = scan.getStartRow();
            switch (this.bloomFilterType) {
                case ROW: {
                    return this.passesBloomFilter(row, 0, row.length, null, 0, 0);
                }
                case ROWCOL: {
                    if (columns != null && columns.size() == 1) {
                        byte[] column = columns.first();
                        return this.passesBloomFilter(row, 0, row.length, column, 0, column.length);
                    }
                    return true;
                }
            }
            return true;
        }

        public boolean passesBloomFilter(byte[] row, int rowOffset, int rowLen, byte[] col, int colOffset, int colLen) {
            byte[] key;
            if (this.bloomFilter == null) {
                return true;
            }
            switch (this.bloomFilterType) {
                case ROW: {
                    if (col != null) {
                        throw new RuntimeException("Row-only Bloom filter called with column specified");
                    }
                    if (rowOffset != 0 || rowLen != row.length) {
                        throw new AssertionError((Object)"For row-only Bloom filters the row must occupy the whole array");
                    }
                    key = row;
                    break;
                }
                case ROWCOL: {
                    key = this.bloomFilter.createBloomKey(row, rowOffset, rowLen, col, colOffset, colLen);
                    break;
                }
                default: {
                    return true;
                }
            }
            BloomFilter bloomFilter = this.bloomFilter;
            if (bloomFilter == null) {
                return true;
            }
            if (this.reader.getTrailer().getEntryCount() == 0L) {
                return false;
            }
            try {
                boolean shouldCheckBloom;
                ByteBuffer bloom;
                if (bloomFilter.supportsAutoLoading()) {
                    bloom = null;
                    shouldCheckBloom = true;
                } else {
                    bloom = this.reader.getMetaBlock("BLOOM_FILTER_DATA", true);
                    boolean bl = shouldCheckBloom = bloom != null;
                }
                if (shouldCheckBloom) {
                    boolean exists;
                    boolean keyIsAfterLast;
                    boolean bl = keyIsAfterLast = this.lastBloomKey != null && bloomFilter.getComparator().compare((Object)key, (Object)this.lastBloomKey) > 0;
                    if (this.bloomFilterType == BloomType.ROWCOL) {
                        byte[] rowBloomKey = bloomFilter.createBloomKey(row, 0, row.length, null, 0, 0);
                        exists = keyIsAfterLast && bloomFilter.getComparator().compare((Object)rowBloomKey, (Object)this.lastBloomKey) > 0 ? false : this.bloomFilter.contains(key, 0, key.length, bloom) || this.bloomFilter.contains(rowBloomKey, 0, rowBloomKey.length, bloom);
                    } else {
                        exists = !keyIsAfterLast && this.bloomFilter.contains(key, 0, key.length, bloom);
                    }
                    return exists;
                }
            }
            catch (IOException e) {
                LOG.error((Object)"Error reading bloom filter data -- proceeding without", (Throwable)e);
                this.setBloomFilterFaulty();
            }
            catch (IllegalArgumentException e) {
                LOG.error((Object)"Bad bloom filter data -- proceeding without", (Throwable)e);
                this.setBloomFilterFaulty();
            }
            return true;
        }

        public Map<byte[], byte[]> loadFileInfo() throws IOException {
            Map<byte[], byte[]> fi = this.reader.loadFileInfo();
            byte[] b = fi.get(BLOOM_FILTER_TYPE_KEY);
            if (b != null) {
                this.bloomFilterType = BloomType.valueOf(Bytes.toString(b));
            }
            this.lastBloomKey = fi.get(LAST_BLOOM_KEY);
            return fi;
        }

        public void loadBloomfilter() {
            if (this.bloomFilter != null) {
                return;
            }
            try {
                DataInput bloomMeta = this.reader.getBloomFilterMetadata();
                if (bloomMeta != null) {
                    if (this.bloomFilterType == BloomType.NONE) {
                        throw new IOException("valid bloom filter type not found in FileInfo");
                    }
                    this.bloomFilter = BloomFilterFactory.createFromMeta(bloomMeta, this.reader);
                    LOG.info((Object)("Loaded " + (Object)((Object)this.bloomFilterType) + " " + this.bloomFilter.getClass().getSimpleName() + " metadata for " + this.reader.getName()));
                }
            }
            catch (IOException e) {
                LOG.error((Object)"Error reading bloom filter meta -- proceeding without", (Throwable)e);
                this.bloomFilter = null;
            }
            catch (IllegalArgumentException e) {
                LOG.error((Object)"Bad bloom filter meta -- proceeding without", (Throwable)e);
                this.bloomFilter = null;
            }
        }

        public long getFilterEntries() {
            return this.bloomFilter != null ? this.bloomFilter.getKeyCount() : this.reader.getEntries();
        }

        public void setBloomFilterFaulty() {
            this.bloomFilter = null;
        }

        public byte[] getLastKey() {
            return this.reader.getLastKey();
        }

        public byte[] midkey() throws IOException {
            return this.reader.midkey();
        }

        public long length() {
            return this.reader.length();
        }

        public long getTotalUncompressedBytes() {
            return this.reader.getTrailer().getTotalUncompressedBytes();
        }

        public long getEntries() {
            return this.reader.getEntries();
        }

        public byte[] getFirstKey() {
            return this.reader.getFirstKey();
        }

        public long indexSize() {
            return this.reader.indexSize();
        }

        public BloomType getBloomFilterType() {
            return this.bloomFilterType;
        }

        public long getSequenceID() {
            return this.sequenceID;
        }

        public void setSequenceID(long sequenceID) {
            this.sequenceID = sequenceID;
        }

        BloomFilter getBloomFilter() {
            return this.bloomFilter;
        }

        long getUncompressedDataIndexSize() {
            return this.reader.getTrailer().getUncompressedDataIndexSize();
        }

        public long getTotalBloomSize() {
            if (this.bloomFilter == null) {
                return 0L;
            }
            return this.bloomFilter.getByteSize();
        }

        public int getHFileVersion() {
            return this.reader.getTrailer().getVersion();
        }

        HFile.Reader getHFileReader() {
            return this.reader;
        }

        void disableBloomFilterForTesting() {
            this.bloomFilter = null;
        }
    }

    public static class Writer {
        private final BloomFilterWriter bloomFilterWriter;
        private final BloomType bloomType;
        private byte[] lastBloomKey;
        private int lastBloomKeyOffset;
        private int lastBloomKeyLen;
        private KeyValue.KVComparator kvComparator;
        private KeyValue lastKv = null;
        TimeRangeTracker timeRangeTracker = new TimeRangeTracker();
        boolean isTimeRangeTrackerSet = false;
        protected HFile.Writer writer;

        public Writer(FileSystem fs, Path path, int blocksize, Compression.Algorithm compress, Configuration conf, CacheConfig cacheConf, KeyValue.KVComparator comparator, BloomType bloomType, long maxKeys) throws IOException {
            this.writer = HFile.getWriterFactory(conf, cacheConf).createWriter(fs, path, blocksize, compress, comparator.getRawComparator());
            this.kvComparator = comparator;
            this.bloomFilterWriter = BloomFilterFactory.createBloomAtWrite(conf, cacheConf, bloomType, (int)Math.min(maxKeys, Integer.MAX_VALUE), this.writer);
            if (this.bloomFilterWriter != null) {
                this.bloomType = bloomType;
                LOG.info((Object)("Bloom filter type for " + path + ": " + (Object)((Object)this.bloomType) + ", " + this.bloomFilterWriter.getClass().getSimpleName()));
            } else {
                this.bloomType = BloomType.NONE;
            }
        }

        public void appendMetadata(long maxSequenceId, boolean majorCompaction) throws IOException {
            this.writer.appendFileInfo(MAX_SEQ_ID_KEY, Bytes.toBytes(maxSequenceId));
            this.writer.appendFileInfo(MAJOR_COMPACTION_KEY, Bytes.toBytes(majorCompaction));
            this.appendTimeRangeMetadata();
        }

        public void appendTimeRangeMetadata() throws IOException {
            this.appendFileInfo(TIMERANGE_KEY, WritableUtils.toByteArray((Writable[])new Writable[]{this.timeRangeTracker}));
        }

        public void setTimeRangeTracker(TimeRangeTracker trt) {
            this.timeRangeTracker = trt;
            this.isTimeRangeTrackerSet = true;
        }

        public void includeInTimeRangeTracker(KeyValue kv) {
            if (!this.isTimeRangeTrackerSet) {
                this.timeRangeTracker.includeTimestamp(kv);
            }
        }

        public void includeInTimeRangeTracker(byte[] key) {
            if (!this.isTimeRangeTrackerSet) {
                this.timeRangeTracker.includeTimestamp(key);
            }
        }

        public void append(KeyValue kv) throws IOException {
            if (this.bloomFilterWriter != null) {
                boolean newKey = true;
                if (this.lastKv != null) {
                    switch (this.bloomType) {
                        case ROW: {
                            newKey = !this.kvComparator.matchingRows(kv, this.lastKv);
                            break;
                        }
                        case ROWCOL: {
                            newKey = !this.kvComparator.matchingRowColumn(kv, this.lastKv);
                            break;
                        }
                        case NONE: {
                            newKey = false;
                            break;
                        }
                        default: {
                            throw new IOException("Invalid Bloom filter type: " + (Object)((Object)this.bloomType));
                        }
                    }
                }
                if (newKey) {
                    int bloomKeyLen;
                    int bloomKeyOffset;
                    byte[] bloomKey;
                    switch (this.bloomType) {
                        case ROW: {
                            bloomKey = kv.getBuffer();
                            bloomKeyOffset = kv.getRowOffset();
                            bloomKeyLen = kv.getRowLength();
                            break;
                        }
                        case ROWCOL: {
                            bloomKey = this.bloomFilterWriter.createBloomKey(kv.getBuffer(), kv.getRowOffset(), kv.getRowLength(), kv.getBuffer(), kv.getQualifierOffset(), kv.getQualifierLength());
                            bloomKeyOffset = 0;
                            bloomKeyLen = bloomKey.length;
                            break;
                        }
                        default: {
                            throw new IOException("Invalid Bloom filter type: " + (Object)((Object)this.bloomType) + " (ROW or ROWCOL expected)");
                        }
                    }
                    this.bloomFilterWriter.add(bloomKey, bloomKeyOffset, bloomKeyLen);
                    if (this.lastBloomKey != null && this.bloomFilterWriter.getComparator().compare(bloomKey, bloomKeyOffset, bloomKeyLen, this.lastBloomKey, this.lastBloomKeyOffset, this.lastBloomKeyLen) <= 0) {
                        throw new IOException("Non-increasing Bloom keys: " + Bytes.toStringBinary(bloomKey, bloomKeyOffset, bloomKeyLen) + " after " + Bytes.toStringBinary(this.lastBloomKey, this.lastBloomKeyOffset, this.lastBloomKeyLen));
                    }
                    this.lastBloomKey = bloomKey;
                    this.lastBloomKeyOffset = bloomKeyOffset;
                    this.lastBloomKeyLen = bloomKeyLen;
                    this.lastKv = kv;
                }
            }
            this.writer.append(kv);
            this.includeInTimeRangeTracker(kv);
        }

        public Path getPath() {
            return this.writer.getPath();
        }

        boolean hasBloom() {
            return this.bloomFilterWriter != null;
        }

        BloomFilterWriter getBloomWriter() {
            return this.bloomFilterWriter;
        }

        public void close() throws IOException {
            boolean haveBloom;
            boolean bl = haveBloom = this.bloomFilterWriter != null && this.bloomFilterWriter.getKeyCount() > 0L;
            if (haveBloom) {
                this.bloomFilterWriter.compactBloom();
                this.writer.addBloomFilter(this.bloomFilterWriter);
                this.writer.appendFileInfo(BLOOM_FILTER_TYPE_KEY, Bytes.toBytes(this.bloomType.toString()));
                if (this.lastBloomKey != null) {
                    this.writer.appendFileInfo(LAST_BLOOM_KEY, Arrays.copyOfRange(this.lastBloomKey, this.lastBloomKeyOffset, this.lastBloomKeyOffset + this.lastBloomKeyLen));
                }
            }
            this.writer.close();
            if (haveBloom && this.bloomFilterWriter.getMaxKeys() > 0L) {
                LOG.info((Object)("Bloom added to HFile (" + this.getPath() + "): " + this.bloomFilterWriter.toString().replace("\n", "; ")));
            }
        }

        public void appendFileInfo(byte[] key, byte[] value) throws IOException {
            this.writer.appendFileInfo(key, value);
        }
    }

    public static enum BloomType {
        NONE,
        ROW,
        ROWCOL;

    }
}

