/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.zookeeper;

import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.RetryCounter;
import org.apache.hadoop.hbase.util.RetryCounterFactory;
import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;

public class RecoverableZooKeeper {
    private static final Log LOG = LogFactory.getLog(RecoverableZooKeeper.class);
    private ZooKeeper zk;
    private final RetryCounterFactory retryCounterFactory;
    private final String identifier;
    private final byte[] id;
    private int retryIntervalMillis;
    private static final int ID_OFFSET = 4;
    private static final byte MAGIC = -1;
    private static final int MAGIC_OFFSET = 1;

    public RecoverableZooKeeper(String quorumServers, int seesionTimeout, Watcher watcher, int maxRetries, int retryIntervalMillis) throws IOException {
        this.zk = new ZooKeeper(quorumServers, seesionTimeout, watcher);
        this.retryCounterFactory = new RetryCounterFactory(maxRetries, retryIntervalMillis);
        this.retryIntervalMillis = retryIntervalMillis;
        this.identifier = ManagementFactory.getRuntimeMXBean().getName();
        LOG.info((Object)("The identifier of this process is " + this.identifier));
        this.id = Bytes.toBytes(this.identifier);
    }

    public void delete(String path, int version) throws InterruptedException, KeeperException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        boolean isRetry = false;
        while (true) {
            try {
                this.zk.delete(path, version);
                return;
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case NONODE: {
                        if (isRetry) {
                            LOG.info((Object)("Node " + path + " already deleted. Assuming that a " + "previous attempt succeeded."));
                            return;
                        }
                        LOG.warn((Object)("Node " + path + " already deleted, and this is not a " + "retry"));
                        throw e;
                    }
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper delete failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                isRetry = true;
                continue;
            }
            break;
        }
    }

    public Stat exists(String path, Watcher watcher) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                return this.zk.exists(path, watcher);
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper exists failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public Stat exists(String path, boolean watch) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                return this.zk.exists(path, watch);
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper exists failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public List<String> getChildren(String path, Watcher watcher) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                return this.zk.getChildren(path, watcher);
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper getChildren failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public List<String> getChildren(String path, boolean watch) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                return this.zk.getChildren(path, watch);
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper getChildren failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public byte[] getData(String path, Watcher watcher, Stat stat) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                byte[] revData = this.zk.getData(path, watcher, stat);
                return this.removeMetaData(revData);
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper getData failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public byte[] getData(String path, boolean watch, Stat stat) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                byte[] revData = this.zk.getData(path, watch, stat);
                return this.removeMetaData(revData);
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper getData failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public Stat setData(String path, byte[] data, int version) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        byte[] newData = this.appendMetaData(data);
        while (true) {
            try {
                return this.zk.setData(path, newData, version);
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper setData failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    case BADVERSION: {
                        Stat stat = new Stat();
                        byte[] revData = this.zk.getData(path, false, stat);
                        int idLength = Bytes.toInt(revData, 4);
                        int dataLength = revData.length - 4 - idLength;
                        int dataOffset = 4 + idLength;
                        if (Bytes.compareTo(revData, 4, this.id.length, revData, dataOffset, dataLength) == 0) {
                            return stat;
                        }
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public String create(String path, byte[] data, List<ACL> acl, CreateMode createMode) throws KeeperException, InterruptedException {
        byte[] newData = this.appendMetaData(data);
        switch (createMode) {
            case EPHEMERAL: 
            case PERSISTENT: {
                return this.createNonSequential(path, newData, acl, createMode);
            }
            case EPHEMERAL_SEQUENTIAL: 
            case PERSISTENT_SEQUENTIAL: {
                return this.createSequential(path, newData, acl, createMode);
            }
        }
        throw new IllegalArgumentException("Unrecognized CreateMode: " + createMode);
    }

    private String createNonSequential(String path, byte[] data, List<ACL> acl, CreateMode createMode) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        boolean isRetry = false;
        while (true) {
            try {
                return this.zk.create(path, data, acl, createMode);
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case NODEEXISTS: {
                        if (isRetry) {
                            byte[] currentData = this.zk.getData(path, false, null);
                            if (currentData != null && Bytes.compareTo(currentData, data) == 0) {
                                return path;
                            }
                            LOG.error((Object)("Node " + path + " already exists with " + Bytes.toStringBinary(currentData) + ", could not write " + Bytes.toStringBinary(data)));
                            throw e;
                        }
                        LOG.error((Object)("Node " + path + " already exists and this is not a " + "retry"));
                        throw e;
                    }
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper create failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                isRetry = true;
                continue;
            }
            break;
        }
    }

    private String createSequential(String path, byte[] data, List<ACL> acl, CreateMode createMode) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        boolean first = true;
        String newPath = path + this.identifier;
        while (true) {
            try {
                String previousResult;
                if (!first && (previousResult = this.findPreviousSequentialNode(newPath)) != null) {
                    return previousResult;
                }
                first = false;
                return this.zk.create(newPath, data, acl, createMode);
            }
            catch (KeeperException e) {
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case OPERATIONTIMEOUT: {
                        LOG.warn((Object)("Possibly transient ZooKeeper exception: " + (Object)((Object)e)));
                        if (retryCounter.shouldRetry()) break;
                        LOG.error((Object)("ZooKeeper create failed after " + retryCounter.getMaxRetries() + " retries"));
                        throw e;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    private String findPreviousSequentialNode(String path) throws KeeperException, InterruptedException {
        int lastSlashIdx = path.lastIndexOf(47);
        assert (lastSlashIdx != -1);
        String parent = path.substring(0, lastSlashIdx);
        String nodePrefix = path.substring(lastSlashIdx + 1);
        List nodes = this.zk.getChildren(parent, false);
        List<String> matching = RecoverableZooKeeper.filterByPrefix(nodes, nodePrefix);
        for (String node : matching) {
            String nodePath = parent + "/" + node;
            Stat stat = this.zk.exists(nodePath, false);
            if (stat == null) continue;
            return nodePath;
        }
        return null;
    }

    public byte[] removeMetaData(byte[] data) {
        if (data == null || data.length == 0) {
            return data;
        }
        byte magic = data[0];
        if (magic != -1) {
            return data;
        }
        int idLength = Bytes.toInt(data, 1);
        int dataLength = data.length - 1 - 4 - idLength;
        int dataOffset = 5 + idLength;
        byte[] newData = new byte[dataLength];
        System.arraycopy(data, dataOffset, newData, 0, dataLength);
        return newData;
    }

    private byte[] appendMetaData(byte[] data) {
        if (data == null || data.length == 0) {
            return data;
        }
        byte[] newData = new byte[5 + this.id.length + data.length];
        int pos = 0;
        pos = Bytes.putByte(newData, pos, (byte)-1);
        pos = Bytes.putInt(newData, pos, this.id.length);
        pos = Bytes.putBytes(newData, pos, this.id, 0, this.id.length);
        pos = Bytes.putBytes(newData, pos, data, 0, data.length);
        return newData;
    }

    public long getSessionId() {
        return this.zk.getSessionId();
    }

    public void close() throws InterruptedException {
        this.zk.close();
    }

    public ZooKeeper.States getState() {
        return this.zk.getState();
    }

    public ZooKeeper getZooKeeper() {
        return this.zk;
    }

    public byte[] getSessionPasswd() {
        return this.zk.getSessionPasswd();
    }

    public void sync(String path, AsyncCallback.VoidCallback cb, Object ctx) {
        this.zk.sync(path, null, null);
    }

    private static List<String> filterByPrefix(List<String> nodes, String ... prefixes) {
        ArrayList<String> lockChildren = new ArrayList<String>();
        block0: for (String child : nodes) {
            for (String prefix : prefixes) {
                if (!child.startsWith(prefix)) continue;
                lockChildren.add(child);
                continue block0;
            }
        }
        return lockChildren;
    }
}

