/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.io.HbaseMapWritable;
import org.apache.hadoop.hbase.io.HeapSize;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BoundedRangeFileInputStream;
import org.apache.hadoop.hbase.io.hfile.Compression;
import org.apache.hadoop.hbase.io.hfile.HFileScanner;
import org.apache.hadoop.hbase.regionserver.TimeRangeTracker;
import org.apache.hadoop.hbase.util.ByteBloomFilter;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.hbase.util.CompressionTest;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.Writables;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.RawComparator;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.compress.Compressor;
import org.apache.hadoop.io.compress.Decompressor;

public class HFile {
    static final Log LOG = LogFactory.getLog(HFile.class);
    static final byte[] DATABLOCKMAGIC = new byte[]{68, 65, 84, 65, 66, 76, 75, 42};
    static final byte[] INDEXBLOCKMAGIC = new byte[]{73, 68, 88, 66, 76, 75, 41, 43};
    static final byte[] METABLOCKMAGIC = new byte[]{77, 69, 84, 65, 66, 76, 75, 99};
    static final byte[] TRAILERBLOCKMAGIC = new byte[]{84, 82, 65, 66, 76, 75, 34, 36};
    public static final int MAXIMUM_KEY_LENGTH = Integer.MAX_VALUE;
    public static final int DEFAULT_BLOCKSIZE = 65536;
    public static final Compression.Algorithm DEFAULT_COMPRESSION_ALGORITHM = Compression.Algorithm.NONE;
    public static final String DEFAULT_COMPRESSION = DEFAULT_COMPRESSION_ALGORITHM.getName();
    private static volatile long readOps;
    private static volatile long readTime;
    private static volatile long writeOps;
    private static volatile long writeTime;

    public static final long getReadOps() {
        long ret = readOps;
        readOps = 0L;
        return ret;
    }

    public static final long getReadTime() {
        long ret = readTime;
        readTime = 0L;
        return ret;
    }

    public static final long getWriteOps() {
        long ret = writeOps;
        writeOps = 0L;
        return ret;
    }

    public static final long getWriteTime() {
        long ret = writeTime;
        writeTime = 0L;
        return ret;
    }

    public static boolean isReservedFileInfoKey(byte[] key) {
        return Bytes.startsWith(key, FileInfo.RESERVED_PREFIX_BYTES);
    }

    public static String[] getSupportedCompressionAlgorithms() {
        return Compression.getSupportedAlgorithms();
    }

    static int longToInt(long l) {
        return (int)(l & 0xFFFFFFFFL);
    }

    static List<Path> getStoreFiles(FileSystem fs, Path regionDir) throws IOException {
        FileStatus[] familyDirs;
        ArrayList<Path> res = new ArrayList<Path>();
        FSUtils.DirFilter dirFilter = new FSUtils.DirFilter(fs);
        for (FileStatus dir : familyDirs = fs.listStatus(regionDir, (PathFilter)dirFilter)) {
            FileStatus[] files;
            for (FileStatus file : files = fs.listStatus(dir.getPath())) {
                if (file.isDir()) continue;
                res.add(file.getPath());
            }
        }
        return res;
    }

    public static void main(String[] args) throws IOException {
        try {
            Options options = new Options();
            options.addOption("v", "verbose", false, "Verbose output; emits file and meta data delimiters");
            options.addOption("p", "printkv", false, "Print key/value pairs");
            options.addOption("e", "printkey", false, "Print keys");
            options.addOption("m", "printmeta", false, "Print meta data of file");
            options.addOption("b", "printblocks", false, "Print block index meta data");
            options.addOption("k", "checkrow", false, "Enable row order check; looks for out-of-order keys");
            options.addOption("a", "checkfamily", false, "Enable family check");
            options.addOption("f", "file", true, "File to scan. Pass full-path; e.g. hdfs://a:9000/hbase/.META./12/34");
            options.addOption("r", "region", true, "Region to scan. Pass region name; e.g. '.META.,,1'");
            if (args.length == 0) {
                HelpFormatter formatter = new HelpFormatter();
                formatter.printHelp("HFile ", options, true);
                System.exit(-1);
            }
            PosixParser parser = new PosixParser();
            CommandLine cmd = parser.parse(options, args);
            boolean verbose = cmd.hasOption("v");
            boolean printValue = cmd.hasOption("p");
            boolean printKey = cmd.hasOption("e") || printValue;
            boolean printMeta = cmd.hasOption("m");
            boolean printBlocks = cmd.hasOption("b");
            boolean checkRow = cmd.hasOption("k");
            boolean checkFamily = cmd.hasOption("a");
            Configuration conf = HBaseConfiguration.create();
            conf.set("fs.defaultFS", conf.get("hbase.rootdir"));
            conf.set("fs.default.name", conf.get("hbase.rootdir"));
            ArrayList<Path> files = new ArrayList<Path>();
            if (cmd.hasOption("f")) {
                files.add(new Path(cmd.getOptionValue("f")));
            }
            if (cmd.hasOption("r")) {
                String regionName = cmd.getOptionValue("r");
                byte[] rn = Bytes.toBytes(regionName);
                byte[][] hri = HRegionInfo.parseRegionName(rn);
                Path rootDir = FSUtils.getRootDir(conf);
                Path tableDir = new Path(rootDir, Bytes.toString(hri[0]));
                String enc = HRegionInfo.encodeRegionName(rn);
                Path regionDir = new Path(tableDir, enc);
                if (verbose) {
                    System.out.println("region dir -> " + regionDir);
                }
                List<Path> regionFiles = HFile.getStoreFiles(FileSystem.get((Configuration)conf), regionDir);
                if (verbose) {
                    System.out.println("Number of region files found -> " + regionFiles.size());
                }
                if (verbose) {
                    int i = 1;
                    for (Path p : regionFiles) {
                        if (!verbose) continue;
                        System.out.println("Found file[" + i++ + "] -> " + p);
                    }
                }
                files.addAll(regionFiles);
            }
            for (Path file : files) {
                FileSystem fs;
                if (verbose) {
                    System.out.println("Scanning -> " + file);
                }
                if (!(fs = file.getFileSystem(conf)).exists(file)) {
                    System.err.println("ERROR, file doesnt exist: " + file);
                    continue;
                }
                Reader reader = new Reader(fs, file, null, false);
                Map<byte[], byte[]> fileInfo = reader.loadFileInfo();
                int count = 0;
                if (verbose || printKey || checkRow || checkFamily) {
                    HFileScanner scanner = reader.getScanner(false, false);
                    scanner.seekTo();
                    KeyValue pkv = null;
                    do {
                        KeyValue kv = scanner.getKeyValue();
                        if (printKey) {
                            System.out.print("K: " + kv);
                            if (printValue) {
                                System.out.print(" V: " + Bytes.toStringBinary(kv.getValue()));
                            }
                            System.out.println();
                        }
                        if (checkRow && pkv != null && Bytes.compareTo(pkv.getRow(), kv.getRow()) > 0) {
                            System.err.println("WARNING, previous row is greater then current row\n\tfilename -> " + file + "\n\tprevious -> " + Bytes.toStringBinary(pkv.getKey()) + "\n\tcurrent  -> " + Bytes.toStringBinary(kv.getKey()));
                        }
                        if (checkFamily) {
                            String fam = Bytes.toString(kv.getFamily());
                            if (!file.toString().contains(fam)) {
                                System.err.println("WARNING, filename does not match kv family,\n\tfilename -> " + file + "\n\tkeyvalue -> " + Bytes.toStringBinary(kv.getKey()));
                            }
                            if (pkv != null && Bytes.compareTo(pkv.getFamily(), kv.getFamily()) != 0) {
                                System.err.println("WARNING, previous kv has different family compared to current key\n\tfilename -> " + file + "\n\tprevious -> " + Bytes.toStringBinary(pkv.getKey()) + "\n\tcurrent  -> " + Bytes.toStringBinary(kv.getKey()));
                            }
                        }
                        pkv = kv;
                        ++count;
                    } while (scanner.next());
                }
                if (verbose || printKey) {
                    System.out.println("Scanned kv count -> " + count);
                }
                if (printMeta) {
                    System.out.println("Block index size as per heapsize: " + reader.indexSize());
                    System.out.println(reader.toString());
                    System.out.println(reader.getTrailerInfo());
                    System.out.println("Fileinfo:");
                    for (Map.Entry<byte[], byte[]> e : fileInfo.entrySet()) {
                        System.out.print(Bytes.toString(e.getKey()) + " = ");
                        if (Bytes.compareTo(e.getKey(), Bytes.toBytes("MAX_SEQ_ID_KEY")) == 0) {
                            long seqid = Bytes.toLong(e.getValue());
                            System.out.println(seqid);
                            continue;
                        }
                        if (Bytes.compareTo(e.getKey(), Bytes.toBytes("TIMERANGE")) == 0) {
                            TimeRangeTracker timeRangeTracker = new TimeRangeTracker();
                            Writables.copyWritable(e.getValue(), (Writable)timeRangeTracker);
                            System.out.println(timeRangeTracker.getMinimumTimestamp() + "...." + timeRangeTracker.getMaximumTimestamp());
                            continue;
                        }
                        if (Bytes.compareTo(e.getKey(), FileInfo.AVG_KEY_LEN) == 0 || Bytes.compareTo(e.getKey(), FileInfo.AVG_VALUE_LEN) == 0) {
                            System.out.println(Bytes.toInt(e.getValue()));
                            continue;
                        }
                        System.out.println(Bytes.toStringBinary(e.getValue()));
                    }
                    ByteBuffer b = reader.getMetaBlock("BLOOM_FILTER_META", false);
                    if (b != null) {
                        ByteBloomFilter bloomFilter = new ByteBloomFilter(b);
                        System.out.println("BloomSize: " + bloomFilter.getByteSize());
                        System.out.println("No of Keys in bloom: " + bloomFilter.getKeyCount());
                        System.out.println("Max Keys for bloom: " + bloomFilter.getMaxKeys());
                    } else {
                        System.out.println("Could not get bloom data from meta block");
                    }
                }
                if (printBlocks) {
                    System.out.println("Block Index:");
                    System.out.println(reader.blockIndex);
                }
                reader.close();
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    static class FileInfo
    extends HbaseMapWritable<byte[], byte[]> {
        static final String RESERVED_PREFIX = "hfile.";
        static final byte[] RESERVED_PREFIX_BYTES = Bytes.toBytes("hfile.");
        static final byte[] LASTKEY = Bytes.toBytes("hfile.LASTKEY");
        static final byte[] AVG_KEY_LEN = Bytes.toBytes("hfile.AVG_KEY_LEN");
        static final byte[] AVG_VALUE_LEN = Bytes.toBytes("hfile.AVG_VALUE_LEN");
        static final byte[] COMPARATOR = Bytes.toBytes("hfile.COMPARATOR");

        FileInfo() {
        }
    }

    static class BlockIndex
    implements HeapSize {
        int count = 0;
        byte[][] blockKeys;
        long[] blockOffsets;
        int[] blockDataSizes;
        int size = 0;
        final RawComparator<byte[]> comparator;

        private BlockIndex() {
            this(null);
        }

        BlockIndex(RawComparator<byte[]> c) {
            this.comparator = c;
            this.size += 32;
        }

        boolean isEmpty() {
            return this.blockKeys.length <= 0;
        }

        void add(byte[] key, long offset, int dataSize) {
            this.blockOffsets[this.count] = offset;
            this.blockKeys[this.count] = key;
            this.blockDataSizes[this.count] = dataSize;
            ++this.count;
            this.size += 8 + key.length;
        }

        int blockContainingKey(byte[] key, int offset, int length) {
            int pos = Bytes.binarySearch(this.blockKeys, key, offset, length, this.comparator);
            if (pos < 0) {
                ++pos;
                if ((pos *= -1) == 0) {
                    return -1;
                }
                return --pos;
            }
            return pos;
        }

        byte[] midkey() throws IOException {
            int pos = (this.count - 1) / 2;
            if (pos < 0) {
                throw new IOException("HFile empty");
            }
            return this.blockKeys[pos];
        }

        static long writeIndex(FSDataOutputStream o, List<byte[]> keys, List<Long> offsets, List<Integer> sizes) throws IOException {
            long pos = o.getPos();
            if (keys.size() > 0) {
                o.write(INDEXBLOCKMAGIC);
                for (int i = 0; i < keys.size(); ++i) {
                    o.writeLong(offsets.get(i).longValue());
                    o.writeInt(sizes.get(i).intValue());
                    byte[] key = keys.get(i);
                    Bytes.writeByteArray((DataOutput)o, key);
                }
            }
            return pos;
        }

        static BlockIndex readIndex(RawComparator<byte[]> c, DataInputStream in, int indexSize) throws IOException {
            BlockIndex bi = new BlockIndex(c);
            bi.blockOffsets = new long[indexSize];
            bi.blockKeys = new byte[indexSize][];
            bi.blockDataSizes = new int[indexSize];
            if (indexSize > 0) {
                byte[] magic = new byte[INDEXBLOCKMAGIC.length];
                in.readFully(magic);
                if (!Arrays.equals(magic, INDEXBLOCKMAGIC)) {
                    throw new IOException("Index block magic is wrong: " + Arrays.toString(magic));
                }
                for (int i = 0; i < indexSize; ++i) {
                    long offset = in.readLong();
                    int dataSize = in.readInt();
                    byte[] key = Bytes.readByteArray(in);
                    bi.add(key, offset, dataSize);
                }
            }
            return bi;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("size=" + this.count).append("\n");
            for (int i = 0; i < this.count; ++i) {
                sb.append("key=").append(KeyValue.keyToString(this.blockKeys[i])).append("\n  offset=").append(this.blockOffsets[i]).append(", dataSize=" + this.blockDataSizes[i]).append("\n");
            }
            return sb.toString();
        }

        @Override
        public long heapSize() {
            long heapsize = ClassSize.align(ClassSize.OBJECT + 8 + 4 * ClassSize.REFERENCE);
            if (this.blockKeys != null) {
                heapsize += (long)ClassSize.align(ClassSize.ARRAY + this.blockKeys.length * ClassSize.REFERENCE);
                for (byte[] bs : this.blockKeys) {
                    heapsize += (long)ClassSize.align(ClassSize.ARRAY + bs.length);
                }
            }
            if (this.blockOffsets != null) {
                heapsize += (long)ClassSize.align(ClassSize.ARRAY + this.blockOffsets.length * 8);
            }
            if (this.blockDataSizes != null) {
                heapsize += (long)ClassSize.align(ClassSize.ARRAY + this.blockDataSizes.length * 4);
            }
            return ClassSize.align(heapsize);
        }
    }

    private static class FixedFileTrailer {
        long fileinfoOffset;
        long dataIndexOffset;
        int dataIndexCount;
        long metaIndexOffset;
        int metaIndexCount;
        long totalUncompressedBytes;
        int entryCount;
        int compressionCodec;
        int version = 1;

        FixedFileTrailer() {
        }

        static int trailerSize() {
            return 52 + TRAILERBLOCKMAGIC.length;
        }

        void serialize(DataOutputStream outputStream) throws IOException {
            outputStream.write(TRAILERBLOCKMAGIC);
            outputStream.writeLong(this.fileinfoOffset);
            outputStream.writeLong(this.dataIndexOffset);
            outputStream.writeInt(this.dataIndexCount);
            outputStream.writeLong(this.metaIndexOffset);
            outputStream.writeInt(this.metaIndexCount);
            outputStream.writeLong(this.totalUncompressedBytes);
            outputStream.writeInt(this.entryCount);
            outputStream.writeInt(this.compressionCodec);
            outputStream.writeInt(this.version);
        }

        void deserialize(DataInputStream inputStream) throws IOException {
            byte[] header = new byte[TRAILERBLOCKMAGIC.length];
            inputStream.readFully(header);
            if (!Arrays.equals(header, TRAILERBLOCKMAGIC)) {
                throw new IOException("Trailer 'header' is wrong; does the trailer size match content?");
            }
            this.fileinfoOffset = inputStream.readLong();
            this.dataIndexOffset = inputStream.readLong();
            this.dataIndexCount = inputStream.readInt();
            this.metaIndexOffset = inputStream.readLong();
            this.metaIndexCount = inputStream.readInt();
            this.totalUncompressedBytes = inputStream.readLong();
            this.entryCount = inputStream.readInt();
            this.compressionCodec = inputStream.readInt();
            this.version = inputStream.readInt();
            if (this.version != 1) {
                throw new IOException("Wrong version: " + this.version);
            }
        }

        public String toString() {
            return "fileinfoOffset=" + this.fileinfoOffset + ", dataIndexOffset=" + this.dataIndexOffset + ", dataIndexCount=" + this.dataIndexCount + ", metaIndexOffset=" + this.metaIndexOffset + ", metaIndexCount=" + this.metaIndexCount + ", totalBytes=" + this.totalUncompressedBytes + ", entryCount=" + this.entryCount + ", version=" + this.version;
        }
    }

    public static class Reader
    implements Closeable {
        private FSDataInputStream istream;
        private boolean closeIStream;
        BlockIndex blockIndex;
        private BlockIndex metaIndex;
        FixedFileTrailer trailer;
        private volatile boolean fileInfoLoaded = false;
        private Compression.Algorithm compressAlgo;
        private byte[] lastkey = null;
        private int avgKeyLen = -1;
        private int avgValueLen = -1;
        RawComparator<byte[]> comparator;
        private final long fileSize;
        private final BlockCache cache;
        public int cacheHits = 0;
        public int blockLoads = 0;
        public int metaLoads = 0;
        private boolean inMemory = false;
        protected String name;

        public Reader(FileSystem fs, Path path, BlockCache cache, boolean inMemory) throws IOException {
            this(fs.open(path), fs.getFileStatus(path).getLen(), cache, inMemory);
            this.closeIStream = true;
            this.name = path.toString();
        }

        public Reader(FSDataInputStream fsdis, long size, BlockCache cache, boolean inMemory) {
            this.cache = cache;
            this.fileSize = size;
            this.istream = fsdis;
            this.closeIStream = false;
            this.name = this.istream == null ? "" : this.istream.toString();
            this.inMemory = inMemory;
        }

        public String toString() {
            return "reader=" + this.name + (!this.isFileInfoLoaded() ? "" : ", compression=" + this.compressAlgo.getName() + ", inMemory=" + this.inMemory + ", firstKey=" + this.toStringFirstKey() + ", lastKey=" + this.toStringLastKey()) + ", avgKeyLen=" + this.avgKeyLen + ", avgValueLen=" + this.avgValueLen + ", entries=" + this.trailer.entryCount + ", length=" + this.fileSize;
        }

        protected String toStringFirstKey() {
            return KeyValue.keyToString(this.getFirstKey());
        }

        protected String toStringLastKey() {
            return KeyValue.keyToString(this.getLastKey());
        }

        public long length() {
            return this.fileSize;
        }

        public boolean inMemory() {
            return this.inMemory;
        }

        private byte[] readAllIndex(FSDataInputStream in, long indexOffset, int indexSize) throws IOException {
            byte[] allIndex = new byte[indexSize];
            in.seek(indexOffset);
            IOUtils.readFully((InputStream)in, (byte[])allIndex, (int)0, (int)allIndex.length);
            return allIndex;
        }

        public Map<byte[], byte[]> loadFileInfo() throws IOException {
            this.trailer = this.readTrailer();
            this.istream.seek(this.trailer.fileinfoOffset);
            FileInfo fi = new FileInfo();
            fi.readFields((DataInput)this.istream);
            this.lastkey = (byte[])fi.get(FileInfo.LASTKEY);
            this.avgKeyLen = Bytes.toInt((byte[])fi.get(FileInfo.AVG_KEY_LEN));
            this.avgValueLen = Bytes.toInt((byte[])fi.get(FileInfo.AVG_VALUE_LEN));
            String clazzName = Bytes.toString((byte[])fi.get(FileInfo.COMPARATOR));
            this.comparator = this.getComparator(clazzName);
            int allIndexSize = (int)(this.fileSize - this.trailer.dataIndexOffset - (long)FixedFileTrailer.trailerSize());
            byte[] dataAndMetaIndex = this.readAllIndex(this.istream, this.trailer.dataIndexOffset, allIndexSize);
            ByteArrayInputStream bis = new ByteArrayInputStream(dataAndMetaIndex);
            DataInputStream dis = new DataInputStream(bis);
            this.blockIndex = BlockIndex.readIndex(this.comparator, dis, this.trailer.dataIndexCount);
            if (this.trailer.metaIndexCount > 0) {
                this.metaIndex = BlockIndex.readIndex(Bytes.BYTES_RAWCOMPARATOR, dis, this.trailer.metaIndexCount);
            }
            this.fileInfoLoaded = true;
            if (null != dis) {
                dis.close();
            }
            return fi;
        }

        boolean isFileInfoLoaded() {
            return this.fileInfoLoaded;
        }

        private RawComparator<byte[]> getComparator(String clazzName) throws IOException {
            if (clazzName == null || clazzName.length() == 0) {
                return null;
            }
            try {
                return (RawComparator)Class.forName(clazzName).newInstance();
            }
            catch (InstantiationException e) {
                throw new IOException(e);
            }
            catch (IllegalAccessException e) {
                throw new IOException(e);
            }
            catch (ClassNotFoundException e) {
                throw new IOException(e);
            }
        }

        private FixedFileTrailer readTrailer() throws IOException {
            FixedFileTrailer fft = new FixedFileTrailer();
            long seekPoint = this.fileSize - (long)FixedFileTrailer.trailerSize();
            this.istream.seek(seekPoint);
            fft.deserialize((DataInputStream)this.istream);
            this.compressAlgo = Compression.Algorithm.values()[fft.compressionCodec];
            CompressionTest.testCompression(this.compressAlgo);
            return fft;
        }

        public HFileScanner getScanner(boolean cacheBlocks, boolean pread) {
            return new Scanner(this, cacheBlocks, pread);
        }

        protected int blockContainingKey(byte[] key, int offset, int length) {
            if (this.blockIndex == null) {
                throw new RuntimeException("Block index not loaded");
            }
            return this.blockIndex.blockContainingKey(key, offset, length);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public ByteBuffer getMetaBlock(String metaBlockName, boolean cacheBlock) throws IOException {
            if (this.trailer.metaIndexCount == 0) {
                return null;
            }
            if (this.metaIndex == null) {
                throw new IOException("Meta index not loaded");
            }
            byte[] mbname = Bytes.toBytes(metaBlockName);
            int block = this.metaIndex.blockContainingKey(mbname, 0, mbname.length);
            if (block == -1) {
                return null;
            }
            long blockSize = block == this.metaIndex.count - 1 ? this.trailer.fileinfoOffset - this.metaIndex.blockOffsets[block] : this.metaIndex.blockOffsets[block + 1] - this.metaIndex.blockOffsets[block];
            long now = System.currentTimeMillis();
            byte[] byArray = this.metaIndex.blockKeys[block];
            synchronized (byArray) {
                ByteBuffer cachedBuf;
                ++this.metaLoads;
                if (this.cache != null && (cachedBuf = this.cache.getBlock(this.name + "meta" + block, cacheBlock)) != null) {
                    ++this.cacheHits;
                    return cachedBuf.duplicate();
                }
                ByteBuffer buf = this.decompress(this.metaIndex.blockOffsets[block], HFile.longToInt(blockSize), this.metaIndex.blockDataSizes[block], true);
                byte[] magic = new byte[METABLOCKMAGIC.length];
                buf.get(magic, 0, magic.length);
                if (!Arrays.equals(magic, METABLOCKMAGIC)) {
                    throw new IOException("Meta magic is bad in block " + block);
                }
                buf = buf.slice();
                readTime += System.currentTimeMillis() - now;
                readOps++;
                if (cacheBlock && this.cache != null) {
                    this.cache.cacheBlock(this.name + "meta" + block, buf.duplicate(), this.inMemory);
                }
                return buf;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        ByteBuffer readBlock(int block, boolean cacheBlock, boolean pread) throws IOException {
            if (this.blockIndex == null) {
                throw new IOException("Block index not loaded");
            }
            if (block < 0 || block >= this.blockIndex.count) {
                throw new IOException("Requested block is out of range: " + block + ", max: " + this.blockIndex.count);
            }
            byte[] byArray = this.blockIndex.blockKeys[block];
            synchronized (byArray) {
                long onDiskBlockSize;
                ByteBuffer cachedBuf;
                ++this.blockLoads;
                if (this.cache != null && (cachedBuf = this.cache.getBlock(this.name + block, cacheBlock)) != null) {
                    ++this.cacheHits;
                    return cachedBuf.duplicate();
                }
                long now = System.currentTimeMillis();
                if (block == this.blockIndex.count - 1) {
                    long offset = this.metaIndex != null ? this.metaIndex.blockOffsets[0] : this.trailer.fileinfoOffset;
                    onDiskBlockSize = offset - this.blockIndex.blockOffsets[block];
                } else {
                    onDiskBlockSize = this.blockIndex.blockOffsets[block + 1] - this.blockIndex.blockOffsets[block];
                }
                ByteBuffer buf = this.decompress(this.blockIndex.blockOffsets[block], HFile.longToInt(onDiskBlockSize), this.blockIndex.blockDataSizes[block], pread);
                byte[] magic = new byte[DATABLOCKMAGIC.length];
                buf.get(magic, 0, magic.length);
                if (!Arrays.equals(magic, DATABLOCKMAGIC)) {
                    throw new IOException("Data magic is bad in block " + block);
                }
                buf = buf.slice();
                readTime += System.currentTimeMillis() - now;
                readOps++;
                if (cacheBlock && this.cache != null) {
                    this.cache.cacheBlock(this.name + block, buf.duplicate(), this.inMemory);
                }
                return buf;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private ByteBuffer decompress(long offset, int compressedSize, int decompressedSize, boolean pread) throws IOException {
            Decompressor decompressor = null;
            ByteBuffer buf = null;
            try {
                decompressor = this.compressAlgo.getDecompressor();
                InputStream is = this.compressAlgo.createDecompressionStream(new BufferedInputStream(new BoundedRangeFileInputStream(this.istream, offset, compressedSize, pread), Math.min(65536, compressedSize)), decompressor, 0);
                buf = ByteBuffer.allocate(decompressedSize);
                IOUtils.readFully((InputStream)is, (byte[])buf.array(), (int)0, (int)buf.capacity());
                is.close();
            }
            finally {
                if (null != decompressor) {
                    this.compressAlgo.returnDecompressor(decompressor);
                }
            }
            return buf;
        }

        public byte[] getFirstKey() {
            if (this.blockIndex == null) {
                throw new RuntimeException("Block index not loaded");
            }
            return this.blockIndex.isEmpty() ? null : this.blockIndex.blockKeys[0];
        }

        public byte[] getFirstRowKey() {
            byte[] firstKey = this.getFirstKey();
            if (firstKey == null) {
                return null;
            }
            return KeyValue.createKeyValueFromKey(firstKey).getRow();
        }

        public int getEntries() {
            if (!this.isFileInfoLoaded()) {
                throw new RuntimeException("File info not loaded");
            }
            return this.trailer.entryCount;
        }

        public byte[] getLastKey() {
            if (!this.isFileInfoLoaded()) {
                throw new RuntimeException("Load file info first");
            }
            return this.blockIndex.isEmpty() ? null : this.lastkey;
        }

        public byte[] getLastRowKey() {
            byte[] lastKey = this.getLastKey();
            if (lastKey == null) {
                return null;
            }
            return KeyValue.createKeyValueFromKey(lastKey).getRow();
        }

        public int getFilterEntries() {
            return this.getEntries();
        }

        public RawComparator<byte[]> getComparator() {
            return this.comparator;
        }

        public Compression.Algorithm getCompressionAlgorithm() {
            return this.compressAlgo;
        }

        public long indexSize() {
            return (this.blockIndex != null ? this.blockIndex.heapSize() : 0L) + (this.metaIndex != null ? this.metaIndex.heapSize() : 0L);
        }

        public byte[] midkey() throws IOException {
            if (!this.isFileInfoLoaded() || this.blockIndex.isEmpty()) {
                return null;
            }
            return this.blockIndex.midkey();
        }

        @Override
        public void close() throws IOException {
            if (this.closeIStream && this.istream != null) {
                this.istream.close();
                this.istream = null;
            }
        }

        public String getName() {
            return this.name;
        }

        public String getTrailerInfo() {
            return this.trailer.toString();
        }

        protected static class Scanner
        implements HFileScanner {
            private final Reader reader;
            private ByteBuffer block;
            private int currBlock;
            private final boolean cacheBlocks;
            private final boolean pread;
            private int currKeyLen = 0;
            private int currValueLen = 0;
            public int blockFetches = 0;

            public Scanner(Reader r, boolean cacheBlocks, boolean pread) {
                this.reader = r;
                this.cacheBlocks = cacheBlocks;
                this.pread = pread;
            }

            @Override
            public KeyValue getKeyValue() {
                if (this.block == null) {
                    return null;
                }
                return new KeyValue(this.block.array(), this.block.arrayOffset() + this.block.position() - 8, this.currKeyLen + this.currValueLen + 8);
            }

            @Override
            public ByteBuffer getKey() {
                if (this.block == null || this.currKeyLen == 0) {
                    throw new RuntimeException("you need to seekTo() before calling getKey()");
                }
                ByteBuffer keyBuff = this.block.slice();
                keyBuff.limit(this.currKeyLen);
                keyBuff.rewind();
                return keyBuff;
            }

            @Override
            public ByteBuffer getValue() {
                if (this.block == null || this.currKeyLen == 0) {
                    throw new RuntimeException("you need to seekTo() before calling getValue()");
                }
                ByteBuffer valueBuff = this.block.slice();
                valueBuff.position(this.currKeyLen);
                valueBuff = valueBuff.slice();
                valueBuff.limit(this.currValueLen);
                valueBuff.rewind();
                return valueBuff;
            }

            @Override
            public boolean next() throws IOException {
                if (this.block == null) {
                    throw new IOException("Next called on non-seeked scanner");
                }
                this.block.position(this.block.position() + this.currKeyLen + this.currValueLen);
                if (this.block.remaining() <= 0) {
                    ++this.currBlock;
                    if (this.currBlock >= this.reader.blockIndex.count) {
                        this.currBlock = 0;
                        this.block = null;
                        return false;
                    }
                    this.block = this.reader.readBlock(this.currBlock, this.cacheBlocks, this.pread);
                    this.currKeyLen = Bytes.toInt(this.block.array(), this.block.arrayOffset() + this.block.position(), 4);
                    this.currValueLen = Bytes.toInt(this.block.array(), this.block.arrayOffset() + this.block.position() + 4, 4);
                    this.block.position(this.block.position() + 8);
                    ++this.blockFetches;
                    return true;
                }
                this.currKeyLen = Bytes.toInt(this.block.array(), this.block.arrayOffset() + this.block.position(), 4);
                this.currValueLen = Bytes.toInt(this.block.array(), this.block.arrayOffset() + this.block.position() + 4, 4);
                this.block.position(this.block.position() + 8);
                return true;
            }

            @Override
            public int seekTo(byte[] key) throws IOException {
                return this.seekTo(key, 0, key.length);
            }

            @Override
            public int seekTo(byte[] key, int offset, int length) throws IOException {
                int b = this.reader.blockContainingKey(key, offset, length);
                if (b < 0) {
                    return -1;
                }
                this.loadBlock(b, true);
                return this.blockSeek(key, offset, length, false);
            }

            @Override
            public int reseekTo(byte[] key) throws IOException {
                return this.reseekTo(key, 0, key.length);
            }

            @Override
            public int reseekTo(byte[] key, int offset, int length) throws IOException {
                ByteBuffer bb;
                int compared;
                if (this.block != null && this.currKeyLen != 0 && (compared = this.reader.comparator.compare(key, offset, length, (bb = this.getKey()).array(), bb.arrayOffset(), bb.limit())) < 1) {
                    return compared;
                }
                int b = this.reader.blockContainingKey(key, offset, length);
                if (b < 0) {
                    return -1;
                }
                this.loadBlock(b, false);
                return this.blockSeek(key, offset, length, false);
            }

            private int blockSeek(byte[] key, int offset, int length, boolean seekBefore) {
                int lastLen = 0;
                do {
                    int klen = this.block.getInt();
                    int vlen = this.block.getInt();
                    int comp = this.reader.comparator.compare(key, offset, length, this.block.array(), this.block.arrayOffset() + this.block.position(), klen);
                    if (comp == 0) {
                        if (seekBefore) {
                            this.block.position(this.block.position() - lastLen - 16);
                            this.currKeyLen = this.block.getInt();
                            this.currValueLen = this.block.getInt();
                            return 1;
                        }
                        this.currKeyLen = klen;
                        this.currValueLen = vlen;
                        return 0;
                    }
                    if (comp < 0) {
                        this.block.position(this.block.position() - lastLen - 16);
                        this.currKeyLen = this.block.getInt();
                        this.currValueLen = this.block.getInt();
                        return 1;
                    }
                    this.block.position(this.block.position() + klen + vlen);
                    lastLen = klen + vlen;
                } while (this.block.remaining() > 0);
                this.block.position(this.block.position() - lastLen - 8);
                this.currKeyLen = this.block.getInt();
                this.currValueLen = this.block.getInt();
                return 1;
            }

            @Override
            public boolean seekBefore(byte[] key) throws IOException {
                return this.seekBefore(key, 0, key.length);
            }

            @Override
            public boolean seekBefore(byte[] key, int offset, int length) throws IOException {
                int b = this.reader.blockContainingKey(key, offset, length);
                if (b < 0) {
                    return false;
                }
                if (this.reader.comparator.compare(this.reader.blockIndex.blockKeys[b], 0, this.reader.blockIndex.blockKeys[b].length, key, offset, length) == 0) {
                    if (b == 0) {
                        return false;
                    }
                    --b;
                }
                this.loadBlock(b, true);
                this.blockSeek(key, offset, length, true);
                return true;
            }

            @Override
            public String getKeyString() {
                return Bytes.toStringBinary(this.block.array(), this.block.arrayOffset() + this.block.position(), this.currKeyLen);
            }

            @Override
            public String getValueString() {
                return Bytes.toString(this.block.array(), this.block.arrayOffset() + this.block.position() + this.currKeyLen, this.currValueLen);
            }

            @Override
            public Reader getReader() {
                return this.reader;
            }

            @Override
            public boolean isSeeked() {
                return this.block != null;
            }

            @Override
            public boolean seekTo() throws IOException {
                if (this.reader.blockIndex.isEmpty()) {
                    return false;
                }
                if (this.block != null && this.currBlock == 0) {
                    this.block.rewind();
                    this.currKeyLen = this.block.getInt();
                    this.currValueLen = this.block.getInt();
                    return true;
                }
                this.currBlock = 0;
                this.block = this.reader.readBlock(this.currBlock, this.cacheBlocks, this.pread);
                this.currKeyLen = this.block.getInt();
                this.currValueLen = this.block.getInt();
                ++this.blockFetches;
                return true;
            }

            private void loadBlock(int bloc, boolean rewind) throws IOException {
                if (this.block == null) {
                    this.block = this.reader.readBlock(bloc, this.cacheBlocks, this.pread);
                    this.currBlock = bloc;
                    ++this.blockFetches;
                } else if (bloc != this.currBlock) {
                    this.block = this.reader.readBlock(bloc, this.cacheBlocks, this.pread);
                    this.currBlock = bloc;
                    ++this.blockFetches;
                } else if (rewind) {
                    this.block.rewind();
                } else {
                    this.block.position(this.block.position() - 8);
                }
            }

            public String toString() {
                return "HFileScanner for reader " + String.valueOf(this.reader);
            }
        }
    }

    public static class Writer
    implements Closeable {
        private FSDataOutputStream outputStream;
        private boolean closeOutputStream;
        protected String name;
        private long totalBytes = 0L;
        private int entryCount = 0;
        private long keylength = 0L;
        private long valuelength = 0L;
        private final RawComparator<byte[]> comparator;
        private DataOutputStream out;
        private int blocksize;
        private long blockBegin;
        private byte[] firstKey = null;
        private byte[] lastKeyBuffer = null;
        private int lastKeyOffset = -1;
        private int lastKeyLength = -1;
        ArrayList<byte[]> blockKeys = new ArrayList();
        ArrayList<Long> blockOffsets = new ArrayList();
        ArrayList<Integer> blockDataSizes = new ArrayList();
        private ArrayList<byte[]> metaNames = new ArrayList();
        private ArrayList<Writable> metaData = new ArrayList();
        private final Compression.Algorithm compressAlgo;
        private Compressor compressor;
        private FileInfo fileinfo = new FileInfo();
        private Path path = null;

        public Writer(FileSystem fs, Path path) throws IOException {
            this(fs, path, 65536, (Compression.Algorithm)null, null);
        }

        public Writer(FileSystem fs, Path path, int blocksize, String compress, KeyValue.KeyComparator comparator) throws IOException {
            this(fs, path, blocksize, compress == null ? DEFAULT_COMPRESSION_ALGORITHM : Compression.getCompressionAlgorithmByName(compress), comparator);
        }

        public Writer(FileSystem fs, Path path, int blocksize, Compression.Algorithm compress, KeyValue.KeyComparator comparator) throws IOException {
            this(fs.create(path), blocksize, compress, comparator);
            this.closeOutputStream = true;
            this.name = path.toString();
            this.path = path;
        }

        public Writer(FSDataOutputStream ostream, int blocksize, String compress, KeyValue.KeyComparator c) throws IOException {
            this(ostream, blocksize, Compression.getCompressionAlgorithmByName(compress), c);
        }

        public Writer(FSDataOutputStream ostream, int blocksize, Compression.Algorithm compress, KeyValue.KeyComparator c) throws IOException {
            this.outputStream = ostream;
            this.closeOutputStream = false;
            this.blocksize = blocksize;
            this.comparator = c == null ? Bytes.BYTES_RAWCOMPARATOR : c;
            this.name = this.outputStream.toString();
            this.compressAlgo = compress == null ? DEFAULT_COMPRESSION_ALGORITHM : compress;
        }

        private void checkBlockBoundary() throws IOException {
            if (this.out != null && this.out.size() < this.blocksize) {
                return;
            }
            this.finishBlock();
            this.newBlock();
        }

        private void finishBlock() throws IOException {
            if (this.out == null) {
                return;
            }
            long now = System.currentTimeMillis();
            int size = this.releaseCompressingStream(this.out);
            this.out = null;
            this.blockKeys.add(this.firstKey);
            this.blockOffsets.add(this.blockBegin);
            this.blockDataSizes.add(size);
            this.totalBytes += (long)size;
            writeTime += System.currentTimeMillis() - now;
            writeOps++;
        }

        private void newBlock() throws IOException {
            this.blockBegin = this.outputStream.getPos();
            this.out = this.getCompressingStream();
            this.out.write(DATABLOCKMAGIC);
            this.firstKey = null;
        }

        private DataOutputStream getCompressingStream() throws IOException {
            this.compressor = this.compressAlgo.getCompressor();
            OutputStream os = this.compressAlgo.createCompressionStream((OutputStream)this.outputStream, this.compressor, 0);
            return new DataOutputStream(os);
        }

        private int releaseCompressingStream(DataOutputStream dos) throws IOException {
            dos.flush();
            this.compressAlgo.returnCompressor(this.compressor);
            this.compressor = null;
            return dos.size();
        }

        public void appendMetaBlock(String metaBlockName, Writable content) {
            byte[] cur;
            int i;
            byte[] key = Bytes.toBytes(metaBlockName);
            for (i = 0; i < this.metaNames.size() && Bytes.BYTES_RAWCOMPARATOR.compare(cur = this.metaNames.get(i), 0, cur.length, key, 0, key.length) <= 0; ++i) {
            }
            this.metaNames.add(i, key);
            this.metaData.add(i, content);
        }

        public void appendFileInfo(byte[] k, byte[] v) throws IOException {
            Writer.appendFileInfo(this.fileinfo, k, v, true);
        }

        static FileInfo appendFileInfo(FileInfo fi, byte[] k, byte[] v, boolean checkPrefix) throws IOException {
            if (k == null || v == null) {
                throw new NullPointerException("Key nor value may be null");
            }
            if (checkPrefix && Bytes.startsWith(k, FileInfo.RESERVED_PREFIX_BYTES)) {
                throw new IOException("Keys with a hfile. are reserved");
            }
            fi.put(k, v);
            return fi;
        }

        public Path getPath() {
            return this.path;
        }

        public String toString() {
            return "writer=" + this.name + ", compression=" + this.compressAlgo.getName();
        }

        public void append(KeyValue kv) throws IOException {
            this.append(kv.getBuffer(), kv.getKeyOffset(), kv.getKeyLength(), kv.getBuffer(), kv.getValueOffset(), kv.getValueLength());
        }

        public void append(byte[] key, byte[] value) throws IOException {
            this.append(key, 0, key.length, value, 0, value.length);
        }

        private void append(byte[] key, int koffset, int klength, byte[] value, int voffset, int vlength) throws IOException {
            boolean dupKey = this.checkKey(key, koffset, klength);
            this.checkValue(value, voffset, vlength);
            if (!dupKey) {
                this.checkBlockBoundary();
            }
            this.out.writeInt(klength);
            this.keylength += (long)klength;
            this.out.writeInt(vlength);
            this.valuelength += (long)vlength;
            this.out.write(key, koffset, klength);
            this.out.write(value, voffset, vlength);
            if (this.firstKey == null) {
                this.firstKey = new byte[klength];
                System.arraycopy(key, koffset, this.firstKey, 0, klength);
            }
            this.lastKeyBuffer = key;
            this.lastKeyOffset = koffset;
            this.lastKeyLength = klength;
            ++this.entryCount;
        }

        private boolean checkKey(byte[] key, int offset, int length) throws IOException {
            boolean dupKey = false;
            if (key == null || length <= 0) {
                throw new IOException("Key cannot be null or empty");
            }
            if (length > Integer.MAX_VALUE) {
                throw new IOException("Key length " + length + " > " + Integer.MAX_VALUE);
            }
            if (this.lastKeyBuffer != null) {
                int keyComp = this.comparator.compare(this.lastKeyBuffer, this.lastKeyOffset, this.lastKeyLength, key, offset, length);
                if (keyComp > 0) {
                    throw new IOException("Added a key not lexically larger than previous key=" + Bytes.toStringBinary(key, offset, length) + ", lastkey=" + Bytes.toStringBinary(this.lastKeyBuffer, this.lastKeyOffset, this.lastKeyLength));
                }
                if (keyComp == 0) {
                    dupKey = true;
                }
            }
            return dupKey;
        }

        private void checkValue(byte[] value, int offset, int length) throws IOException {
            if (value == null) {
                throw new IOException("Value cannot be null");
            }
        }

        public long getTotalBytes() {
            return this.totalBytes;
        }

        @Override
        public void close() throws IOException {
            if (this.outputStream == null) {
                return;
            }
            this.finishBlock();
            FixedFileTrailer trailer = new FixedFileTrailer();
            ArrayList<Long> metaOffsets = null;
            ArrayList<Integer> metaDataSizes = null;
            if (this.metaNames.size() > 0) {
                metaOffsets = new ArrayList<Long>(this.metaNames.size());
                metaDataSizes = new ArrayList<Integer>(this.metaNames.size());
                for (int i = 0; i < this.metaNames.size(); ++i) {
                    long curPos = this.outputStream.getPos();
                    metaOffsets.add(curPos);
                    DataOutputStream dos = this.getCompressingStream();
                    dos.write(METABLOCKMAGIC);
                    this.metaData.get(i).write((DataOutput)dos);
                    int size = this.releaseCompressingStream(dos);
                    metaDataSizes.add(size);
                }
            }
            trailer.fileinfoOffset = this.writeFileInfo(this.outputStream);
            trailer.dataIndexOffset = BlockIndex.writeIndex(this.outputStream, this.blockKeys, this.blockOffsets, this.blockDataSizes);
            if (this.metaNames.size() > 0) {
                trailer.metaIndexOffset = BlockIndex.writeIndex(this.outputStream, this.metaNames, metaOffsets, metaDataSizes);
            }
            trailer.dataIndexCount = this.blockKeys.size();
            trailer.metaIndexCount = this.metaNames.size();
            trailer.totalUncompressedBytes = this.totalBytes;
            trailer.entryCount = this.entryCount;
            trailer.compressionCodec = this.compressAlgo.ordinal();
            trailer.serialize((DataOutputStream)this.outputStream);
            if (this.closeOutputStream) {
                this.outputStream.close();
                this.outputStream = null;
            }
        }

        private long writeFileInfo(FSDataOutputStream o) throws IOException {
            if (this.lastKeyBuffer != null) {
                byte[] b = new byte[this.lastKeyLength];
                System.arraycopy(this.lastKeyBuffer, this.lastKeyOffset, b, 0, this.lastKeyLength);
                Writer.appendFileInfo(this.fileinfo, FileInfo.LASTKEY, b, false);
            }
            int avgKeyLen = this.entryCount == 0 ? 0 : (int)(this.keylength / (long)this.entryCount);
            Writer.appendFileInfo(this.fileinfo, FileInfo.AVG_KEY_LEN, Bytes.toBytes(avgKeyLen), false);
            int avgValueLen = this.entryCount == 0 ? 0 : (int)(this.valuelength / (long)this.entryCount);
            Writer.appendFileInfo(this.fileinfo, FileInfo.AVG_VALUE_LEN, Bytes.toBytes(avgValueLen), false);
            Writer.appendFileInfo(this.fileinfo, FileInfo.COMPARATOR, Bytes.toBytes(this.comparator.getClass().getName()), false);
            long pos = o.getPos();
            this.fileinfo.write((DataOutput)o);
            return pos;
        }
    }
}

