/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.TreeSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSImageStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FSImageTransactionalStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FileJournalManager;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.util.MD5FileUtils;

public class NNStorageRetentionManager {
    private final int numCheckpointsToRetain;
    private final long numExtraEditsToRetain;
    private static final Log LOG = LogFactory.getLog(NNStorageRetentionManager.class);
    private final NNStorage storage;
    private final StoragePurger purger;
    private final FSEditLog editLog;

    public NNStorageRetentionManager(Configuration conf, NNStorage storage, FSEditLog editLog, StoragePurger purger) {
        this.numCheckpointsToRetain = conf.getInt("dfs.namenode.num.checkpoints.retained", 2);
        this.numExtraEditsToRetain = conf.getLong("dfs.namenode.num.extra.edits.retained", 1000000L);
        Preconditions.checkArgument((this.numCheckpointsToRetain > 0 ? 1 : 0) != 0, (Object)"Must retain at least one checkpoint");
        Preconditions.checkArgument((this.numExtraEditsToRetain >= 0L ? 1 : 0) != 0, (Object)"dfs.namenode.num.extra.edits.retained must not be negative");
        this.storage = storage;
        this.editLog = editLog;
        this.purger = purger;
    }

    public NNStorageRetentionManager(Configuration conf, NNStorage storage, FSEditLog editLog) {
        this(conf, storage, editLog, new DeletionStoragePurger());
    }

    public void purgeOldStorage() throws IOException {
        FSImageTransactionalStorageInspector inspector = new FSImageTransactionalStorageInspector();
        this.storage.inspectStorageDirs(inspector);
        long minImageTxId = this.getImageTxIdToRetain(inspector);
        this.purgeCheckpointsOlderThan(inspector, minImageTxId);
        long purgeLogsFrom = Math.max(0L, minImageTxId + 1L - this.numExtraEditsToRetain);
        this.editLog.purgeLogsOlderThan(purgeLogsFrom);
    }

    private void purgeCheckpointsOlderThan(FSImageTransactionalStorageInspector inspector, long minTxId) {
        for (FSImageStorageInspector.FSImageFile image : inspector.getFoundImages()) {
            if (image.getCheckpointTxId() >= minTxId) continue;
            LOG.info((Object)("Purging old image " + image));
            this.purger.purgeImage(image);
        }
    }

    private long getImageTxIdToRetain(FSImageTransactionalStorageInspector inspector) {
        List<FSImageStorageInspector.FSImageFile> images = inspector.getFoundImages();
        TreeSet imageTxIds = Sets.newTreeSet();
        for (FSImageStorageInspector.FSImageFile image : images) {
            imageTxIds.add(image.getCheckpointTxId());
        }
        ArrayList imageTxIdsList = Lists.newArrayList((Iterable)imageTxIds);
        if (imageTxIdsList.isEmpty()) {
            return 0L;
        }
        Collections.reverse(imageTxIdsList);
        int toRetain = Math.min(this.numCheckpointsToRetain, imageTxIdsList.size());
        long minTxId = (Long)imageTxIdsList.get(toRetain - 1);
        LOG.info((Object)("Going to retain " + toRetain + " images with txid >= " + minTxId));
        return minTxId;
    }

    static class DeletionStoragePurger
    implements StoragePurger {
        DeletionStoragePurger() {
        }

        @Override
        public void purgeLog(FileJournalManager.EditLogFile log) {
            DeletionStoragePurger.deleteOrWarn(log.getFile());
        }

        @Override
        public void purgeImage(FSImageStorageInspector.FSImageFile image) {
            DeletionStoragePurger.deleteOrWarn(image.getFile());
            DeletionStoragePurger.deleteOrWarn(MD5FileUtils.getDigestFileForFile(image.getFile()));
        }

        private static void deleteOrWarn(File file) {
            if (!file.delete()) {
                LOG.warn((Object)("Could not delete " + file));
            }
        }
    }

    static interface StoragePurger {
        public void purgeLog(FileJournalManager.EditLogFile var1);

        public void purgeImage(FSImageStorageInspector.FSImageFile var1);
    }
}

