package org.pentaho.gwt.widgets.client.colorpicker;

/**
 * Copyright (c) 2007, AurorisNET.
 * 
 * Everyone is permitted to copy and distribute verbatim copies of this license 
 * document, but changing it is not allowed.
 * 
 * Preamble
 * 
 * This license establishes the terms under which a given free software Package 
 * may be copied, modified, distributed, and/or redistributed. The intent is 
 * that the Copyright Holder maintains some artistic control over the 
 * development of that Package while still keeping the Package available as open
 * source and free software.
 * 
 * You are always permitted to make arrangements wholly outside of this license 
 * directly with the Copyright Holder of a given Package. If the terms of this 
 * license do not permit the full use that you propose to make of the Package, 
 * you should contact the Copyright Holder and seek a different licensing 
 * arrangement.
 * 
 * Definitions
 * 
 * "Copyright Holder" means the individual(s) or organization(s) named in the 
 * copyright notice for the entire Package.
 * 
 * "Contributor" means any party that has contributed code or other material to 
 * the Package, in accordance with the Copyright Holder's procedures.
 * 
 * "You" and "your" means any person who would like to copy, distribute, or 
 * modify the Package.
 * 
 * "Package" means the collection of files distributed by the Copyright Holder, 
 * and derivatives of that collection and/or of those files. A given Package may
 * consist of either the Standard Version, or a Modified Version.
 * 
 * "Distribute" means providing a copy of the Package or making it accessible to
 * anyone else, or in the case of a company or organization, to others outside 
 * of your company or organization.
 * 
 * "Distributor Fee" means any fee that you charge for Distributing this Package
 * or providing support for this Package to another party. It does not mean 
 * licensing fees.
 * 
 * "Standard Version" refers to the Package if it has not been modified, or has
 * been modified only in ways explicitly requested by the Copyright Holder.
 * 
 * "Modified Version" means the Package, if it has been changed, and such 
 * changes were not explicitly requested by the Copyright Holder.
 * 
 * "Original License" means this Artistic License as Distributed with the 
 * Standard Version of the Package, in its current version or as it may be 
 * modified by The Perl Foundation in the future.
 * 
 * "Source" form means the source code, documentation source, and configuration 
 * files for the Package.
 * 
 * "Compiled" form means the compiled bytecode, object code, binary, or any 
 * other form resulting from mechanical transformation or translation of the 
 * Source form.
 * 
 * Permission for Use and Modification Without Distribution
 * 
 * (1) You are permitted to use the Standard Version and create and use Modified
 * Versions for any purpose without restriction, provided that you do not 
 * Distribute the Modified Version.
 * 
 * Permissions for Redistribution of the Standard Version
 * 
 * (2) You may Distribute verbatim copies of the Source form of the Standard 
 * Version of this Package in any medium without restriction, either gratis or 
 * for a Distributor Fee, provided that you duplicate all of the original 
 * copyright notices and associated disclaimers. At your discretion, such 
 * verbatim copies may or may not include a Compiled form of the Package.
 * 
 * (3) You may apply any bug fixes, portability changes, and other modifications
 * made available from the Copyright Holder. The resulting Package will still be
 * considered the Standard Version, and as such will be subject to the Original 
 * License.
 * 
 * Distribution of Modified Versions of the Package as Source
 * 
 * (4) You may Distribute your Modified Version as Source (either gratis or for
 * a Distributor Fee, and with or without a Compiled form of the Modified
 * Version) provided that you clearly document how it differs from the Standard
 * Version, including, but not limited to, documenting any non-standard 
 * features, executables, or modules, and provided that you do at least ONE of 
 * the following:
 * 
 *      (a) make the Modified Version available to the Copyright Holder of the 
 *          Standard Version, under the Original License, so that the Copyright 
 *          Holder may include your modifications in the Standard Version.
 *
 *      (b) ensure that installation of your Modified Version does not prevent 
 *          the user installing or running the Standard Version. In addition, 
 *          the Modified Version must bear a name that is different from the 
 *          name of the Standard Version.
 *
 *      (c) allow anyone who receives a copy of the Modified Version to make the
 *          Source form of the Modified Version available to others under
 *
 *          (i) the Original License or
 *
 *          (ii) a license that permits the licensee to freely copy, modify and 
 *              redistribute the Modified Version using the same licensing terms
 *              that apply to the copy that the licensee received, and requires 
 *              that the Source form ofthe Modified Version, and of any works 
 *              derived from it, be made freely available in that license fees 
 *              are prohibited but Distributor Fees are allowed.
 *
 * Distribution of Compiled Forms of the Standard Version or Modified Versions
 * without the Source
 * 
 * (5) You may Distribute Compiled forms of the Standard Version without the 
 * Source, provided that you include complete instructions on how to get the 
 * Source of the Standard Version. Such instructions must be valid at the time 
 * of your distribution. If these instructions, at any time while you are 
 * carrying out such distribution, become invalid, you must provide new 
 * instructions on demand or cease further distribution. If you provide valid 
 * instructions or cease distribution within thirty days after you become aware 
 * that the instructions are invalid, then you do not forfeit any of your rights
 * under this license.
 *
 * (6) You may Distribute a Modified Version in Compiled form without the 
 * Source, provided that you comply with Section 4 with respect to the Source of
 * the Modified Version.
 *
 * Aggregating or Linking the Package
 *
 * (7) You may aggregate the Package (either the Standard Version or Modified 
 * Version) with other packages and Distribute the resulting aggregation 
 * provided that you do not charge a licensing fee for the Package. Distributor 
 * Fees are permitted, and licensing fees for other components in the 
 * aggregation are permitted. The terms of this license apply to the use and
 * Distribution of the Standard or Modified Versions as included in the
 * aggregation.
 *
 * (8) You are permitted to link Modified and Standard Versions with other 
 * works, to embed the Package in a larger work of your own, or to build 
 * stand-alone binary or bytecode versions of applications that include the 
 * Package, and Distribute the result without restriction, provided the result 
 * does not expose a direct interface to the Package.
 * 
 * Items That are Not Considered Part of a Modified Version
 *
 * (9) Works (including, but not limited to, modules and scripts) that merely 
 * extend or make use of the Package, do not, by themselves, cause the Package 
 * to be a Modified Version. In addition, such works are not considered parts of
 * the Package itself, and are not subject to the terms of this license. 
 * 
 * General Provisions
 *
 * (10) Any use, modification, and distribution of the Standard or Modified
 * Versions is governed by this Artistic License. By using, modifying or 
 * distributing the Package, you accept this license. Do not use, modify, or 
 * distribute the Package, if you do not accept this license.
 *
 * (11) If your Modified Version has been derived from a Modified Version made 
 * by someone other than you, you are nevertheless required to ensure that your 
 * Modified Version complies with the requirements of this license.
 *
 * (12) This license does not grant you the right to use any trademark, service 
 * mark, tradename, or logo of the Copyright Holder.
 *
 * (13) This license includes the non-exclusive, worldwide, free-of-charge 
 * patent license to make, have made, use, offer to sell, sell, import and 
 * otherwise transfer the Package with respect to any patent claims licensable 
 * by the Copyright Holder that are necessarily infringed by the Package. If you
 * institute patent litigation (including a cross-claim or counterclaim) against
 * any party alleging that the Package constitutes direct or contributory patent
 * infringement, then this Artistic License to you shall terminate on the date 
 * that such litigation is filed.
 *
 * (14) Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER 
 * AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES. THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY YOUR LOCAL LAW. 
 * UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR CONTRIBUTOR WILL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING IN ANY WAY
 * OUT OF THE USE OF THE PACKAGE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH 
 * DAMAGE.
 */

/**
 * A helpful class to convert between the HSV and RGB colorspaces.
 * @author AurorisNET
 */
public class Color
{
	private float h; //[0,360] Hue
	private float s; //[0-1] Saturation
	private float v; //[0-1] Value
	private float r; //[0-1] Red
	private float g; //[0-1] Green
	private float b; //[0-1] Blue
	private String hex; // Hexadecimal notation of RGB

	/**
	 * Set the Hue, Saturation and Value (Brightness) variables. This will automatically populate the Red, Green, Blue, and Hexadecimal fields, too.
	 *
	 * It represents points in the RGB color space, which attempt to describe perceptual color relationships more accurately than RGB. HSV describes colors as points in a cylinder whose central axis ranges from black at the bottom to white at the top with neutral colors between them, where angle around the axis corresponds to hue, distance from the axis corresponds to saturation, and distance along the axis corresponds to lightness, value, or brightness.
	 * @param h Hue - valid range is 0-359
	 * @param s Saturation - valid range is 0-100
	 * @param v Value (Brightness) - valid range is 0-100
	 * @throws java.lang.Exception A general exception if the Hue, Saturation, or Value variables are out of range.
	 */
	public void setHSV(int h, int s, int v) throws Exception
	{
		if (h < 0 || h > 360) throw new Exception();
		if (s < 0 || s > 100) throw new Exception();
		if (v < 0 || v > 100) throw new Exception();

		this.h = (float)h;
		this.s = (float)s / 100;
		this.v = (float)v / 100;

		HSVtoRGB(this.h, this.s, this.v);

		setHex();
	}

	/**
	 * Sets the Red, Green, and Blue color variables. This will automatically convert to Hue, Saturation and Brightness and Hexadecimal.
	 *
	 * The RGB color model is an additive color model in which red, green, and blue light are added together in various ways to reproduce a broad array of colors. The name of the model comes from the initials of the three additive primary colors, red, green, and blue.
	 * @param r Red - valid range is 0-255
	 * @param g Green - valid range is 0-255
	 * @param b Blue - valid range is 0-255
	 * @throws java.lang.Exception Exception if the Red, Green or Blue variables are out of range.
	 */
	public void setRGB(int r, int g, int b) throws Exception
	{
		if (r < 0 || r > 255) throw new Exception();
		if (g < 0 || g > 255) throw new Exception();
		if (b < 0 || b > 255) throw new Exception();

		this.r = (float)r / 255;
		this.g = (float)g / 255;
		this.b = (float)b / 255;

		RGBtoHSV(this.r, this.g, this.b);

		setHex();
	}

	/**
	 * Sets the hexadecimal representation of Red, Green and Blue.
	 * @param hex The hexadecimal string notation. It must be 6 letters long and consist of the characters 0-9 and A-F.
	 * @throws java.lang.Exception Exception if the hexadecimal string cannot be parsed into its Red, Green, and Blue components.
	 */
	public void setHex(String hex) throws Exception
	{
		if (hex.length() == 6)
		{
			setRGB(Integer.parseInt(hex.substring(0,2), 16),
				Integer.parseInt(hex.substring(2,4), 16),
				Integer.parseInt(hex.substring(4,6), 16));
		}
		else
		{
			throw new Exception();
		}
	}

	/**
	 * Converts from RGB to Hexadecimal notation.
	 */
    private void setHex()
	{
		String hRed = Integer.toHexString(getRed());
		String hGreen = Integer.toHexString(getGreen());
		String hBlue = Integer.toHexString(getBlue());

		if (hRed.length() == 0) { hRed = "00"; } //$NON-NLS-1$
		if (hRed.length() == 1) { hRed = "0" + hRed; } //$NON-NLS-1$
		if (hGreen.length() == 0) { hGreen = "00"; } //$NON-NLS-1$
		if (hGreen.length() == 1) { hGreen = "0" + hGreen; } //$NON-NLS-1$
		if (hBlue.length() == 0) { hBlue = "00"; } //$NON-NLS-1$
		if (hBlue.length() == 1) { hBlue = "0" + hBlue; } //$NON-NLS-1$

		this.hex = hRed + hGreen + hBlue;
	}

	/**
	 * Returns the integer of the red component of the RGB colorspace, or 0 if it hasn't been set.
	 * @return red component
	 */
	public int getRed()
	{
		return (int)(r * 255);
	}

	/**
	 * Returns the integer of the green component of the RGB colorspace, or 0 if it hasn't been set.
	 * @return green component
	 */
	public int getGreen()
	{
		return (int)(g * 255);
	}

	/**
	 * Returns the integer of the blue component of the RGB colorspace, or 0 if it hasn't been set.
	 * @return blue component
	 */
	public int getBlue()
	{
		return (int)(b * 255);
	}

	/**
	 * Returns the integer of the hue component of the HSV colorspace, or 0 if it hasn't been set.
	 * @return hue component
	 */
	public int getHue()
	{
		return (int)h;
	}

	/**
	 * Returns the integer of the saturation component of the HSV colorspace, or 0 if it hasn't been set.
	 * @return saturation component
	 */
	public int getSaturation()
	{
		return (int)(s * 100);
	}

	/**
	 * Returns the integer of the value (brightness) component of the HSV colorspace, or 0 if it hasn't been set.
	 * @return value component
	 */
	public int getValue()
	{
		return (int)(v * 100);
	}

	/**
	 * Returns the hexadecimal representation of the RGB colorspace.
	 * @return hexadecimal representation
	 */
	public String getHex()
	{
		return hex;
	}

	// r,g,b values are from 0 to 1
	// h = [0,360], s = [0,1], v = [0,1]
	//		if s == 0, then h = (undefined)
	/**
	 * Converts the RGB colorspace into the HSV colorspace.  This private method works exclusively with internal variables and does not return anything.
	 * @param r Red
	 * @param g Green
	 * @param b Blue
	 */
	private void RGBtoHSV(float r, float g, float b)
	{
		float min = 0;
		float max = 0;
		float delta = 0;

		min = MIN(r, g, b);
		max = MAX(r, g, b);

		this.v = max;				// v

		delta = max - min;

		if(max != 0)
		{
			this.s = delta / max;		// s
		}
		else
		{
			// r = g = b = 0		// s = 0, v is undefined
			this.s = 0;
			this.h = 0;
			return;
		}

		if (delta == 0)
		{
			h = 0;
			return;
		}

		if(r == max)
		{
			this.h = (g - b) / delta;		// between yellow & magenta
		}
		else if(g == max)
		{
			this.h = 2 + (b - r) / delta;	// between cyan & yellow
		}
		else
		{
			this.h = 4 + (r - g) / delta;	// between magenta & cyan
		}

		this.h *= 60;				// degrees

		if(this.h < 0)
			this.h += 360;
	}

	/**
	 * Converts the HSV colorspace into the RGB colorspace. This private method works exclusively with internal variables and does not return anything.
	 * @param h Hue
	 * @param s Saturation
	 * @param v Value (Brightness)
	 */
	private void HSVtoRGB(float h, float s, float v)
	{
		int i;
		float f;
		float p;
		float q;
		float t;
		if(s == 0)
		{
			// achromatic (grey)
			this.r = v;
			this.g = v;
			this.b = v;
			return;
		}
		h /= 60;			// sector 0 to 5
		i = (int)Math.floor(h);
		f = h - i;			// factorial part of h
		p = v * (1 - s);
		q = v * (1 - s * f);
		t = v * (1 - s * (1 - f));
		switch( i )
		{
			case 0:
				this.r = v;
				this.g = t;
				this.b = p;
				break;
			case 1:
				this.r = q;
				this.g = v;
				this.b = p;
				break;
			case 2:
				this.r = p;
				this.g = v;
				this.b = t;
				break;
			case 3:
				this.r = p;
				this.g = q;
				this.b = v;
				break;
			case 4:
				this.r = t;
				this.g = p;
				this.b = v;
				break;
			default:		// case 5:
				this.r = v;
				this.g = p;
				this.b = q;
				break;
		}
	}

	/**
	 * A helper function to determine the largest value between the three inputs.
	 * @param a First value
	 * @param b Second value
	 * @param c Third value
	 * @return The value of the largest of the inputs.
	 */
	private float MAX(float a, float b, float c)
	{
		float max = Integer.MIN_VALUE;
		if (a > max)
			max = a;
		if (b > max)
			max = b;
		if (c > max)
			max = c;
		return max;
	}

	/**
	 * A helper function to determine the smallest value between the three inputs.
	 * @param a First value
	 * @param b Second value
	 * @param c Third value
	 * @return The value of the smallest of the inputs.
	 */
	private float MIN(float a, float b, float c)
	{
		float min = Integer.MAX_VALUE;
		if (a < min)
			min = a;
		if (b < min)
			min = b;
		if (c < min)
			min = c;
		return min;
	}
}