package org.pentaho.gwt.widgets.client.colorpicker;

/**
 * Copyright (c) 2007, AurorisNET.
 *
 * Everyone is permitted to copy and distribute verbatim copies of this license
 * document, but changing it is not allowed.
 *
 * Preamble
 *
 * This license establishes the terms under which a given free software Package
 * may be copied, modified, distributed, and/or redistributed. The intent is
 * that the Copyright Holder maintains some artistic control over the
 * development of that Package while still keeping the Package available as open
 * source and free software.
 *
 * You are always permitted to make arrangements wholly outside of this license
 * directly with the Copyright Holder of a given Package. If the terms of this
 * license do not permit the full use that you propose to make of the Package,
 * you should contact the Copyright Holder and seek a different licensing
 * arrangement.
 *
 * Definitions
 *
 * "Copyright Holder" means the individual(s) or organization(s) named in the
 * copyright notice for the entire Package.
 *
 * "Contributor" means any party that has contributed code or other material to
 * the Package, in accordance with the Copyright Holder's procedures.
 *
 * "You" and "your" means any person who would like to copy, distribute, or
 * modify the Package.
 *
 * "Package" means the collection of files distributed by the Copyright Holder,
 * and derivatives of that collection and/or of those files. A given Package may
 * consist of either the Standard Version, or a Modified Version.
 *
 * "Distribute" means providing a copy of the Package or making it accessible to
 * anyone else, or in the case of a company or organization, to others outside
 * of your company or organization.
 *
 * "Distributor Fee" means any fee that you charge for Distributing this Package
 * or providing support for this Package to another party. It does not mean
 * licensing fees.
 *
 * "Standard Version" refers to the Package if it has not been modified, or has
 * been modified only in ways explicitly requested by the Copyright Holder.
 *
 * "Modified Version" means the Package, if it has been changed, and such
 * changes were not explicitly requested by the Copyright Holder.
 *
 * "Original License" means this Artistic License as Distributed with the
 * Standard Version of the Package, in its current version or as it may be
 * modified by The Perl Foundation in the future.
 *
 * "Source" form means the source code, documentation source, and configuration
 * files for the Package.
 *
 * "Compiled" form means the compiled bytecode, object code, binary, or any
 * other form resulting from mechanical transformation or translation of the
 * Source form.
 *
 * Permission for Use and Modification Without Distribution
 *
 * (1) You are permitted to use the Standard Version and create and use Modified
 * Versions for any purpose without restriction, provided that you do not
 * Distribute the Modified Version.
 *
 * Permissions for Redistribution of the Standard Version
 *
 * (2) You may Distribute verbatim copies of the Source form of the Standard
 * Version of this Package in any medium without restriction, either gratis or
 * for a Distributor Fee, provided that you duplicate all of the original
 * copyright notices and associated disclaimers. At your discretion, such
 * verbatim copies may or may not include a Compiled form of the Package.
 *
 * (3) You may apply any bug fixes, portability changes, and other modifications
 * made available from the Copyright Holder. The resulting Package will still be
 * considered the Standard Version, and as such will be subject to the Original
 * License.
 *
 * Distribution of Modified Versions of the Package as Source
 *
 * (4) You may Distribute your Modified Version as Source (either gratis or for
 * a Distributor Fee, and with or without a Compiled form of the Modified
 * Version) provided that you clearly document how it differs from the Standard
 * Version, including, but not limited to, documenting any non-standard
 * features, executables, or modules, and provided that you do at least ONE of
 * the following:
 *
 *      (a) make the Modified Version available to the Copyright Holder of the
 *          Standard Version, under the Original License, so that the Copyright
 *          Holder may include your modifications in the Standard Version.
 *
 *      (b) ensure that installation of your Modified Version does not prevent
 *          the user installing or running the Standard Version. In addition,
 *          the Modified Version must bear a name that is different from the
 *          name of the Standard Version.
 *
 *      (c) allow anyone who receives a copy of the Modified Version to make the
 *          Source form of the Modified Version available to others under
 *
 *          (i) the Original License or
 *
 *          (ii) a license that permits the licensee to freely copy, modify and
 *              redistribute the Modified Version using the same licensing terms
 *              that apply to the copy that the licensee received, and requires
 *              that the Source form ofthe Modified Version, and of any works
 *              derived from it, be made freely available in that license fees
 *              are prohibited but Distributor Fees are allowed.
 *
 * Distribution of Compiled Forms of the Standard Version or Modified Versions
 * without the Source
 *
 * (5) You may Distribute Compiled forms of the Standard Version without the
 * Source, provided that you include complete instructions on how to get the
 * Source of the Standard Version. Such instructions must be valid at the time
 * of your distribution. If these instructions, at any time while you are
 * carrying out such distribution, become invalid, you must provide new
 * instructions on demand or cease further distribution. If you provide valid
 * instructions or cease distribution within thirty days after you become aware
 * that the instructions are invalid, then you do not forfeit any of your rights
 * under this license.
 *
 * (6) You may Distribute a Modified Version in Compiled form without the
 * Source, provided that you comply with Section 4 with respect to the Source of
 * the Modified Version.
 *
 * Aggregating or Linking the Package
 *
 * (7) You may aggregate the Package (either the Standard Version or Modified
 * Version) with other packages and Distribute the resulting aggregation
 * provided that you do not charge a licensing fee for the Package. Distributor
 * Fees are permitted, and licensing fees for other components in the
 * aggregation are permitted. The terms of this license apply to the use and
 * Distribution of the Standard or Modified Versions as included in the
 * aggregation.
 *
 * (8) You are permitted to link Modified and Standard Versions with other
 * works, to embed the Package in a larger work of your own, or to build
 * stand-alone binary or bytecode versions of applications that include the
 * Package, and Distribute the result without restriction, provided the result
 * does not expose a direct interface to the Package.
 *
 * Items That are Not Considered Part of a Modified Version
 *
 * (9) Works (including, but not limited to, modules and scripts) that merely
 * extend or make use of the Package, do not, by themselves, cause the Package
 * to be a Modified Version. In addition, such works are not considered parts of
 * the Package itself, and are not subject to the terms of this license.
 *
 * General Provisions
 *
 * (10) Any use, modification, and distribution of the Standard or Modified
 * Versions is governed by this Artistic License. By using, modifying or
 * distributing the Package, you accept this license. Do not use, modify, or
 * distribute the Package, if you do not accept this license.
 *
 * (11) If your Modified Version has been derived from a Modified Version made
 * by someone other than you, you are nevertheless required to ensure that your
 * Modified Version complies with the requirements of this license.
 *
 * (12) This license does not grant you the right to use any trademark, service
 * mark, tradename, or logo of the Copyright Holder.
 *
 * (13) This license includes the non-exclusive, worldwide, free-of-charge
 * patent license to make, have made, use, offer to sell, sell, import and
 * otherwise transfer the Package with respect to any patent claims licensable
 * by the Copyright Holder that are necessarily infringed by the Package. If you
 * institute patent litigation (including a cross-claim or counterclaim) against
 * any party alleging that the Package constitutes direct or contributory patent
 * infringement, then this Artistic License to you shall terminate on the date
 * that such litigation is filed.
 *
 * (14) Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
 * AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES. THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 * NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY YOUR LOCAL LAW.
 * UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR CONTRIBUTOR WILL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING IN ANY WAY
 * OUT OF THE USE OF THE PACKAGE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.ui.ChangeListener;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.KeyboardListener;
import com.google.gwt.user.client.ui.RadioButton;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.Widget;

/**
 * This is the implementation of the Colorpicker. It defines the user interface, the glue and calculations necessary for colorpicker functionality.
 *
 * <h1>Example</h1>
 * <CODE>
 * public class ColorPickerExample implements EntryPoint {
 *
 *  public void onModuleLoad() {
 *    // Make a new colorpicker
 *    ColorPicker picker = new ColorPicker();
 *
 *    // Add it to the root panel.
 *    RootPanel.get().add(picker);
 *
 *    // Make a new button that does something when you click it.
 *    Button b = new Button("Pick!", new ClickListener() {
 *      public void onClick(Widget sender) {
 *        Window.alert("You chose " + picker.getHexColor());
 *      }
 *    });
 *
 *    // Add it to the root panel.
 *    RootPanel.get().add(b);
 *  }
 * }
 * </CODE>
 * @author AurorisNET
 * @copyright (C) 2007 AurorisNET. All Rights Reserved.
 */
@SuppressWarnings("deprecation")
public class ColorPicker extends Composite implements ClickListener, KeyboardListener, ChangeListener
{
	// Elements
	private HTML colorpreview;
	private SliderMap slidermap; // main color picking slider. Leftmost UI component.
	private SliderBar sliderbar; // auxilliary color picking slider. Center UI component.

	// Textboxes
        private TextBox tbHue;
	private TextBox tbSaturation;
	private TextBox tbBrightness;
	private TextBox tbRed;
	private TextBox tbGreen;
	private TextBox tbBlue;
	private TextBox tbHexColor;

	// Radiobuttons
        private RadioButton rbHue;
	private RadioButton rbSaturation;
	private RadioButton rbBrightness;
	private RadioButton rbRed;
	private RadioButton rbGreen;
	private RadioButton rbBlue;

	private int colorMode; // Which color picking mode we are in
	private int red;
	private int green;
	private int blue;
	private int hue;
	private int saturation;
	private int brightness;

	public ColorPicker()
	{
		// UI Drawing
		//------------------

		hue = 0;
		saturation = 100;
		brightness = 100;
		red = 255;
		green = 0;
		blue = 0;

		HorizontalPanel panel = new HorizontalPanel();
		FlexTable table = new FlexTable();

		// Add the large slider map
		slidermap = new SliderMap(this);
		panel.add(slidermap);
		panel.setCellWidth(slidermap, "258px"); //$NON-NLS-1$
		panel.setCellHeight(slidermap, "258px"); //$NON-NLS-1$

		// Add the small slider bar
		sliderbar = new SliderBar(this);
		panel.add(sliderbar);
		panel.setCellWidth(sliderbar, "40px"); //$NON-NLS-1$
		panel.setCellHeight(sliderbar, "258px"); //$NON-NLS-1$

		// Define the Flextable's content
		// Color preview at the top
		colorpreview = new HTML(""); //$NON-NLS-1$
		colorpreview.setWidth("50px"); //$NON-NLS-1$
		colorpreview.setHeight("50px"); //$NON-NLS-1$
		DOM.setStyleAttribute(colorpreview.getElement(), "border", "1px solid black"); //$NON-NLS-1$ //$NON-NLS-2$

		// Radio buttons
		rbHue = new RadioButton("color", "H:"); //$NON-NLS-1$ //$NON-NLS-2$
		rbHue.addClickListener(this);
		rbSaturation = new RadioButton("color", "S:"); //$NON-NLS-1$ //$NON-NLS-2$
		rbSaturation.addClickListener(this);
		rbBrightness = new RadioButton("color", "V:"); //$NON-NLS-1$ //$NON-NLS-2$
		rbBrightness.addClickListener(this);
		rbRed = new RadioButton("color", "R:"); //$NON-NLS-1$ //$NON-NLS-2$
		rbRed.addClickListener(this);
		rbGreen = new RadioButton("color", "G:"); //$NON-NLS-1$ //$NON-NLS-2$
		rbGreen.addClickListener(this);
		rbBlue = new RadioButton("color", "B:"); //$NON-NLS-1$ //$NON-NLS-2$
		rbBlue.addClickListener(this);

		// Textboxes
		tbHue = new TextBox();
		tbHue.setText(new Integer(hue).toString());
		tbHue.setMaxLength(3);
		tbHue.setVisibleLength(4);
		tbHue.addKeyboardListener(this);
		tbHue.addChangeListener(this);
		tbSaturation = new TextBox();
		tbSaturation.setText(new Integer(saturation).toString());
		tbSaturation.setMaxLength(3);
		tbSaturation.setVisibleLength(4);
		tbSaturation.addKeyboardListener(this);
		tbSaturation.addChangeListener(this);
		tbBrightness = new TextBox();
		tbBrightness.setText(new Integer(brightness).toString());
		tbBrightness.setMaxLength(3);
		tbBrightness.setVisibleLength(4);
		tbBrightness.addKeyboardListener(this);
		tbBrightness.addChangeListener(this);
		tbRed = new TextBox();
		tbRed.setText(new Integer(red).toString());
		tbRed.setMaxLength(3);
		tbRed.setVisibleLength(4);
		tbRed.addKeyboardListener(this);
		tbRed.addChangeListener(this);
		tbGreen = new TextBox();
		tbGreen.setText(new Integer(green).toString());
		tbGreen.setMaxLength(3);
		tbGreen.setVisibleLength(4);
		tbGreen.addKeyboardListener(this);
		tbGreen.addChangeListener(this);
		tbBlue = new TextBox();
		tbBlue.setText(new Integer(blue).toString());
		tbBlue.setMaxLength(3);
		tbBlue.setVisibleLength(4);
		tbBlue.addKeyboardListener(this);
		tbBlue.addChangeListener(this);
		tbHexColor = new TextBox();
		tbHexColor.setText("ff0000"); //$NON-NLS-1$
		tbHexColor.setMaxLength(6);
		tbHexColor.setVisibleLength(6);
		tbHexColor.addKeyboardListener(this);
		tbHexColor.addChangeListener(this);

		// Put together the FlexTable
		table.setWidget(0, 0, colorpreview);
		table.getFlexCellFormatter().setColSpan(0, 0, 3);
		table.setWidget(1, 0, rbHue);
		table.setWidget(1, 1, tbHue);
		table.setWidget(1, 2, new HTML("&deg;")); //$NON-NLS-1$
		table.setWidget(2, 0, rbSaturation);
		table.setWidget(2, 1, tbSaturation);
		table.setText(2, 2, "%"); //$NON-NLS-1$
		table.setWidget(3, 0, rbBrightness);
		table.setWidget(3, 1, tbBrightness);
		table.setText(3, 2, "%"); //$NON-NLS-1$
		table.setWidget(4, 0, rbRed);
		table.setWidget(4, 1, tbRed);
		table.setWidget(5, 0, rbGreen);
		table.setWidget(5, 1, tbGreen);
		table.setWidget(6, 0, rbBlue);
		table.setWidget(6, 1, tbBlue);
		table.setText(7,0, "#:"); //$NON-NLS-1$
		table.setWidget(7,1, tbHexColor);
		table.getFlexCellFormatter().setColSpan(7, 1, 2);

		// Final setup
		panel.add(table);
		rbSaturation.setChecked(true);
		setPreview("ff0000"); //$NON-NLS-1$
		DOM.setStyleAttribute(colorpreview.getElement(), "cursor", "default"); //$NON-NLS-1$ //$NON-NLS-2$

		// First event
		onClick(rbSaturation);

		initWidget(panel);
	}

	/**
	 * This method is called when a widget is attached to the browser's document. To receive notification after a Widget has been added to the document, override the Widget.onLoad() method.
	 *
	 * Subclasses that override this method must call <tt>super.onAttach()</tt> before doing anything else to ensure that the Widget has been properly attached to its underlying Element.
	 */
	public void onAttach()
	{
		// Called when we are shown (from being hidden)
		super.onAttach();
		colorMode = -1;
		updateSliders();
	}

	/**
	 * Called when the widget wants to update the preview color sample box in the top-right corner of the UI.
	 * @param hex Hexadecimal notation of RGB
	 */
	private void setPreview(String hex)
	{
		DOM.setStyleAttribute(colorpreview.getElement(), "backgroundColor", "#" + hex); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * Fires whenever the user generates picking events on the color picker map.
	 *
	 * Subclasses that override this method must call <tt>super.onMapSelected(x,y)</tt> to ensure that the Widget recieves its events.
	 * @param x the distance along the x-axis of the user's selection, between 0 and 255, inclusive.
	 * @param y the distance along the y-axis of the user's selection, between 0 and 255, inclusive.
	 */
	public void onMapSelected(int x, int y)
	{
		switch (colorMode)
		{
			case SliderMap.Hue:
				saturation = percentOf(x, 100);
				brightness = 100 - percentOf(y, 100);
				tbSaturation.setText(Integer.toString(saturation));
				tbBrightness.setText(Integer.toString(brightness));
				onChange(tbHue);
				break;
			case SliderMap.Saturation:
				hue = percentOf(x, 360);
				brightness = 100 - percentOf(y, 100);
				tbHue.setText(Integer.toString(hue));
				tbBrightness.setText(Integer.toString(brightness));
				onChange(tbSaturation);
				break;
			case SliderMap.Brightness:
				hue = percentOf(x, 360);
				saturation = 100 - percentOf(y, 100);
				tbHue.setText(Integer.toString(hue));
				tbSaturation.setText(Integer.toString(saturation));
				onChange(tbBrightness);
				break;
			case SliderMap.Red:
				blue = x;
				green = 256 - y;
				tbBlue.setText(Integer.toString(blue));
				tbGreen.setText(Integer.toString(green));
				onChange(tbRed);
				break;
			case SliderMap.Green:
				blue = x;
				red = 256 - y;
				tbBlue.setText(Integer.toString(blue));
				tbRed.setText(Integer.toString(red));
				onChange(tbGreen);
				break;
			case SliderMap.Blue:
				red = x;
				green = 256 - y;
				tbRed.setText(Integer.toString(red));
				tbGreen.setText(Integer.toString(green));
				onChange(tbBlue);
				break;
		}
	}

	/**
	 * Fires whenever the user generates picking events along the color picker bar.
	 *
	 * Subclasses that override this method must call <tt>super.onBarSelected(y)</tt> to ensure that the Widget recieves its events.
	 * @param y the distance along the y-axis of the user's selection, between 0 and 255, inclusive.
	 */
	public void onBarSelected(int y)
	{
		switch (colorMode)
		{
			case SliderMap.Hue:
				hue = 360 - percentOf(y, 360);
				tbHue.setText(Integer.toString(hue));
				onChange(tbHue);
				break;
			case SliderMap.Saturation:
				saturation = 100 - percentOf(y, 100);
				tbSaturation.setText(Integer.toString(saturation));
				onChange(tbSaturation);
				break;
			case SliderMap.Brightness:
				brightness = 100 - percentOf(y, 100);
				tbBrightness.setText(Integer.toString(brightness));
				onChange(tbBrightness);
				break;
			case SliderMap.Red:
				red = 255 - y;
				tbRed.setText(Integer.toString(red));
				onChange(tbRed);
				break;
			case SliderMap.Green:
				green = 255 - y;
				tbGreen.setText(Integer.toString(green));
				onChange(tbGreen);
				break;
			case SliderMap.Blue:
				blue = 255 - y;
				tbBlue.setText(Integer.toString(blue));
				onChange(tbBlue);
				break;
		}
	}

	/**
	 * Fired when the user clicks on a widget.
	 *
	 * Subclasses that override this method must call <tt>super.onClick(sender)</tt> to ensure that the Widget recieves its events.
	 * @param sender the widget sending the event.
	 */
	public void onClick(Widget sender)
	{
		if (sender == rbHue)
		{
			if (colorMode != SliderMap.Hue)
			{
				colorMode = SliderMap.Hue;
				slidermap.setColorSelectMode(SliderMap.Hue);
				sliderbar.setColorSelectMode(SliderBar.Hue);
				slidermap.setOverlayOpacity(100);
				sliderbar.setLayerOpacity(100, SliderBar.BarD);
			}

			try
			{
				Color color = new Color();
				color.setHSV(hue, 100, 100);
				slidermap.setOverlayColor("#" + color.getHex()); //$NON-NLS-1$
			}
			catch (Exception e) {}

			sliderbar.setSliderPosition(256 - (int)((new Integer(hue).floatValue() / 360) * 256));
			slidermap.setSliderPosition((int)((new Integer(saturation).floatValue() / 100) * 256),
										256 - (int)((new Integer(brightness).floatValue() / 100) * 256));
		}
		else if(sender == rbSaturation)
		{
			if (colorMode != SliderMap.Saturation)
			{
				colorMode = SliderMap.Saturation;
				slidermap.setColorSelectMode(SliderMap.Saturation);
				sliderbar.setColorSelectMode(SliderBar.Saturation);
				slidermap.setOverlayColor("transparent"); //$NON-NLS-1$
				sliderbar.setLayerColor("#ffffff", SliderBar.BarC); //$NON-NLS-1$
				sliderbar.setLayerOpacity(100, SliderBar.BarD);
			}

			slidermap.setOverlayOpacity(100 - saturation);
			sliderbar.setLayerColor(tbHexColor.getText(), SliderBar.BarD);

			sliderbar.setSliderPosition(256 - (int)((new Integer(saturation).floatValue() / 100) * 256));
			slidermap.setSliderPosition((int)((new Integer(hue).floatValue() / 360) * 256),
										256 - (int)((new Integer(brightness).floatValue() / 100) * 256));
		}
		else if(sender == rbBrightness)
		{
			if (colorMode != SliderMap.Brightness)
			{
				colorMode = SliderMap.Brightness;
				slidermap.setColorSelectMode(SliderMap.Brightness);
				sliderbar.setColorSelectMode(SliderBar.Brightness);
				slidermap.setUnderlayColor("#000000"); //$NON-NLS-1$
				slidermap.setOverlayColor("transparent"); //$NON-NLS-1$
				sliderbar.setLayerOpacity(100, SliderBar.BarD);
			}

			try
			{
				Color color = new Color();
				color.setHSV(hue, saturation, 100);
				sliderbar.setLayerColor("#" + color.getHex(), SliderBar.BarD); //$NON-NLS-1$
			}
			catch (Exception e) {}

			slidermap.setOverlayOpacity(brightness);

			sliderbar.setSliderPosition(256 - (int)((new Integer(brightness).floatValue() / 100) * 256));
			slidermap.setSliderPosition((int)((new Integer(hue).floatValue() / 360) * 256),
										256 - (int)((new Integer(saturation).floatValue() / 100) * 256));
		}
		else if(sender == rbRed)
		{
			if (colorMode != SliderMap.Red)
			{
				colorMode = SliderMap.Red;
				slidermap.setColorSelectMode(SliderMap.Red);
				sliderbar.setColorSelectMode(SliderBar.Red);
			}
			slidermap.setOverlayOpacity(percentOf(red, 100));

			sliderbar.setSliderPosition(256 - red);
			slidermap.setSliderPosition(blue, 256 - green);
		}
		else if(sender == rbGreen)
		{
			if (colorMode != SliderMap.Green)
			{
				colorMode = SliderMap.Green;
				slidermap.setColorSelectMode(SliderMap.Green);
				sliderbar.setColorSelectMode(SliderBar.Green);
			}

			slidermap.setOverlayOpacity(percentOf(green, 100));

			sliderbar.setSliderPosition(256 - green);
			slidermap.setSliderPosition(blue, 256 - red);
		}
		else if(sender == rbBlue)
		{
			if (colorMode != SliderMap.Blue)
			{
				colorMode = SliderMap.Blue;
				slidermap.setColorSelectMode(SliderMap.Blue);
				sliderbar.setColorSelectMode(SliderBar.Blue);
			}

			slidermap.setOverlayOpacity(percentOf(blue, 100));

			sliderbar.setSliderPosition(256 - blue);
			slidermap.setSliderPosition(red, 256 - green);
		}

		if (colorMode == SliderMap.Red || colorMode == SliderMap.Green || colorMode == SliderMap.Blue)
		{
			int x = 0;
			int y = 0;

			if (colorMode == SliderMap.Red)
			{
				x = blue;
				y = green;
			}

			if (colorMode == SliderMap.Green)
			{
				x = blue;
				y = red;
			}

			if (colorMode == SliderMap.Blue)
			{
				x = red;
				y = green;
			}

			int horzPer = (int)((new Float(x).floatValue() / 256) * 100);
			int vertPer = (int)((new Float(y).floatValue() / 256) * 100);
			int horzPerRev = (int)(((256 - new Float(x).floatValue()) / 256) * 100);
			int vertPerRev = (int)(((256 - new Float(y).floatValue()) / 256) * 100);

			if (vertPerRev > horzPerRev)
				sliderbar.setLayerOpacity(horzPerRev, SliderBar.BarD);
			else
				sliderbar.setLayerOpacity(vertPerRev, SliderBar.BarD);
			if (vertPerRev > horzPer)
				sliderbar.setLayerOpacity(horzPer, SliderBar.BarC);
			else
				sliderbar.setLayerOpacity(vertPerRev, SliderBar.BarC);
			if (vertPer > horzPer)
				sliderbar.setLayerOpacity(horzPer, SliderBar.BarB);
			else
				sliderbar.setLayerOpacity(vertPer, SliderBar.BarB);
			if (vertPer > horzPerRev)
				sliderbar.setLayerOpacity(horzPerRev, SliderBar.BarA);
			else
				sliderbar.setLayerOpacity(vertPer, SliderBar.BarA);
		}
	}

	/**
	 * Fired whenever something in this widget changes.
	 *
	 * Subclasses that override this method must call <tt>super.onChange(sender)</tt> to ensure that the Widget recieves its events.
	 * @param sender the widget that has changed.
	 */
	public void onChange(Widget sender)
	{
	    if (sender == tbHexColor)
        {
			// Figure out colors
			// Color class will do bounds check on hex input
			try
			{
				Color color = new Color();
				color.setHex(tbHexColor.getText());
				tbHue.setText(Integer.toString(color.getHue()));
				tbSaturation.setText(Integer.toString(color.getSaturation()));
				tbBrightness.setText(Integer.toString(color.getValue()));
				tbRed.setText(Integer.toString(color.getRed()));
				tbGreen.setText(Integer.toString(color.getGreen()));
				tbBlue.setText(Integer.toString(color.getBlue()));
				tbHexColor.setText(color.getHex());
				setPreview(color.getHex());
			}
			catch (Exception e)
			{
			}
		}

		if (sender == tbRed || sender == tbGreen || sender == tbBlue)
		{
			// Don't allow this value to overflow or underflow
			try
			{
				if (Integer.parseInt(((TextBox)sender).getText()) > 255)
				{
					((TextBox)sender).setText("255"); //$NON-NLS-1$
				}
				if (Integer.parseInt(((TextBox)sender).getText()) < 0)
				{
					((TextBox)sender).setText("0"); //$NON-NLS-1$
				}
			}
			catch (Exception e)
			{
			}

			red = Integer.parseInt(tbRed.getText());
			green = Integer.parseInt(tbGreen.getText());
			blue = Integer.parseInt(tbBlue.getText());
			hue = Integer.parseInt(tbHue.getText());
			saturation = Integer.parseInt(tbSaturation.getText());
			brightness = Integer.parseInt(tbBrightness.getText());

			// Figure out the colors
			try
			{
				Color color = new Color();
				color.setRGB(red, green, blue);
				tbHue.setText(Integer.toString(color.getHue()));
				tbSaturation.setText(Integer.toString(color.getSaturation()));
				tbBrightness.setText(Integer.toString(color.getValue()));
				tbHexColor.setText(color.getHex());
				setPreview(color.getHex());
			}
			catch (Exception e)
			{
			}
		}
		else if (sender == tbHue || sender == tbSaturation || sender == tbBrightness)
		{
			// Don't allow this value to overflow
			try
			{
				if (Integer.parseInt(tbHue.getText()) > 359)
				{
					tbHue.setText("359"); //$NON-NLS-1$
				}

				if (Integer.parseInt(tbSaturation.getText()) > 100)
				{
					tbSaturation.setText("100"); //$NON-NLS-1$
				}

				if (Integer.parseInt(tbBrightness.getText()) > 100)
				{
					tbBrightness.setText("100"); //$NON-NLS-1$
				}
			}
			catch (Exception e)
			{
			}

			red = Integer.parseInt(tbRed.getText());
			green = Integer.parseInt(tbGreen.getText());
			blue = Integer.parseInt(tbBlue.getText());
			hue = Integer.parseInt(tbHue.getText());
			saturation = Integer.parseInt(tbSaturation.getText());
			brightness = Integer.parseInt(tbBrightness.getText());

			// Figure out colors
			try
			{
				Color color = new Color();
				color.setHSV(hue, saturation, brightness);
				tbRed.setText(Integer.toString(color.getRed()));
				tbGreen.setText(Integer.toString(color.getGreen()));
				tbBlue.setText(Integer.toString(color.getBlue()));
				tbHexColor.setText(color.getHex());
				setPreview(color.getHex());
			}
			catch (Exception e)
			{
			}
		}

		// Let the sliders know something's changed
		updateSliders();
	}

	/**
	 * Fired when a keyboard action generates a character. This occurs after onKeyDown and onKeyUp are fired for the physical key that was pressed.
	 *
	 * It should be noted that many browsers do not generate keypress events for non-printing keyCode values, such as KEY_ENTER or arrow keys. These keyCodes can be reliably captured either with onKeyDown(Widget, char, int) or onKeyUp(Widget, char, int).
	 *
	 * Subclasses that override this method must call <tt>super.onKeyPress(sender, keyCode, modifiers)</tt> to ensure that the Widget recieves its events.
	 * @param sender the widget that was focused when the event occurred.
	 * @param keyCode the Unicode character that was generated by the keyboard action.
	 * @param modifiers the modifier keys pressed at when the event occurred. This value is a combination of the bits defined by MODIFIER_SHIFT, MODIFIER_CTRL, and MODIFIER_ALT.
	 * @see com.google.gwt.user.client.ui.KeyboardListener
	 */
	public void onKeyPress(Widget sender, char keyCode, int modifiers)
	{
        if (sender == tbHexColor)
        {
			// Disallow non-hex in hexadecimal boxes
			if ((!Character.isDigit(keyCode))
				&& (keyCode != 'A') && (keyCode != 'a')
				&& (keyCode != 'B') && (keyCode != 'b')
				&& (keyCode != 'C') && (keyCode != 'c')
				&& (keyCode != 'D') && (keyCode != 'd')
				&& (keyCode != 'E') && (keyCode != 'e')
				&& (keyCode != 'F') && (keyCode != 'f')
				&& (keyCode != (char) KeyboardListener.KEY_TAB)
				&& (keyCode != (char) KeyboardListener.KEY_BACKSPACE)
				&& (keyCode != (char) KeyboardListener.KEY_DELETE) && (keyCode != (char) KeyboardListener.KEY_ENTER)
				&& (keyCode != (char) KeyboardListener.KEY_HOME) && (keyCode != (char) KeyboardListener.KEY_END)
				&& (keyCode != (char) KeyboardListener.KEY_LEFT) && (keyCode != (char) KeyboardListener.KEY_UP)
				&& (keyCode != (char) KeyboardListener.KEY_RIGHT) && (keyCode != (char) KeyboardListener.KEY_DOWN))
			{
				((TextBox)sender).cancelKey();
			}
		}
		else
		{
			// Disallow non-numerics in numeric boxes
			if ((!Character.isDigit(keyCode))
				&& (keyCode != (char) KeyboardListener.KEY_TAB)
				&& (keyCode != (char) KeyboardListener.KEY_BACKSPACE)
				&& (keyCode != (char) KeyboardListener.KEY_DELETE) && (keyCode != (char) KeyboardListener.KEY_ENTER)
				&& (keyCode != (char) KeyboardListener.KEY_HOME) && (keyCode != (char) KeyboardListener.KEY_END)
				&& (keyCode != (char) KeyboardListener.KEY_LEFT) && (keyCode != (char) KeyboardListener.KEY_UP)
				&& (keyCode != (char) KeyboardListener.KEY_RIGHT) && (keyCode != (char) KeyboardListener.KEY_DOWN))
			{
				((TextBox)sender).cancelKey();
			}
		}
	}

	/**
	 * Fired when the user releases a physical key.
	 *
	 * Subclasses that override this method must call <tt>super.onKeyUp(sender, keyCode, modifiers)</tt> to ensure that the Widget recieves its events.
	 * @param sender the widget that was focused when the event occurred.
	 * @param keyCode the physical key that was released. Constants for this value are defined in this interface with the KEY prefix.
	 * @param modifiers the modifier keys pressed at when the event occurred. This value is a combination of the bits defined by MODIFIER_SHIFT, MODIFIER_CTRL, and MODIFIER_ALT.
	 * @see com.google.gwt.user.client.ui.KeyboardListener
	 */
	public void onKeyUp(Widget sender, char keyCode, int modifiers)
	{
		onChange(sender);
	}

	/**
	 * Fired when the user depresses a physical key.
	 *
	 * Subclasses that override this method must call <tt>super.onKeyDown(sender, keyCode, modifiers)</tt> to ensure that the Widget recieves its events.
	 * @param sender the widget that was focused when the event occurred.
	 * @param keyCode the physical key that was depressed. Constants for this value are defined in this interface with the KEY prefix.
	 * @param modifiers he modifier keys pressed at when the event occurred. This value is a combination of the bits defined by MODIFIER_SHIFT, MODIFIER_CTRL, and MODIFIER_ALT.
	 * @see com.google.gwt.user.client.ui.KeyboardListener
	 */
	public void onKeyDown(Widget sender, char keyCode, int modifiers) { }

	/**
	 * Sets the Red, Green, and Blue color variables. This will automatically populate the Hue, Saturation and Brightness and Hexadecimal fields, too.
	 *
	 * The RGB color model is an additive color model in which red, green, and blue light are added together in various ways to reproduce a broad array of colors. The name of the model comes from the initials of the three additive primary colors, red, green, and blue.
	 * @param red strength - valid range is 0-255
	 * @param green strength - valid range is 0-255
	 * @param blue strength - valid range is 0-255
	 * @throws java.lang.Exception Exception if the Red, Green or Blue variables are out of range.
	 */
	public void setRGB(int red, int green, int blue) throws Exception
	{
		Color color = new Color();
		color.setRGB(red, green, blue);

		this.red = red;
		this.green = green;
		this.blue = blue;
		this.hue = color.getHue();
		this.saturation = color.getSaturation();
		this.brightness = color.getValue();

		tbRed.setText(Integer.toString(this.red));
		tbGreen.setText(Integer.toString(this.green));
		tbBlue.setText(Integer.toString(this.blue));
		tbHue.setText(Integer.toString(this.hue));
		tbSaturation.setText(Integer.toString(this.saturation));
		tbBrightness.setText(Integer.toString(this.brightness));
		tbHexColor.setText(color.getHex());
		setPreview(color.getHex());

		updateSliders();
	}

	/**
	 * Set the Hue, Saturation and Value (Brightness) variables. This will automatically populate the Red, Green, Blue, and Hexadecimal fields, too.
	 *
	 * HSV represents points in the RGB color space, which attempt to describe perceptual color relationships more accurately than RGB. HSV describes colors as points in a cylinder whose central axis ranges from black at the bottom to white at the top with neutral colors between them, where angle around the axis corresponds to hue, distance from the axis corresponds to saturation, and distance along the axis corresponds to lightness, value, or brightness.
	 * @param hue angle - valid range is 0-359
	 * @param saturation percent - valid range is 0-100
	 * @param value percent (Brightness) - valid range is 0-100
	 * @throws java.lang.Exception A general exception if the Hue, Saturation, or Value variables are out of range.
	 */
	public void setHSV(int hue, int saturation, int value) throws Exception
	{
		Color color = new Color();
		color.setHSV(hue, saturation, value);

		this.red = color.getRed();
		this.green = color.getGreen();
		this.blue = color.getBlue();
		this.hue = hue;
		this.saturation = saturation;
		this.brightness = value;

		tbRed.setText(Integer.toString(this.red));
		tbGreen.setText(Integer.toString(this.green));
		tbBlue.setText(Integer.toString(this.blue));
		tbHue.setText(Integer.toString(this.hue));
		tbSaturation.setText(Integer.toString(this.saturation));
		tbBrightness.setText(Integer.toString(this.brightness));
		tbHexColor.setText(color.getHex());
		setPreview(color.getHex());

		updateSliders();
	}

	/**
	 * Sets the hexadecimal notation for Red, Green, and Blue. This will automatically populate all the other fields, too.
	 * @param hex Hexadecimal notation of Red, Green and Blue in the range of 000000-FFFFFF
	 * @throws java.lang.Exception A generic exception if the hexadecimal notation is bad.
	 */
	public void setHex(String hex) throws Exception
	{
    if (hex.startsWith("#")) { //$NON-NLS-1$
      hex = hex.substring(1);
    }
	  
		Color color = new Color();
		color.setHex(hex);

		this.red = color.getRed();
		this.green = color.getGreen();
		this.blue = color.getBlue();
		this.hue = color.getHue();
		this.saturation = color.getSaturation();
		this.brightness = color.getValue();

		tbRed.setText(Integer.toString(this.red));
		tbGreen.setText(Integer.toString(this.green));
		tbBlue.setText(Integer.toString(this.blue));
		tbHue.setText(Integer.toString(this.hue));
		tbSaturation.setText(Integer.toString(this.saturation));
		tbBrightness.setText(Integer.toString(this.brightness));
		tbHexColor.setText(color.getHex());
		setPreview(color.getHex());

		updateSliders();
	}

	/**
	 * Returns the hexadecimal notation of the current selected color.
	 * @return Hexadecimal in the range of 000000-FFFFFF
	 */
	public String getHexColor()
	{
		return tbHexColor.getText();
	}

	/*Helper functions -- for common calculations
	 */
	/**
	 * Divides the first value by 256, then multiplies it by the second value.
	 * @param val1 first value.
	 * @param val2 second value.
	 * @return result.
	 */
	private int percentOf(int val1, int val2)
	{
	 	return (int)(new Float(val1).floatValue() / 256 * val2);
	}

	/**
	 * Called whenever the internal state has been changed and needs to synchronize the other components.
	 */
	private void updateSliders()
	{
		// Let the sliders know something's changed
		if (rbHue.isChecked()) onClick(rbHue);
		if (rbSaturation.isChecked()) onClick(rbSaturation);
		if (rbBrightness.isChecked()) onClick(rbBrightness);
		if (rbRed.isChecked()) onClick(rbRed);
		if (rbGreen.isChecked()) onClick(rbGreen);
		if (rbBlue.isChecked()) onClick(rbBlue);
	}
}