/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.InvalidFamilyOperationException;
import org.apache.hadoop.hbase.RemoteExceptionHandler;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.backup.HFileArchiver;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.SplitLogManager;
import org.apache.hadoop.hbase.master.metrics.MasterMetrics;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.wal.HLog;
import org.apache.hadoop.hbase.regionserver.wal.HLogSplitter;
import org.apache.hadoop.hbase.regionserver.wal.OrphanHLogAfterSplitException;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FSTableDescriptors;
import org.apache.hadoop.hbase.util.FSUtils;

public class MasterFileSystem {
    private static final Log LOG = LogFactory.getLog((String)MasterFileSystem.class.getName());
    Configuration conf;
    Server master;
    MasterMetrics metrics;
    private String clusterId;
    private final FileSystem fs;
    private volatile boolean fsOk = true;
    private final Path oldLogDir;
    private final Path rootdir;
    private final Path tempdir;
    final Lock splitLogLock = new ReentrantLock();
    final boolean distributedLogSplitting;
    final SplitLogManager splitLogManager;
    private final MasterServices services;
    private static final PathFilter META_FILTER = new PathFilter(){

        public boolean accept(Path p) {
            return HLog.isMetaFile(p);
        }
    };
    private static final PathFilter NON_META_FILTER = new PathFilter(){

        public boolean accept(Path p) {
            return !HLog.isMetaFile(p);
        }
    };

    public MasterFileSystem(Server master, MasterServices services, MasterMetrics metrics, boolean masterRecovery) throws IOException {
        this.conf = master.getConfiguration();
        this.master = master;
        this.services = services;
        this.metrics = metrics;
        this.rootdir = FSUtils.getRootDir(this.conf);
        this.tempdir = new Path(this.rootdir, ".tmp");
        this.fs = this.rootdir.getFileSystem(this.conf);
        String fsUri = this.fs.getUri().toString();
        this.conf.set("fs.default.name", fsUri);
        this.conf.set("fs.defaultFS", fsUri);
        this.fs.setConf(this.conf);
        this.distributedLogSplitting = this.conf.getBoolean("hbase.master.distributed.log.splitting", true);
        if (this.distributedLogSplitting) {
            this.splitLogManager = new SplitLogManager(master.getZooKeeper(), master.getConfiguration(), master, this.services, master.getServerName().toString());
            this.splitLogManager.finishInitialization(masterRecovery);
        } else {
            this.splitLogManager = null;
        }
        this.oldLogDir = this.createInitialFileSystemLayout();
    }

    private Path createInitialFileSystemLayout() throws IOException {
        this.checkRootDir(this.rootdir, this.conf, this.fs);
        this.checkTempDir(this.tempdir, this.conf, this.fs);
        Path oldLogDir = new Path(this.rootdir, ".oldlogs");
        if (!this.fs.exists(oldLogDir)) {
            this.fs.mkdirs(oldLogDir);
        }
        return oldLogDir;
    }

    public FileSystem getFileSystem() {
        return this.fs;
    }

    public Path getOldLogDir() {
        return this.oldLogDir;
    }

    public boolean checkFileSystem() {
        if (this.fsOk) {
            try {
                FSUtils.checkFileSystemAvailable(this.fs);
                FSUtils.checkDfsSafeMode(this.conf);
            }
            catch (IOException e) {
                this.master.abort("Shutting down HBase cluster: file system not available", e);
                this.fsOk = false;
            }
        }
        return this.fsOk;
    }

    public Path getRootDir() {
        return this.rootdir;
    }

    public Path getTempDir() {
        return this.tempdir;
    }

    public String getClusterId() {
        return this.clusterId;
    }

    void splitLogAfterStartup() {
        boolean retrySplitting = !this.conf.getBoolean("hbase.hlog.split.skip.errors", false);
        Path logsDirPath = new Path(this.rootdir, ".logs");
        do {
            if (this.master.isStopped()) {
                LOG.warn((Object)"Master stopped while splitting logs");
                break;
            }
            ArrayList<ServerName> serverNames = new ArrayList<ServerName>();
            try {
                if (!this.fs.exists(logsDirPath)) {
                    return;
                }
                FileStatus[] logFolders = FSUtils.listStatus(this.fs, logsDirPath, null);
                Set<ServerName> onlineServers = ((HMaster)this.master).getServerManager().getOnlineServers().keySet();
                if (logFolders == null || logFolders.length == 0) {
                    LOG.debug((Object)"No log files to split, proceeding...");
                    return;
                }
                for (FileStatus status : logFolders) {
                    ServerName serverName;
                    String sn = status.getPath().getName();
                    if (sn.endsWith("-splitting")) {
                        sn = sn.substring(0, sn.length() - "-splitting".length());
                    }
                    if (!onlineServers.contains(serverName = ServerName.parseServerName(sn))) {
                        LOG.info((Object)("Log folder " + status.getPath() + " doesn't belong " + "to a known region server, splitting"));
                        serverNames.add(serverName);
                        continue;
                    }
                    LOG.info((Object)("Log folder " + status.getPath() + " belongs to an existing region server"));
                }
                if (this.services.shouldSplitMetaSeparately()) {
                    this.splitLog(serverNames, META_FILTER);
                    this.splitLog(serverNames, NON_META_FILTER);
                } else {
                    this.splitAllLogs(serverNames);
                }
                retrySplitting = false;
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Failed splitting of " + serverNames), (Throwable)ioe);
                if (!this.checkFileSystem()) {
                    LOG.warn((Object)"Bad Filesystem, exiting");
                    Runtime.getRuntime().halt(1);
                }
                try {
                    if (!retrySplitting) continue;
                    Thread.sleep(this.conf.getInt("hbase.hlog.split.failure.retry.interval", 30000));
                }
                catch (InterruptedException e) {
                    LOG.warn((Object)"Interrupted, aborting since cannot return w/o splitting");
                    Thread.currentThread().interrupt();
                    retrySplitting = false;
                    Runtime.getRuntime().halt(1);
                }
            }
        } while (retrySplitting);
    }

    public void splitLog(ServerName serverName) throws IOException {
        ArrayList<ServerName> serverNames = new ArrayList<ServerName>();
        serverNames.add(serverName);
        this.splitLog(serverNames);
    }

    public void splitAllLogs(ServerName serverName) throws IOException {
        ArrayList<ServerName> serverNames = new ArrayList<ServerName>();
        serverNames.add(serverName);
        this.splitAllLogs(serverNames);
    }

    public void splitMetaLog(ServerName serverName) throws IOException {
        long splitTime = 0L;
        long splitLogSize = 0L;
        ArrayList<ServerName> serverNames = new ArrayList<ServerName>();
        serverNames.add(serverName);
        List<Path> logDirs = this.getLogDirs(serverNames);
        if (logDirs.isEmpty()) {
            LOG.info((Object)"No meta logs to split");
            return;
        }
        this.splitLogManager.handleDeadWorkers(serverNames);
        splitTime = EnvironmentEdgeManager.currentTimeMillis();
        splitLogSize = this.splitLogManager.splitLogDistributed(logDirs, META_FILTER);
        splitTime = EnvironmentEdgeManager.currentTimeMillis() - splitTime;
        if (this.metrics != null) {
            this.metrics.addSplit(splitTime, splitLogSize);
        }
    }

    private List<Path> getLogDirs(List<ServerName> serverNames) throws IOException {
        ArrayList<Path> logDirs = new ArrayList<Path>();
        for (ServerName serverName : serverNames) {
            Path logDir = new Path(this.rootdir, HLog.getHLogDirectoryName(serverName.toString()));
            Path splitDir = logDir.suffix("-splitting");
            if (this.fs.exists(logDir)) {
                if (!this.fs.rename(logDir, splitDir)) {
                    throw new IOException("Failed fs.rename for log split: " + logDir);
                }
                logDir = splitDir;
                LOG.debug((Object)("Renamed region directory: " + splitDir));
            } else if (!this.fs.exists(splitDir)) {
                LOG.info((Object)("Log dir for server " + serverName + " does not exist"));
                continue;
            }
            logDirs.add(splitDir);
        }
        return logDirs;
    }

    public void splitLog(List<ServerName> serverNames) throws IOException {
        this.splitLog(serverNames, NON_META_FILTER);
    }

    public void splitAllLogs(List<ServerName> serverNames) throws IOException {
        this.splitLog(serverNames, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void splitLog(List<ServerName> serverNames, PathFilter filter) throws IOException {
        long splitTime = 0L;
        long splitLogSize = 0L;
        List<Path> logDirs = this.getLogDirs(serverNames);
        if (logDirs.isEmpty()) {
            LOG.info((Object)"No logs to split");
            return;
        }
        if (this.distributedLogSplitting) {
            this.splitLogManager.handleDeadWorkers(serverNames);
            splitTime = EnvironmentEdgeManager.currentTimeMillis();
            splitLogSize = this.splitLogManager.splitLogDistributed(logDirs, filter);
            splitTime = EnvironmentEdgeManager.currentTimeMillis() - splitTime;
        } else {
            for (Path logDir : logDirs) {
                this.splitLogLock.lock();
                try {
                    HLogSplitter splitter = HLogSplitter.createLogSplitter(this.conf, this.rootdir, logDir, this.oldLogDir, this.fs);
                    try {
                        FSUtils.waitOnSafeMode(this.conf, this.conf.getInt("hbase.server.thread.wakefrequency", 1000));
                        splitter.splitLog();
                    }
                    catch (OrphanHLogAfterSplitException e) {
                        LOG.warn((Object)"Retrying splitting because of:", (Throwable)e);
                        splitter = HLogSplitter.createLogSplitter(this.conf, this.rootdir, logDir, this.oldLogDir, this.fs);
                        splitter.splitLog();
                    }
                    splitTime = splitter.getTime();
                    splitLogSize = splitter.getSize();
                }
                finally {
                    this.splitLogLock.unlock();
                }
            }
        }
        if (this.metrics != null) {
            this.metrics.addSplit(splitTime, splitLogSize);
        }
    }

    private Path checkRootDir(Path rd, Configuration c, FileSystem fs) throws IOException {
        FSUtils.waitOnSafeMode(c, c.getInt("hbase.server.thread.wakefrequency", 10000));
        try {
            if (!fs.exists(rd)) {
                fs.mkdirs(rd);
                FSUtils.setVersion(fs, rd, c.getInt("hbase.server.thread.wakefrequency", 10000), c.getInt("hbase.server.versionfile.writeattempts", 3));
            } else {
                if (!fs.isDirectory(rd)) {
                    throw new IllegalArgumentException(rd.toString() + " is not a directory");
                }
                FSUtils.checkVersion(fs, rd, true, c.getInt("hbase.server.thread.wakefrequency", 10000), c.getInt("hbase.server.versionfile.writeattempts", 3));
            }
        }
        catch (IllegalArgumentException iae) {
            LOG.fatal((Object)("Please fix invalid configuration for hbase.rootdir " + rd.toString()), (Throwable)iae);
            throw iae;
        }
        if (!FSUtils.checkClusterIdExists(fs, rd, c.getInt("hbase.server.thread.wakefrequency", 10000))) {
            FSUtils.setClusterId(fs, rd, UUID.randomUUID().toString(), c.getInt("hbase.server.thread.wakefrequency", 10000));
        }
        this.clusterId = FSUtils.getClusterId(fs, rd);
        if (!FSUtils.rootRegionExists(fs, rd)) {
            MasterFileSystem.bootstrap(rd, c);
        }
        this.createRootTableInfo(rd);
        return rd;
    }

    private void createRootTableInfo(Path rd) throws IOException {
        if (!FSTableDescriptors.isTableInfoExists(this.fs, rd, Bytes.toString(HRegionInfo.ROOT_REGIONINFO.getTableName()))) {
            FSTableDescriptors.createTableDescriptor(HTableDescriptor.ROOT_TABLEDESC, this.conf);
        }
    }

    private void checkTempDir(Path tmpdir, Configuration c, FileSystem fs) throws IOException {
        if (fs.exists(tmpdir)) {
            for (Path tabledir : FSUtils.getTableDirs(fs, tmpdir)) {
                for (Path regiondir : FSUtils.getRegionDirs(fs, tabledir)) {
                    HFileArchiver.archiveRegion(fs, this.rootdir, tabledir, regiondir);
                }
            }
            if (!fs.delete(tmpdir, true)) {
                throw new IOException("Unable to clean the temp directory: " + tmpdir);
            }
        }
        if (!fs.mkdirs(tmpdir)) {
            throw new IOException("HBase temp directory '" + tmpdir + "' creation failure.");
        }
    }

    private static void bootstrap(Path rd, Configuration c) throws IOException {
        LOG.info((Object)"BOOTSTRAP: creating ROOT and first META regions");
        try {
            HRegionInfo rootHRI = new HRegionInfo(HRegionInfo.ROOT_REGIONINFO);
            MasterFileSystem.setInfoFamilyCachingForRoot(false);
            HRegionInfo metaHRI = new HRegionInfo(HRegionInfo.FIRST_META_REGIONINFO);
            MasterFileSystem.setInfoFamilyCachingForMeta(false);
            HRegion root = HRegion.createHRegion(rootHRI, rd, c, HTableDescriptor.ROOT_TABLEDESC);
            HRegion meta = HRegion.createHRegion(metaHRI, rd, c, HTableDescriptor.META_TABLEDESC);
            MasterFileSystem.setInfoFamilyCachingForRoot(true);
            MasterFileSystem.setInfoFamilyCachingForMeta(true);
            HRegion.addRegionToMETA(root, meta);
            root.close();
            root.getLog().closeAndDelete();
            meta.close();
            meta.getLog().closeAndDelete();
        }
        catch (IOException e) {
            e = RemoteExceptionHandler.checkIOException(e);
            LOG.error((Object)"bootstrap", (Throwable)e);
            throw e;
        }
    }

    public static void setInfoFamilyCachingForRoot(boolean b) {
        for (HColumnDescriptor hcd : HTableDescriptor.ROOT_TABLEDESC.getColumnFamilies()) {
            if (!Bytes.equals(hcd.getName(), HConstants.CATALOG_FAMILY)) continue;
            hcd.setBlockCacheEnabled(b);
            hcd.setInMemory(b);
        }
    }

    public static void setInfoFamilyCachingForMeta(boolean b) {
        for (HColumnDescriptor hcd : HTableDescriptor.META_TABLEDESC.getColumnFamilies()) {
            if (!Bytes.equals(hcd.getName(), HConstants.CATALOG_FAMILY)) continue;
            hcd.setBlockCacheEnabled(b);
            hcd.setInMemory(b);
        }
    }

    public void deleteRegion(HRegionInfo region) throws IOException {
        HFileArchiver.archiveRegion(this.conf, this.fs, region);
    }

    public void deleteTable(byte[] tableName) throws IOException {
        this.fs.delete(new Path(this.rootdir, Bytes.toString(tableName)), true);
    }

    public Path moveToTemp(Path path) throws IOException {
        Path tempPath = new Path(this.tempdir, path.getName());
        if (!this.fs.exists(this.tempdir) && !this.fs.mkdirs(this.tempdir)) {
            throw new IOException("HBase temp directory '" + this.tempdir + "' creation failure.");
        }
        if (!this.fs.rename(path, tempPath)) {
            throw new IOException("Unable to move '" + path + "' to temp '" + tempPath + "'");
        }
        return tempPath;
    }

    public Path moveTableToTemp(byte[] tableName) throws IOException {
        return this.moveToTemp(HTableDescriptor.getTableDir(this.rootdir, tableName));
    }

    public void updateRegionInfo(HRegionInfo region) {
    }

    public void deleteFamilyFromFS(HRegionInfo region, byte[] familyName) throws IOException {
        Path tableDir = new Path(this.rootdir, region.getTableNameAsString());
        HFileArchiver.archiveFamily(this.fs, this.conf, region, tableDir, familyName);
        Path familyDir = new Path(tableDir, new Path(region.getEncodedName(), Bytes.toString(familyName)));
        if (!this.fs.delete(familyDir, true)) {
            throw new IOException("Could not delete family " + Bytes.toString(familyName) + " from FileSystem for region " + region.getRegionNameAsString() + "(" + region.getEncodedName() + ")");
        }
    }

    public void stop() {
        if (this.splitLogManager != null) {
            this.splitLogManager.stop();
        }
    }

    public void createTableDescriptor(HTableDescriptor htableDescriptor) throws IOException {
        FSTableDescriptors.createTableDescriptor(htableDescriptor, this.conf);
    }

    public HTableDescriptor deleteColumn(byte[] tableName, byte[] familyName) throws IOException {
        LOG.info((Object)("DeleteColumn. Table = " + Bytes.toString(tableName) + " family = " + Bytes.toString(familyName)));
        HTableDescriptor htd = this.services.getTableDescriptors().get(tableName);
        htd.removeFamily(familyName);
        this.services.getTableDescriptors().add(htd);
        return htd;
    }

    public HTableDescriptor modifyColumn(byte[] tableName, HColumnDescriptor hcd) throws IOException {
        LOG.info((Object)("AddModifyColumn. Table = " + Bytes.toString(tableName) + " HCD = " + hcd.toString()));
        HTableDescriptor htd = this.services.getTableDescriptors().get(tableName);
        byte[] familyName = hcd.getName();
        if (!htd.hasFamily(familyName)) {
            throw new InvalidFamilyOperationException("Family '" + Bytes.toString(familyName) + "' doesn't exists so cannot be modified");
        }
        htd.addFamily(hcd);
        this.services.getTableDescriptors().add(htd);
        return htd;
    }

    public HTableDescriptor addColumn(byte[] tableName, HColumnDescriptor hcd) throws IOException {
        LOG.info((Object)("AddColumn. Table = " + Bytes.toString(tableName) + " HCD = " + hcd.toString()));
        HTableDescriptor htd = this.services.getTableDescriptors().get(tableName);
        if (htd == null) {
            throw new InvalidFamilyOperationException("Family '" + hcd.getNameAsString() + "' cannot be modified as HTD is null");
        }
        htd.addFamily(hcd);
        this.services.getTableDescriptors().add(htd);
        return htd;
    }
}

