/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.handler;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.catalog.CatalogTracker;
import org.apache.hadoop.hbase.catalog.MetaEditor;
import org.apache.hadoop.hbase.catalog.MetaReader;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.executor.EventHandler;
import org.apache.hadoop.hbase.master.AssignmentManager;
import org.apache.hadoop.hbase.master.DeadServer;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.zookeeper.ZKAssign;
import org.apache.zookeeper.KeeperException;

public class ServerShutdownHandler
extends EventHandler {
    private static final Log LOG = LogFactory.getLog(ServerShutdownHandler.class);
    protected final ServerName serverName;
    protected final MasterServices services;
    protected final DeadServer deadServers;
    protected final boolean shouldSplitHlog;

    public ServerShutdownHandler(Server server, MasterServices services, DeadServer deadServers, ServerName serverName, boolean shouldSplitHlog) {
        this(server, services, deadServers, serverName, EventHandler.EventType.M_SERVER_SHUTDOWN, shouldSplitHlog);
    }

    ServerShutdownHandler(Server server, MasterServices services, DeadServer deadServers, ServerName serverName, EventHandler.EventType type, boolean shouldSplitHlog) {
        super(server, type);
        this.serverName = serverName;
        this.server = server;
        this.services = services;
        this.deadServers = deadServers;
        if (!this.deadServers.contains(this.serverName)) {
            LOG.warn((Object)(this.serverName + " is NOT in deadservers; it should be!"));
        }
        this.shouldSplitHlog = shouldSplitHlog;
    }

    @Override
    public String getInformativeName() {
        if (this.serverName != null) {
            return this.getClass().getSimpleName() + " for " + this.serverName;
        }
        return super.getInformativeName();
    }

    boolean isCarryingRoot() {
        return false;
    }

    boolean isCarryingMeta() {
        return false;
    }

    @Override
    public String toString() {
        String name = "UnknownServerName";
        if (this.server != null && this.server.getServerName() != null) {
            name = this.server.getServerName().toString();
        }
        return this.getClass().getSimpleName() + "-" + name + "-" + this.getSeqid();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void process() throws IOException {
        ServerName serverName = this.serverName;
        try {
            try {
                if (this.shouldSplitHlog) {
                    LOG.info((Object)("Splitting logs for " + serverName));
                    this.services.getMasterFileSystem().splitLog(serverName);
                } else {
                    LOG.info((Object)("Skipping log splitting for " + serverName));
                }
            }
            catch (IOException ioe) {
                this.services.getExecutorService().submit(this);
                this.deadServers.add(serverName);
                throw new IOException("failed log splitting for " + serverName + ", will retry", ioe);
            }
            if (this.isCarryingRoot() || this.isCarryingMeta()) {
                this.services.getExecutorService().submit(new ServerShutdownHandler(this.server, this.services, this.deadServers, serverName, false));
                this.deadServers.add(serverName);
                return;
            }
            NavigableMap<HRegionInfo, Result> hris = null;
            while (!this.server.isStopped()) {
                try {
                    this.server.getCatalogTracker().waitForMeta();
                    hris = MetaReader.getServerUserRegions(this.server.getCatalogTracker(), this.serverName);
                    break;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new IOException("Interrupted", e);
                }
                catch (IOException ioe) {
                    LOG.info((Object)("Received exception accessing META during server shutdown of " + serverName + ", retrying META read"), (Throwable)ioe);
                }
            }
            Pair<Set<HRegionInfo>, List<AssignmentManager.RegionState>> p = this.services.getAssignmentManager().processServerShutdown(this.serverName);
            Set<HRegionInfo> ritsGoingToServer = p.getFirst();
            List<AssignmentManager.RegionState> ritsOnServer = p.getSecond();
            List<HRegionInfo> regionsToAssign = this.getRegionsToAssign(hris, ritsOnServer, ritsGoingToServer);
            for (HRegionInfo hri : ritsGoingToServer) {
                if (this.services.getAssignmentManager().isRegionAssigned(hri) || regionsToAssign.contains((Object)hri)) continue;
                regionsToAssign.add(hri);
            }
            for (HRegionInfo hri : regionsToAssign) {
                this.services.getAssignmentManager().assign(hri, true);
            }
            LOG.info((Object)(regionsToAssign.size() + " regions which were planned to open on " + this.serverName + " have been re-assigned."));
        }
        finally {
            this.deadServers.finish(serverName);
        }
        LOG.info((Object)("Finished processing of shutdown of " + serverName));
    }

    private List<HRegionInfo> getRegionsToAssign(NavigableMap<HRegionInfo, Result> metaHRIs, List<AssignmentManager.RegionState> ritsOnServer, Set<HRegionInfo> ritsGoingToServer) throws IOException {
        List<HRegionInfo> toAssign = new ArrayList<HRegionInfo>();
        if (metaHRIs == null || metaHRIs.isEmpty()) {
            return toAssign;
        }
        for (AssignmentManager.RegionState regionState : ritsOnServer) {
            if (regionState.isClosing() || regionState.isPendingClose() || regionState.isSplitting()) continue;
            LOG.debug((Object)("Removed " + regionState.getRegion().getRegionNameAsString() + " from list of regions to assign because region state: " + (Object)((Object)regionState.getState())));
            metaHRIs.remove((Object)regionState.getRegion());
        }
        for (Map.Entry entry : metaHRIs.entrySet()) {
            AssignmentManager.RegionState rit = (AssignmentManager.RegionState)this.services.getAssignmentManager().getRegionsInTransition().get(((HRegionInfo)((Object)entry.getKey())).getEncodedName());
            AssignmentManager assignmentManager = this.services.getAssignmentManager();
            if (ServerShutdownHandler.processDeadRegion((HRegionInfo)((Object)entry.getKey()), (Result)entry.getValue(), assignmentManager, this.server.getCatalogTracker())) {
                ServerName addressFromAM = assignmentManager.getRegionServerOfRegion((HRegionInfo)((Object)entry.getKey()));
                if (!(rit == null || rit.isClosing() || rit.isPendingClose() || rit.isSplitting() || ritsGoingToServer.contains(entry.getKey()))) {
                    LOG.info((Object)("Skip assigning region " + rit.toString()));
                } else if (addressFromAM != null && !addressFromAM.equals(this.serverName)) {
                    LOG.debug((Object)("Skip assigning region " + ((HRegionInfo)((Object)entry.getKey())).getRegionNameAsString() + " because it has been opened in " + addressFromAM.getServerName()));
                    ritsGoingToServer.remove(entry.getKey());
                } else {
                    if (rit != null) {
                        try {
                            LOG.info((Object)("Reassigning region with rs =" + rit + " and deleting zk node if exists"));
                            ZKAssign.deleteNodeFailSilent(this.services.getZooKeeper(), (HRegionInfo)((Object)entry.getKey()));
                        }
                        catch (KeeperException ke) {
                            this.server.abort("Unexpected ZK exception deleting unassigned node " + entry.getKey(), ke);
                            return null;
                        }
                    }
                    toAssign.add((HRegionInfo)((Object)entry.getKey()));
                }
            } else if (rit != null && (rit.isSplitting() || rit.isSplit())) {
                HRegionInfo region = rit.getRegion();
                AssignmentManager am = assignmentManager;
                am.regionOffline(region);
                ritsGoingToServer.remove((Object)region);
            }
            toAssign = this.checkForDisablingOrDisabledTables(ritsGoingToServer, toAssign, rit, assignmentManager);
        }
        return toAssign;
    }

    private List<HRegionInfo> checkForDisablingOrDisabledTables(Set<HRegionInfo> regionsFromRIT, List<HRegionInfo> toAssign, AssignmentManager.RegionState rit, AssignmentManager assignmentManager) {
        if (rit == null) {
            return toAssign;
        }
        if (!rit.isClosing() && !rit.isPendingClose()) {
            return toAssign;
        }
        if (!assignmentManager.getZKTable().isDisablingOrDisabledTable(rit.getRegion().getTableNameAsString())) {
            return toAssign;
        }
        HRegionInfo hri = rit.getRegion();
        AssignmentManager am = assignmentManager;
        am.deleteClosingOrClosedNode(hri);
        am.regionOffline(hri);
        toAssign.remove((Object)hri);
        regionsFromRIT.remove((Object)hri);
        return toAssign;
    }

    public static boolean processDeadRegion(HRegionInfo hri, Result result, AssignmentManager assignmentManager, CatalogTracker catalogTracker) throws IOException {
        boolean tablePresent = assignmentManager.getZKTable().isTablePresent(hri.getTableNameAsString());
        if (!tablePresent) {
            LOG.info((Object)("The table " + hri.getTableNameAsString() + " was deleted.  Hence not proceeding."));
            return false;
        }
        boolean disabled = assignmentManager.getZKTable().isDisabledTable(hri.getTableNameAsString());
        if (disabled) {
            LOG.info((Object)("The table " + hri.getTableNameAsString() + " was disabled.  Hence not proceeding."));
            return false;
        }
        if (hri.isOffline() && hri.isSplit()) {
            LOG.debug((Object)("Offlined and split region " + hri.getRegionNameAsString() + "; checking daughter presence"));
            if (MetaReader.getRegion(catalogTracker, hri.getRegionName()) == null) {
                return false;
            }
            ServerShutdownHandler.fixupDaughters(result, assignmentManager, catalogTracker);
            return false;
        }
        boolean disabling = assignmentManager.getZKTable().isDisablingTable(hri.getTableNameAsString());
        if (disabling) {
            LOG.info((Object)("The table " + hri.getTableNameAsString() + " is disabled.  Hence not assigning region" + hri.getEncodedName()));
            return false;
        }
        return true;
    }

    public static int fixupDaughters(Result result, AssignmentManager assignmentManager, CatalogTracker catalogTracker) throws IOException {
        int fixedA = ServerShutdownHandler.fixupDaughter(result, HConstants.SPLITA_QUALIFIER, assignmentManager, catalogTracker);
        int fixedB = ServerShutdownHandler.fixupDaughter(result, HConstants.SPLITB_QUALIFIER, assignmentManager, catalogTracker);
        return fixedA + fixedB;
    }

    static int fixupDaughter(Result result, byte[] qualifier, AssignmentManager assignmentManager, CatalogTracker catalogTracker) throws IOException {
        HRegionInfo daughter = MetaReader.parseHRegionInfoFromCatalogResult(result, qualifier);
        if (daughter == null) {
            return 0;
        }
        if (ServerShutdownHandler.isDaughterMissing(catalogTracker, daughter)) {
            LOG.info((Object)("Fixup; missing daughter " + daughter.getRegionNameAsString()));
            MetaEditor.addDaughter(catalogTracker, daughter, null);
            assignmentManager.assign(daughter, true);
            return 1;
        }
        LOG.debug((Object)("Daughter " + daughter.getRegionNameAsString() + " present"));
        return 0;
    }

    private static boolean isDaughterMissing(CatalogTracker catalogTracker, HRegionInfo daughter) throws IOException {
        FindDaughterVisitor visitor = new FindDaughterVisitor(daughter);
        byte[] startrow = daughter.getRegionName();
        MetaReader.fullScan(catalogTracker, visitor, startrow);
        return !visitor.foundDaughter();
    }

    static class FindDaughterVisitor
    implements MetaReader.Visitor {
        private final HRegionInfo daughter;
        private boolean found = false;

        FindDaughterVisitor(HRegionInfo daughter) {
            this.daughter = daughter;
        }

        boolean foundDaughter() {
            return this.found;
        }

        @Override
        public boolean visit(Result r) throws IOException {
            HRegionInfo hri = MetaReader.parseHRegionInfoFromCatalogResult(r, HConstants.REGIONINFO_QUALIFIER);
            if (hri == null) {
                LOG.warn((Object)("No serialized HRegionInfo in " + r));
                return true;
            }
            byte[] value = r.getValue(HConstants.CATALOG_FAMILY, HConstants.SERVER_QUALIFIER);
            if (value == null) {
                return false;
            }
            if (!Bytes.equals(this.daughter.getTableName(), hri.getTableName())) {
                return false;
            }
            if (!Bytes.equals(this.daughter.getStartKey(), hri.getStartKey())) {
                return false;
            }
            this.found = true;
            return false;
        }
    }
}

