/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.net.InetAddresses;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.NavigableMap;
import java.util.Set;
import java.util.TreeMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.UnregisteredNodeException;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoUnderConstruction;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStatistics;
import org.apache.hadoop.hdfs.server.blockmanagement.DecommissionManager;
import org.apache.hadoop.hdfs.server.blockmanagement.HeartbeatManager;
import org.apache.hadoop.hdfs.server.blockmanagement.Host2NodesMap;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.hdfs.server.protocol.BalancerBandwidthCommand;
import org.apache.hadoop.hdfs.server.protocol.BlockCommand;
import org.apache.hadoop.hdfs.server.protocol.BlockRecoveryCommand;
import org.apache.hadoop.hdfs.server.protocol.DatanodeCommand;
import org.apache.hadoop.hdfs.server.protocol.DatanodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.DisallowedDatanodeException;
import org.apache.hadoop.hdfs.server.protocol.RegisterCommand;
import org.apache.hadoop.hdfs.util.CyclicIteration;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.net.CachedDNSToSwitchMapping;
import org.apache.hadoop.net.DNSToSwitchMapping;
import org.apache.hadoop.net.NetworkTopology;
import org.apache.hadoop.net.Node;
import org.apache.hadoop.net.ScriptBasedMapping;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.HostsFileReader;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class DatanodeManager {
    static final Log LOG = LogFactory.getLog(DatanodeManager.class);
    private final Namesystem namesystem;
    private final BlockManager blockManager;
    private final HeartbeatManager heartbeatManager;
    private final NavigableMap<String, DatanodeDescriptor> datanodeMap = new TreeMap<String, DatanodeDescriptor>();
    private final NetworkTopology networktopology = new NetworkTopology();
    private final Host2NodesMap host2DatanodeMap = new Host2NodesMap();
    private final DNSToSwitchMapping dnsToSwitchMapping;
    private final HostsFileReader hostsReader;
    private final long heartbeatExpireInterval;
    final int blockInvalidateLimit;
    private boolean hasClusterEverBeenMultiRack = false;
    private Daemon decommissionthread = null;

    DatanodeManager(BlockManager blockManager, Namesystem namesystem, Configuration conf) throws IOException {
        this.namesystem = namesystem;
        this.blockManager = blockManager;
        this.heartbeatManager = new HeartbeatManager(namesystem, blockManager, conf);
        this.hostsReader = new HostsFileReader(conf.get("dfs.hosts", ""), conf.get("dfs.hosts.exclude", ""));
        this.dnsToSwitchMapping = (DNSToSwitchMapping)ReflectionUtils.newInstance((Class)conf.getClass("net.topology.node.switch.mapping.impl", ScriptBasedMapping.class, DNSToSwitchMapping.class), (Configuration)conf);
        if (this.dnsToSwitchMapping instanceof CachedDNSToSwitchMapping) {
            this.dnsToSwitchMapping.resolve(new ArrayList(this.hostsReader.getHosts()));
        }
        long heartbeatIntervalSeconds = conf.getLong("dfs.heartbeat.interval", 3L);
        int heartbeatRecheckInterval = conf.getInt("dfs.namenode.heartbeat.recheck-interval", 300000);
        this.heartbeatExpireInterval = (long)(2 * heartbeatRecheckInterval) + 10000L * heartbeatIntervalSeconds;
        int blockInvalidateLimit = Math.max(20 * (int)heartbeatIntervalSeconds, 1000);
        this.blockInvalidateLimit = conf.getInt("dfs.block.invalidate.limit", blockInvalidateLimit);
        LOG.info((Object)("dfs.block.invalidate.limit=" + this.blockInvalidateLimit));
    }

    void activate(Configuration conf) {
        DecommissionManager dm;
        DecommissionManager decommissionManager = dm = new DecommissionManager(this.namesystem, this.blockManager);
        decommissionManager.getClass();
        this.decommissionthread = new Daemon((Runnable)new DecommissionManager.Monitor(decommissionManager, conf.getInt("dfs.namenode.decommission.interval", 30), conf.getInt("dfs.namenode.decommission.nodes.per.interval", 5)));
        this.decommissionthread.start();
        this.heartbeatManager.activate(conf);
    }

    void close() {
        if (this.decommissionthread != null) {
            this.decommissionthread.interrupt();
            try {
                this.decommissionthread.join(3000L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        this.heartbeatManager.close();
    }

    public NetworkTopology getNetworkTopology() {
        return this.networktopology;
    }

    HeartbeatManager getHeartbeatManager() {
        return this.heartbeatManager;
    }

    public DatanodeStatistics getDatanodeStatistics() {
        return this.heartbeatManager;
    }

    public void sortLocatedBlocks(String targethost, List<LocatedBlock> locatedblocks) {
        DatanodeDescriptor client = this.getDatanodeByHost(targethost);
        for (LocatedBlock b : locatedblocks) {
            this.networktopology.pseudoSortByDistance((Node)client, (Node[])b.getLocations());
            Arrays.sort(b.getLocations(), DFSUtil.DECOM_COMPARATOR);
        }
    }

    CyclicIteration<String, DatanodeDescriptor> getDatanodeCyclicIteration(String firstkey) {
        return new CyclicIteration<String, DatanodeDescriptor>(this.datanodeMap, firstkey);
    }

    public DatanodeDescriptor getDatanodeByHost(String host) {
        return this.host2DatanodeMap.getDatanodeByHost(host);
    }

    DatanodeDescriptor getDatanode(String storageID) {
        return (DatanodeDescriptor)this.datanodeMap.get(storageID);
    }

    public DatanodeDescriptor getDatanode(DatanodeID nodeID) throws UnregisteredNodeException {
        DatanodeDescriptor node = this.getDatanode(nodeID.getStorageID());
        if (node == null) {
            return null;
        }
        if (!node.getXferAddr().equals(nodeID.getXferAddr())) {
            UnregisteredNodeException e = new UnregisteredNodeException(nodeID, node);
            NameNode.stateChangeLog.fatal((Object)("BLOCK* NameSystem.getDatanode: " + e.getLocalizedMessage()));
            throw e;
        }
        return node;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void datanodeDump(PrintWriter out) {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            out.println("Metasave: Number of datanodes: " + this.datanodeMap.size());
            for (DatanodeDescriptor node : this.datanodeMap.values()) {
                out.println(node.dumpDatanode());
            }
        }
    }

    private void removeDatanode(DatanodeDescriptor nodeInfo) {
        assert (this.namesystem.hasWriteLock());
        this.heartbeatManager.removeDatanode(nodeInfo);
        this.blockManager.removeBlocksAssociatedTo(nodeInfo);
        this.networktopology.remove((Node)nodeInfo);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("remove datanode " + nodeInfo));
        }
        this.namesystem.checkSafeMode();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeDatanode(DatanodeID node) throws UnregisteredNodeException {
        this.namesystem.writeLock();
        try {
            DatanodeDescriptor descriptor = this.getDatanode(node);
            if (descriptor != null) {
                this.removeDatanode(descriptor);
            } else {
                NameNode.stateChangeLog.warn((Object)("BLOCK* removeDatanode: " + node + " does not exist"));
            }
        }
        finally {
            this.namesystem.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeDeadDatanode(DatanodeID nodeID) {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            DatanodeDescriptor d;
            try {
                d = this.getDatanode(nodeID);
            }
            catch (IOException e) {
                d = null;
            }
            if (d != null && this.isDatanodeDead(d)) {
                NameNode.stateChangeLog.info((Object)("BLOCK* removeDeadDatanode: lost heartbeat from " + d));
                this.removeDatanode(d);
            }
        }
    }

    boolean isDatanodeDead(DatanodeDescriptor node) {
        return node.getLastUpdate() < Time.now() - this.heartbeatExpireInterval;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addDatanode(DatanodeDescriptor node) {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            this.host2DatanodeMap.remove(this.datanodeMap.put(node.getStorageID(), node));
        }
        this.host2DatanodeMap.add(node);
        this.networktopology.add((Node)node);
        this.checkIfClusterIsNowMultiRack(node);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(this.getClass().getSimpleName() + ".addDatanode: " + "node " + node + " is added to datanodeMap."));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void wipeDatanode(DatanodeID node) {
        String key = node.getStorageID();
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            this.host2DatanodeMap.remove((DatanodeDescriptor)this.datanodeMap.remove(key));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)(this.getClass().getSimpleName() + ".wipeDatanode(" + node + "): storage " + key + " is removed from datanodeMap."));
        }
    }

    private void resolveNetworkLocation(DatanodeDescriptor node) {
        String networkLocation;
        ArrayList<String> names = new ArrayList<String>(1);
        if (this.dnsToSwitchMapping instanceof CachedDNSToSwitchMapping) {
            names.add(node.getIpAddr());
        } else {
            names.add(node.getHostName());
        }
        List rName = this.dnsToSwitchMapping.resolve(names);
        if (rName == null) {
            LOG.error((Object)("The resolve call returned null! Using /default-rack for host " + names));
            networkLocation = "/default-rack";
        } else {
            networkLocation = (String)rName.get(0);
        }
        node.setNetworkLocation(networkLocation);
    }

    private boolean inHostsList(DatanodeID node) {
        return DatanodeManager.checkInList(node, this.hostsReader.getHosts(), false);
    }

    private boolean inExcludedHostsList(DatanodeID node) {
        return DatanodeManager.checkInList(node, this.hostsReader.getExcludedHosts(), true);
    }

    private void removeDecomNodeFromList(List<DatanodeDescriptor> nodeList) {
        if (this.hostsReader.getHosts().isEmpty()) {
            return;
        }
        Iterator<DatanodeDescriptor> it = nodeList.iterator();
        while (it.hasNext()) {
            DatanodeDescriptor node = it.next();
            if (this.inHostsList(node) || this.inExcludedHostsList(node) || !node.isDecommissioned()) continue;
            it.remove();
        }
    }

    private static boolean checkInList(DatanodeID node, Set<String> hostsList, boolean isExcludeList) {
        InetAddress iaddr;
        try {
            iaddr = InetAddress.getByName(node.getIpAddr());
        }
        catch (UnknownHostException e) {
            LOG.warn((Object)("Unknown IP: " + node.getIpAddr()), (Throwable)e);
            return isExcludeList;
        }
        if (!isExcludeList && hostsList.isEmpty()) {
            return true;
        }
        return hostsList.contains(iaddr.getHostAddress().toString()) || hostsList.contains(iaddr.getHostAddress().toString() + ":" + node.getXferPort()) || hostsList.contains(iaddr.getHostName()) || hostsList.contains(iaddr.getHostName() + ":" + node.getXferPort()) || node instanceof DatanodeInfo && hostsList.contains(((DatanodeInfo)node).getHostName());
    }

    private void checkDecommissioning(DatanodeDescriptor nodeReg) {
        if (this.inExcludedHostsList(nodeReg)) {
            this.startDecommission(nodeReg);
        }
    }

    boolean checkDecommissionState(DatanodeDescriptor node) {
        if (node.isDecommissionInProgress() && !this.blockManager.isReplicationInProgress(node)) {
            node.setDecommissioned();
            LOG.info((Object)("Decommission complete for " + node));
        }
        return node.isDecommissioned();
    }

    private void startDecommission(DatanodeDescriptor node) {
        if (!node.isDecommissionInProgress() && !node.isDecommissioned()) {
            LOG.info((Object)("Start Decommissioning " + node + " with " + node.numBlocks() + " blocks"));
            this.heartbeatManager.startDecommission(node);
            node.decommissioningStatus.setStartTime(Time.now());
            this.checkDecommissionState(node);
        }
    }

    void stopDecommission(DatanodeDescriptor node) {
        if (node.isDecommissionInProgress() || node.isDecommissioned()) {
            LOG.info((Object)("Stop Decommissioning " + node));
            this.heartbeatManager.stopDecommission(node);
            if (node.isAlive) {
                this.blockManager.processOverReplicatedBlocksOnReCommission(node);
            }
        }
    }

    private String newStorageID() {
        String newID = null;
        while (newID == null) {
            newID = "DS" + Integer.toString(DFSUtil.getRandom().nextInt());
            if (this.datanodeMap.get(newID) == null) continue;
            newID = null;
        }
        return newID;
    }

    public void registerDatanode(DatanodeRegistration nodeReg) throws DisallowedDatanodeException {
        String dnAddress = Server.getRemoteAddress();
        if (dnAddress == null) {
            dnAddress = nodeReg.getIpAddr();
        }
        nodeReg.setIpAddr(dnAddress);
        nodeReg.setExportedKeys(this.blockManager.getBlockKeys());
        if (!this.inHostsList(nodeReg)) {
            throw new DisallowedDatanodeException(nodeReg);
        }
        NameNode.stateChangeLog.info((Object)("BLOCK* registerDatanode: from " + nodeReg + " storage " + nodeReg.getStorageID()));
        DatanodeDescriptor nodeS = (DatanodeDescriptor)this.datanodeMap.get(nodeReg.getStorageID());
        DatanodeDescriptor nodeN = this.getDatanodeByHost(nodeReg.getXferAddr());
        if (nodeN != null && nodeN != nodeS) {
            NameNode.LOG.info((Object)("BLOCK* registerDatanode: " + nodeN));
            this.removeDatanode(nodeN);
            this.wipeDatanode(nodeN);
            nodeN = null;
        }
        if (nodeS != null) {
            if (nodeN == nodeS) {
                if (NameNode.stateChangeLog.isDebugEnabled()) {
                    NameNode.stateChangeLog.debug((Object)"BLOCK* registerDatanode: node restarted.");
                }
            } else {
                NameNode.stateChangeLog.info((Object)("BLOCK* registerDatanode: " + nodeS + " is replaced by " + nodeReg + " with the same storageID " + nodeReg.getStorageID()));
            }
            this.getNetworkTopology().remove((Node)nodeS);
            nodeS.updateRegInfo(nodeReg);
            nodeS.setDisallowed(false);
            this.resolveNetworkLocation(nodeS);
            this.getNetworkTopology().add((Node)nodeS);
            this.heartbeatManager.register(nodeS);
            this.checkDecommissioning(nodeS);
            return;
        }
        if ("".equals(nodeReg.getStorageID())) {
            nodeReg.setStorageID(this.newStorageID());
            if (NameNode.stateChangeLog.isDebugEnabled()) {
                NameNode.stateChangeLog.debug((Object)("BLOCK* NameSystem.registerDatanode: new storageID " + nodeReg.getStorageID() + " assigned."));
            }
        }
        DatanodeDescriptor nodeDescr = new DatanodeDescriptor(nodeReg, "/default-rack");
        this.resolveNetworkLocation(nodeDescr);
        this.addDatanode(nodeDescr);
        this.checkDecommissioning(nodeDescr);
        this.heartbeatManager.addDatanode(nodeDescr);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void refreshNodes(Configuration conf) throws IOException {
        this.refreshHostsReader(conf);
        this.namesystem.writeLock();
        try {
            this.refreshDatanodes();
        }
        finally {
            this.namesystem.writeUnlock();
        }
    }

    private void refreshHostsReader(Configuration conf) throws IOException {
        if (conf == null) {
            conf = new HdfsConfiguration();
        }
        this.hostsReader.updateFileNames(conf.get("dfs.hosts", ""), conf.get("dfs.hosts.exclude", ""));
        this.hostsReader.refresh();
    }

    private void refreshDatanodes() throws IOException {
        for (DatanodeDescriptor node : this.datanodeMap.values()) {
            if (!this.inHostsList(node)) {
                node.setDisallowed(true);
                continue;
            }
            if (this.inExcludedHostsList(node)) {
                this.startDecommission(node);
                continue;
            }
            this.stopDecommission(node);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getNumLiveDataNodes() {
        int numLive = 0;
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                if (this.isDatanodeDead(dn)) continue;
                ++numLive;
            }
        }
        return numLive;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getNumDeadDataNodes() {
        int numDead = 0;
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                if (!this.isDatanodeDead(dn)) continue;
                ++numDead;
            }
        }
        return numDead;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DatanodeDescriptor> getDecommissioningNodes() {
        this.namesystem.readLock();
        try {
            ArrayList<DatanodeDescriptor> decommissioningNodes = new ArrayList<DatanodeDescriptor>();
            List<DatanodeDescriptor> results = this.getDatanodeListForReport(HdfsConstants.DatanodeReportType.LIVE);
            for (DatanodeDescriptor node : results) {
                if (!node.isDecommissionInProgress()) continue;
                decommissioningNodes.add(node);
            }
            ArrayList<DatanodeDescriptor> arrayList = decommissioningNodes;
            return arrayList;
        }
        finally {
            this.namesystem.readUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void fetchDatanodes(List<DatanodeDescriptor> live, List<DatanodeDescriptor> dead, boolean removeDecommissionNode) {
        if (live == null && dead == null) {
            throw new HadoopIllegalArgumentException("Both live and dead lists are null");
        }
        this.namesystem.readLock();
        try {
            List<DatanodeDescriptor> results = this.getDatanodeListForReport(HdfsConstants.DatanodeReportType.ALL);
            for (DatanodeDescriptor node : results) {
                if (this.isDatanodeDead(node)) {
                    if (dead == null) continue;
                    dead.add(node);
                    continue;
                }
                if (live == null) continue;
                live.add(node);
            }
        }
        finally {
            this.namesystem.readUnlock();
        }
        if (removeDecommissionNode) {
            if (live != null) {
                this.removeDecomNodeFromList(live);
            }
            if (dead != null) {
                this.removeDecomNodeFromList(dead);
            }
        }
    }

    boolean hasClusterEverBeenMultiRack() {
        return this.hasClusterEverBeenMultiRack;
    }

    @VisibleForTesting
    void checkIfClusterIsNowMultiRack(DatanodeDescriptor node) {
        if (!this.hasClusterEverBeenMultiRack && this.networktopology.getNumOfRacks() > 1) {
            String message = "DN " + node + " joining cluster has expanded a formerly " + "single-rack cluster to be multi-rack. ";
            if (this.namesystem.isPopulatingReplQueues()) {
                message = message + "Re-checking all blocks for replication, since they should now be replicated cross-rack";
                LOG.info((Object)message);
            } else {
                message = message + "Not checking for mis-replicated blocks because this NN is not yet processing repl queues.";
                LOG.debug((Object)message);
            }
            this.hasClusterEverBeenMultiRack = true;
            if (this.namesystem.isPopulatingReplQueues()) {
                this.blockManager.processMisReplicatedBlocks();
            }
        }
    }

    private DatanodeID parseDNFromHostsEntry(String hostLine) {
        DatanodeID dnId;
        int port;
        String hostStr;
        int idx = hostLine.indexOf(58);
        if (-1 == idx) {
            hostStr = hostLine;
            port = 50010;
        } else {
            hostStr = hostLine.substring(0, idx);
            port = Integer.valueOf(hostLine.substring(idx));
        }
        if (InetAddresses.isInetAddress((String)hostStr)) {
            dnId = new DatanodeID(hostStr, "", "", port, 50075, 50020);
        } else {
            String ipAddr = "";
            try {
                ipAddr = InetAddress.getByName(hostStr).getHostAddress();
            }
            catch (UnknownHostException e) {
                LOG.warn((Object)("Invalid hostname " + hostStr + " in hosts file"));
            }
            dnId = new DatanodeID(ipAddr, hostStr, "", port, 50075, 50020);
        }
        return dnId;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DatanodeDescriptor> getDatanodeListForReport(HdfsConstants.DatanodeReportType type) {
        boolean listLiveNodes = type == HdfsConstants.DatanodeReportType.ALL || type == HdfsConstants.DatanodeReportType.LIVE;
        boolean listDeadNodes = type == HdfsConstants.DatanodeReportType.ALL || type == HdfsConstants.DatanodeReportType.DEAD;
        HashMap mustList = new HashMap();
        if (listDeadNodes) {
            Iterator it = this.hostsReader.getHosts().iterator();
            while (it.hasNext()) {
                mustList.put(it.next(), "");
            }
            it = this.hostsReader.getExcludedHosts().iterator();
            while (it.hasNext()) {
                mustList.put(it.next(), "");
            }
        }
        ArrayList<DatanodeDescriptor> nodes = null;
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            nodes = new ArrayList<DatanodeDescriptor>(this.datanodeMap.size() + mustList.size());
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                boolean isDead = this.isDatanodeDead(dn);
                if (isDead && listDeadNodes || !isDead && listLiveNodes) {
                    nodes.add(dn);
                }
                try {
                    InetAddress inet = InetAddress.getByName(dn.getIpAddr());
                    mustList.remove(inet.getHostName());
                    mustList.remove(inet.getHostName() + ":" + dn.getXferPort());
                    mustList.remove(inet.getHostAddress().toString());
                    mustList.remove(inet.getHostAddress().toString() + ":" + dn.getXferPort());
                }
                catch (UnknownHostException e) {
                    mustList.remove(dn.getName());
                    mustList.remove(dn.getIpAddr());
                    LOG.warn((Object)e);
                }
            }
        }
        if (listDeadNodes) {
            Iterator it = mustList.keySet().iterator();
            while (it.hasNext()) {
                DatanodeDescriptor dn;
                DatanodeID dnId = this.parseDNFromHostsEntry((String)it.next());
                dn = new DatanodeDescriptor(dnId);
                dn.setLastUpdate(0L);
                nodes.add(dn);
            }
        }
        return nodes;
    }

    private void setDatanodeDead(DatanodeDescriptor node) throws IOException {
        node.setLastUpdate(0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DatanodeCommand[] handleHeartbeat(DatanodeRegistration nodeReg, String blockPoolId, long capacity, long dfsUsed, long remaining, long blockPoolUsed, int xceiverCount, int maxTransfers, int failedVolumes) throws IOException {
        HeartbeatManager heartbeatManager = this.heartbeatManager;
        synchronized (heartbeatManager) {
            NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
            synchronized (navigableMap) {
                Block[] blks;
                DatanodeDescriptor nodeinfo = null;
                try {
                    nodeinfo = this.getDatanode(nodeReg);
                }
                catch (UnregisteredNodeException e) {
                    return new DatanodeCommand[]{RegisterCommand.REGISTER};
                }
                if (nodeinfo != null && nodeinfo.isDisallowed()) {
                    this.setDatanodeDead(nodeinfo);
                    throw new DisallowedDatanodeException(nodeinfo);
                }
                if (nodeinfo == null || !nodeinfo.isAlive) {
                    return new DatanodeCommand[]{RegisterCommand.REGISTER};
                }
                this.heartbeatManager.updateHeartbeat(nodeinfo, capacity, dfsUsed, remaining, blockPoolUsed, xceiverCount, failedVolumes);
                BlockInfoUnderConstruction[] blocks = nodeinfo.getLeaseRecoveryCommand(Integer.MAX_VALUE);
                if (blocks != null) {
                    BlockRecoveryCommand brCommand = new BlockRecoveryCommand(blocks.length);
                    for (BlockInfoUnderConstruction b : blocks) {
                        brCommand.add(new BlockRecoveryCommand.RecoveringBlock(new ExtendedBlock(blockPoolId, b), b.getExpectedLocations(), b.getBlockRecoveryId()));
                    }
                    return new DatanodeCommand[]{brCommand};
                }
                ArrayList<DatanodeCommand> cmds = new ArrayList<DatanodeCommand>();
                List<DatanodeDescriptor.BlockTargetPair> pendingList = nodeinfo.getReplicationCommand(maxTransfers);
                if (pendingList != null) {
                    cmds.add(new BlockCommand(1, blockPoolId, pendingList));
                }
                if ((blks = nodeinfo.getInvalidateBlocks(this.blockInvalidateLimit)) != null) {
                    cmds.add(new BlockCommand(2, blockPoolId, blks));
                }
                this.blockManager.addKeyUpdateCommand(cmds, nodeinfo);
                if (nodeinfo.getBalancerBandwidth() > 0L) {
                    cmds.add(new BalancerBandwidthCommand(nodeinfo.getBalancerBandwidth()));
                    nodeinfo.setBalancerBandwidth(0L);
                }
                if (!cmds.isEmpty()) {
                    return cmds.toArray(new DatanodeCommand[cmds.size()]);
                }
            }
        }
        return new DatanodeCommand[0];
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setBalancerBandwidth(long bandwidth) throws IOException {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor nodeInfo : this.datanodeMap.values()) {
                nodeInfo.setBalancerBandwidth(bandwidth);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void markAllDatanodesStale() {
        LOG.info((Object)"Marking all datandoes as stale");
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                dn.markStaleAfterFailover();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clearPendingQueues() {
        NavigableMap<String, DatanodeDescriptor> navigableMap = this.datanodeMap;
        synchronized (navigableMap) {
            for (DatanodeDescriptor dn : this.datanodeMap.values()) {
                dn.clearBlockQueues();
            }
        }
    }

    public String toString() {
        return this.getClass().getSimpleName() + ": " + this.host2DatanodeMap;
    }
}

