/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.mapreduce;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.HFileOutputFormat;
import org.apache.hadoop.hbase.mapreduce.PutSortReducer;
import org.apache.hadoop.hbase.mapreduce.TableMapReduceUtil;
import org.apache.hadoop.hbase.mapreduce.TsvImporterMapper;
import org.apache.hadoop.hbase.util.Base64;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.lib.input.FileInputFormat;
import org.apache.hadoop.mapreduce.lib.input.TextInputFormat;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.util.GenericOptionsParser;

public class ImportTsv {
    static final String NAME = "importtsv";
    static final String MAPPER_CONF_KEY = "importtsv.mapper.class";
    static final String SKIP_LINES_CONF_KEY = "importtsv.skip.bad.lines";
    static final String BULK_OUTPUT_CONF_KEY = "importtsv.bulk.output";
    static final String COLUMNS_CONF_KEY = "importtsv.columns";
    static final String SEPARATOR_CONF_KEY = "importtsv.separator";
    static final String TIMESTAMP_CONF_KEY = "importtsv.timestamp";
    static final String DEFAULT_SEPARATOR = "\t";
    static final Class DEFAULT_MAPPER = TsvImporterMapper.class;
    private static HBaseAdmin hbaseAdmin;

    public static Job createSubmittableJob(Configuration conf, String[] args) throws IOException, ClassNotFoundException {
        String mapperClassName;
        String actualSeparator = conf.get(SEPARATOR_CONF_KEY);
        if (actualSeparator != null) {
            conf.set(SEPARATOR_CONF_KEY, Base64.encodeBytes(actualSeparator.getBytes()));
        }
        Class<?> mapperClass = (mapperClassName = conf.get(MAPPER_CONF_KEY)) != null ? Class.forName(mapperClassName) : DEFAULT_MAPPER;
        String tableName = args[0];
        Path inputDir = new Path(args[1]);
        Job job = new Job(conf, "importtsv_" + tableName);
        job.setJarByClass(mapperClass);
        FileInputFormat.setInputPaths((Job)job, (Path[])new Path[]{inputDir});
        job.setInputFormatClass(TextInputFormat.class);
        job.setMapperClass(mapperClass);
        String hfileOutPath = conf.get(BULK_OUTPUT_CONF_KEY);
        if (hfileOutPath != null) {
            if (!ImportTsv.doesTableExist(tableName)) {
                ImportTsv.createTable(conf, tableName);
            }
            HTable table = new HTable(conf, tableName);
            job.setReducerClass(PutSortReducer.class);
            Path outputDir = new Path(hfileOutPath);
            FileOutputFormat.setOutputPath((Job)job, (Path)outputDir);
            job.setMapOutputKeyClass(ImmutableBytesWritable.class);
            job.setMapOutputValueClass(Put.class);
            HFileOutputFormat.configureIncrementalLoad(job, table);
        } else {
            TableMapReduceUtil.initTableReducerJob(tableName, null, job);
            job.setNumReduceTasks(0);
        }
        TableMapReduceUtil.addDependencyJars(job);
        TableMapReduceUtil.addDependencyJars(job.getConfiguration(), Function.class);
        return job;
    }

    private static boolean doesTableExist(String tableName) throws IOException {
        return hbaseAdmin.tableExists(tableName.getBytes());
    }

    private static void createTable(Configuration conf, String tableName) throws IOException {
        HTableDescriptor htd = new HTableDescriptor(tableName.getBytes());
        String[] columns = conf.getStrings(COLUMNS_CONF_KEY);
        HashSet<String> cfSet = new HashSet<String>();
        for (String aColumn : columns) {
            if (TsvParser.ROWKEY_COLUMN_SPEC.equals(aColumn)) continue;
            cfSet.add(aColumn.split(":", 2)[0]);
        }
        for (String cf : cfSet) {
            HColumnDescriptor hcd = new HColumnDescriptor(Bytes.toBytes(cf));
            htd.addFamily(hcd);
        }
        hbaseAdmin.createTable(htd);
    }

    private static void usage(String errorMsg) {
        if (errorMsg != null && errorMsg.length() > 0) {
            System.err.println("ERROR: " + errorMsg);
        }
        String usage = "Usage: importtsv -Dimporttsv.columns=a,b,c <tablename> <inputdir>\n\nImports the given input directory of TSV data into the specified table.\n\nThe column names of the TSV data must be specified using the -Dimporttsv.columns\noption. This option takes the form of comma-separated column names, where each\ncolumn name is either a simple column family, or a columnfamily:qualifier. The special\ncolumn name HBASE_ROW_KEY is used to designate that this column should be used\nas the row key for each imported record. You must specify exactly one column\nto be the row key, and you must specify a column name for every column that exists in the\ninput data.\n\nBy default importtsv will load data directly into HBase. To instead generate\nHFiles of data to prepare for a bulk data load, pass the option:\n  -Dimporttsv.bulk.output=/path/for/output\n  Note: if you do not use this option, then the target table must already exist in HBase\n\nOther options that may be specified with -D include:\n  -Dimporttsv.skip.bad.lines=false - fail if encountering an invalid line\n  '-Dimporttsv.separator=|' - eg separate on pipes instead of tabs\n  -Dimporttsv.timestamp=currentTimeAsLong - use the specified timestamp for the import\n  -Dimporttsv.mapper.class=my.Mapper - A user-defined Mapper to use instead of " + DEFAULT_MAPPER.getName() + "\n" + "For performance consider the following options:\n" + "  -Dmapred.map.tasks.speculative.execution=false\n" + "  -Dmapred.reduce.tasks.speculative.execution=false";
        System.err.println(usage);
    }

    static void createHbaseAdmin(Configuration conf) throws IOException {
        hbaseAdmin = new HBaseAdmin(conf);
    }

    public static void main(String[] args) throws Exception {
        String[] columns;
        Configuration conf = HBaseConfiguration.create();
        String[] otherArgs = new GenericOptionsParser(conf, args).getRemainingArgs();
        if (otherArgs.length < 2) {
            ImportTsv.usage("Wrong number of arguments: " + otherArgs.length);
            System.exit(-1);
        }
        if ((columns = conf.getStrings(COLUMNS_CONF_KEY)) == null) {
            ImportTsv.usage("No columns specified. Please specify with -Dimporttsv.columns=...");
            System.exit(-1);
        }
        int rowkeysFound = 0;
        for (String col : columns) {
            if (!col.equals(TsvParser.ROWKEY_COLUMN_SPEC)) continue;
            ++rowkeysFound;
        }
        if (rowkeysFound != 1) {
            ImportTsv.usage("Must specify exactly one column as " + TsvParser.ROWKEY_COLUMN_SPEC);
            System.exit(-1);
        }
        if (columns.length < 2) {
            ImportTsv.usage("One or more columns in addition to the row key are required");
            System.exit(-1);
        }
        hbaseAdmin = new HBaseAdmin(conf);
        Job job = ImportTsv.createSubmittableJob(conf, otherArgs);
        System.exit(job.waitForCompletion(true) ? 0 : 1);
    }

    static class TsvParser {
        private final byte[][] families;
        private final byte[][] qualifiers;
        private final byte separatorByte;
        private int rowKeyColumnIndex;
        public static String ROWKEY_COLUMN_SPEC = "HBASE_ROW_KEY";

        public TsvParser(String columnsSpecification, String separatorStr) {
            byte[] separator = Bytes.toBytes(separatorStr);
            Preconditions.checkArgument((separator.length == 1 ? 1 : 0) != 0, (Object)"TsvParser only supports single-byte separators");
            this.separatorByte = separator[0];
            ArrayList columnStrings = Lists.newArrayList((Iterable)Splitter.on((char)',').trimResults().split((CharSequence)columnsSpecification));
            this.families = new byte[columnStrings.size()][];
            this.qualifiers = new byte[columnStrings.size()][];
            for (int i = 0; i < columnStrings.size(); ++i) {
                String str = (String)columnStrings.get(i);
                if (ROWKEY_COLUMN_SPEC.equals(str)) {
                    this.rowKeyColumnIndex = i;
                    continue;
                }
                String[] parts = str.split(":", 2);
                if (parts.length == 1) {
                    this.families[i] = str.getBytes();
                    this.qualifiers[i] = HConstants.EMPTY_BYTE_ARRAY;
                    continue;
                }
                this.families[i] = parts[0].getBytes();
                this.qualifiers[i] = parts[1].getBytes();
            }
        }

        public int getRowKeyColumnIndex() {
            return this.rowKeyColumnIndex;
        }

        public byte[] getFamily(int idx) {
            return this.families[idx];
        }

        public byte[] getQualifier(int idx) {
            return this.qualifiers[idx];
        }

        public ParsedLine parse(byte[] lineBytes, int length) throws BadTsvLineException {
            ArrayList<Integer> tabOffsets = new ArrayList<Integer>(this.families.length);
            for (int i = 0; i < length; ++i) {
                if (lineBytes[i] != this.separatorByte) continue;
                tabOffsets.add(i);
            }
            if (tabOffsets.isEmpty()) {
                throw new BadTsvLineException("No delimiter");
            }
            tabOffsets.add(length);
            if (tabOffsets.size() > this.families.length) {
                throw new BadTsvLineException("Excessive columns");
            }
            if (tabOffsets.size() <= this.getRowKeyColumnIndex()) {
                throw new BadTsvLineException("No row key");
            }
            return new ParsedLine(tabOffsets, lineBytes);
        }

        public static class BadTsvLineException
        extends Exception {
            private static final long serialVersionUID = 1L;

            public BadTsvLineException(String err) {
                super(err);
            }
        }

        class ParsedLine {
            private final ArrayList<Integer> tabOffsets;
            private byte[] lineBytes;

            ParsedLine(ArrayList<Integer> tabOffsets, byte[] lineBytes) {
                this.tabOffsets = tabOffsets;
                this.lineBytes = lineBytes;
            }

            public int getRowKeyOffset() {
                return this.getColumnOffset(TsvParser.this.rowKeyColumnIndex);
            }

            public int getRowKeyLength() {
                return this.getColumnLength(TsvParser.this.rowKeyColumnIndex);
            }

            public int getColumnOffset(int idx) {
                if (idx > 0) {
                    return this.tabOffsets.get(idx - 1) + 1;
                }
                return 0;
            }

            public int getColumnLength(int idx) {
                return this.tabOffsets.get(idx) - this.getColumnOffset(idx);
            }

            public int getColumnCount() {
                return this.tabOffsets.size();
            }

            public byte[] getLineBytes() {
                return this.lineBytes;
            }
        }
    }
}

