/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.DataInput;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hbase.HBaseFileSystem;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HDFSBlocksDistribution;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.RemoteExceptionHandler;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSHDFSUtils;
import org.apache.hadoop.hbase.util.FileSystemVersionException;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.FSConstants;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.hadoop.util.StringUtils;

public abstract class FSUtils {
    private static final Log LOG = LogFactory.getLog(FSUtils.class);
    private static final String FULL_RWX_PERMISSIONS = "777";
    private static final String HBASE_URI_PREFIX = "hbase://";
    private static final byte[] HBASE_URI_PREFIX_BYTES = Bytes.toBytes("hbase://");

    protected FSUtils() {
    }

    public static FSUtils getInstance(FileSystem fs, Configuration conf) {
        String scheme = fs.getUri().getScheme();
        if (scheme == null) {
            LOG.warn((Object)("Could not find scheme for uri " + fs.getUri() + ", default to hdfs"));
            scheme = "hdfs";
        }
        Class fsUtilsClass = conf.getClass("hbase.fsutil." + scheme + ".impl", FSHDFSUtils.class);
        FSUtils fsUtils = (FSUtils)ReflectionUtils.newInstance((Class)fsUtilsClass, (Configuration)conf);
        return fsUtils;
    }

    public static boolean deleteDirectory(FileSystem fs, Path dir) throws IOException {
        return fs.exists(dir) && fs.delete(dir, true);
    }

    public Path checkdir(FileSystem fs, Path dir) throws IOException {
        if (!fs.exists(dir)) {
            HBaseFileSystem.makeDirOnFileSystem(fs, dir);
        }
        return dir;
    }

    public static FSDataOutputStream create(FileSystem fs, Path path, FsPermission perm) throws IOException {
        return FSUtils.create(fs, path, perm, true);
    }

    public static FSDataOutputStream create(FileSystem fs, Path path, FsPermission perm, boolean overwrite) throws IOException {
        LOG.debug((Object)("Creating file=" + path + " with permission=" + perm));
        return HBaseFileSystem.createPathWithPermsOnFileSystem(fs, path, perm, overwrite);
    }

    public static FsPermission getFilePermissions(FileSystem fs, Configuration conf, String permssionConfKey) {
        boolean enablePermissions = conf.getBoolean("hbase.data.umask.enable", false);
        if (enablePermissions) {
            try {
                FsPermission perm = new FsPermission(FULL_RWX_PERMISSIONS);
                String mask = conf.get(permssionConfKey);
                if (mask == null) {
                    return FsPermission.getDefault();
                }
                FsPermission umask = new FsPermission(mask);
                return perm.applyUMask(umask);
            }
            catch (IllegalArgumentException e) {
                LOG.warn((Object)("Incorrect umask attempted to be created: " + conf.get(permssionConfKey) + ", using default file permissions."), (Throwable)e);
                return FsPermission.getDefault();
            }
        }
        return FsPermission.getDefault();
    }

    public static void checkFileSystemAvailable(FileSystem fs) throws IOException {
        if (!(fs instanceof DistributedFileSystem)) {
            return;
        }
        IOException exception = null;
        DistributedFileSystem dfs = (DistributedFileSystem)fs;
        try {
            if (dfs.exists(new Path("/"))) {
                return;
            }
        }
        catch (IOException e) {
            exception = RemoteExceptionHandler.checkIOException(e);
        }
        try {
            fs.close();
        }
        catch (Exception e) {
            LOG.error((Object)"file system close failed: ", (Throwable)e);
        }
        IOException io = new IOException("File system is not available");
        io.initCause(exception);
        throw io;
    }

    private static boolean isInSafeMode(DistributedFileSystem dfs) throws IOException {
        boolean inSafeMode = false;
        try {
            Method m = DistributedFileSystem.class.getMethod("setSafeMode", FSConstants.SafeModeAction.class, Boolean.TYPE);
            inSafeMode = (Boolean)m.invoke((Object)dfs, FSConstants.SafeModeAction.SAFEMODE_GET, true);
        }
        catch (Exception e) {
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            inSafeMode = dfs.setSafeMode(FSConstants.SafeModeAction.SAFEMODE_GET);
        }
        return inSafeMode;
    }

    public static void checkDfsSafeMode(Configuration conf) throws IOException {
        boolean isInSafeMode = false;
        FileSystem fs = FileSystem.get((Configuration)conf);
        if (fs instanceof DistributedFileSystem) {
            DistributedFileSystem dfs = (DistributedFileSystem)fs;
            isInSafeMode = FSUtils.isInSafeMode(dfs);
        }
        if (isInSafeMode) {
            throw new IOException("File system is in safemode, it can't be written now");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String getVersion(FileSystem fs, Path rootdir) throws IOException {
        Path versionFile = new Path(rootdir, "hbase.version");
        String version = null;
        if (fs.exists(versionFile)) {
            FSDataInputStream s = fs.open(versionFile);
            try {
                version = DataInputStream.readUTF((DataInput)s);
            }
            catch (EOFException eof) {
                LOG.warn((Object)"Version file was empty, odd, will try to set it.");
            }
            finally {
                s.close();
            }
        }
        return version;
    }

    public static void checkVersion(FileSystem fs, Path rootdir, boolean message) throws IOException {
        FSUtils.checkVersion(fs, rootdir, message, 0, 3);
    }

    public static void checkVersion(FileSystem fs, Path rootdir, boolean message, int wait, int retries) throws IOException {
        String version = FSUtils.getVersion(fs, rootdir);
        if (version == null) {
            if (!FSUtils.rootRegionExists(fs, rootdir)) {
                FSUtils.setVersion(fs, rootdir, wait, retries);
                return;
            }
        } else if (version.compareTo("7") == 0) {
            return;
        }
        String msg = "HBase file layout needs to be upgraded.  You have version " + version + " and I want version " + "7" + ".  Is your hbase.rootdir valid?  If so, you may need to run " + "'hbase hbck -fixVersionFile'.";
        if (message) {
            System.out.println("WARNING! " + msg);
        }
        throw new FileSystemVersionException(msg);
    }

    public static void setVersion(FileSystem fs, Path rootdir) throws IOException {
        FSUtils.setVersion(fs, rootdir, "7", 0, 3);
    }

    public static void setVersion(FileSystem fs, Path rootdir, int wait, int retries) throws IOException {
        FSUtils.setVersion(fs, rootdir, "7", wait, retries);
    }

    public static long getDefaultBlockSize(FileSystem fs, Path path) throws IOException {
        Method m = null;
        Class<?> cls = fs.getClass();
        try {
            m = cls.getMethod("getDefaultBlockSize", Path.class);
        }
        catch (NoSuchMethodException e) {
            LOG.info((Object)"FileSystem doesn't support getDefaultBlockSize");
        }
        catch (SecurityException e) {
            LOG.info((Object)"Doesn't have access to getDefaultBlockSize on FileSystems", (Throwable)e);
            m = null;
        }
        if (m == null) {
            return fs.getDefaultBlockSize();
        }
        try {
            Object ret = m.invoke((Object)fs, path);
            return (Long)ret;
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    public static short getDefaultReplication(FileSystem fs, Path path) throws IOException {
        Method m = null;
        Class<?> cls = fs.getClass();
        try {
            m = cls.getMethod("getDefaultReplication", Path.class);
        }
        catch (NoSuchMethodException e) {
            LOG.info((Object)"FileSystem doesn't support getDefaultReplication");
        }
        catch (SecurityException e) {
            LOG.info((Object)"Doesn't have access to getDefaultReplication on FileSystems", (Throwable)e);
            m = null;
        }
        if (m == null) {
            return fs.getDefaultReplication();
        }
        try {
            Object ret = m.invoke((Object)fs, path);
            return ((Number)ret).shortValue();
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    public static int getDefaultBufferSize(FileSystem fs) {
        return fs.getConf().getInt("io.file.buffer.size", 4096);
    }

    public static void setVersion(FileSystem fs, Path rootdir, String version, int wait, int retries) throws IOException {
        Path versionFile = new Path(rootdir, "hbase.version");
        while (true) {
            try {
                FSDataOutputStream s = fs.create(versionFile);
                s.writeUTF(version);
                LOG.debug((Object)("Created version file at " + rootdir.toString() + " set its version at:" + version));
                s.close();
                return;
            }
            catch (IOException e) {
                if (retries > 0) {
                    LOG.warn((Object)("Unable to create version file at " + rootdir.toString() + ", retrying: " + e.getMessage()));
                    fs.delete(versionFile, false);
                    try {
                        if (wait > 0) {
                            Thread.sleep(wait);
                        }
                    }
                    catch (InterruptedException ex) {
                        // empty catch block
                    }
                    --retries;
                    continue;
                }
                throw e;
            }
            break;
        }
    }

    public static boolean checkClusterIdExists(FileSystem fs, Path rootdir, int wait) throws IOException {
        while (true) {
            try {
                Path filePath = new Path(rootdir, "hbase.id");
                return fs.exists(filePath);
            }
            catch (IOException ioe) {
                block6: {
                    if (wait > 0) {
                        LOG.warn((Object)("Unable to check cluster ID file in " + rootdir.toString() + ", retrying in " + wait + "msec: " + StringUtils.stringifyException((Throwable)ioe)));
                        try {
                            Thread.sleep(wait);
                        }
                        catch (InterruptedException ie) {
                            Thread.interrupted();
                            break block6;
                        }
                        continue;
                    }
                    throw ioe;
                }
                return false;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String getClusterId(FileSystem fs, Path rootdir) throws IOException {
        Path idPath = new Path(rootdir, "hbase.id");
        String clusterId = null;
        if (fs.exists(idPath)) {
            FSDataInputStream in = fs.open(idPath);
            try {
                clusterId = in.readUTF();
            }
            catch (EOFException eof) {
                LOG.warn((Object)("Cluster ID file " + idPath.toString() + " was empty"));
            }
            finally {
                in.close();
            }
        } else {
            LOG.warn((Object)("Cluster ID file does not exist at " + idPath.toString()));
        }
        return clusterId;
    }

    public static void setClusterId(FileSystem fs, Path rootdir, String clusterId, int wait) throws IOException {
        while (true) {
            try {
                Path filePath = new Path(rootdir, "hbase.id");
                FSDataOutputStream s = fs.create(filePath);
                s.writeUTF(clusterId);
                s.close();
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("Created cluster ID file at " + filePath.toString() + " with ID: " + clusterId));
                }
                return;
            }
            catch (IOException ioe) {
                block7: {
                    if (wait > 0) {
                        LOG.warn((Object)("Unable to create cluster ID file in " + rootdir.toString() + ", retrying in " + wait + "msec: " + StringUtils.stringifyException((Throwable)ioe)));
                        try {
                            Thread.sleep(wait);
                        }
                        catch (InterruptedException ie) {
                            Thread.interrupted();
                            break block7;
                        }
                        continue;
                    }
                    throw ioe;
                }
                return;
            }
            break;
        }
    }

    public static Path validateRootPath(Path root) throws IOException {
        try {
            URI rootURI = new URI(root.toString());
            String scheme = rootURI.getScheme();
            if (scheme == null) {
                throw new IOException("Root directory does not have a scheme");
            }
            return root;
        }
        catch (URISyntaxException e) {
            IOException io = new IOException("Root directory path is not a valid URI -- check your hbase.rootdir configuration");
            io.initCause(e);
            throw io;
        }
    }

    public static void waitOnSafeMode(Configuration conf, long wait) throws IOException {
        FileSystem fs = FileSystem.get((Configuration)conf);
        if (!(fs instanceof DistributedFileSystem)) {
            return;
        }
        DistributedFileSystem dfs = (DistributedFileSystem)fs;
        while (FSUtils.isInSafeMode(dfs)) {
            LOG.info((Object)"Waiting for dfs to exit safe mode...");
            try {
                Thread.sleep(wait);
            }
            catch (InterruptedException e) {}
        }
    }

    public static String getPath(Path p) {
        return p.toUri().getPath();
    }

    public static Path getRootDir(Configuration c) throws IOException {
        Path p = new Path(c.get("hbase.rootdir"));
        FileSystem fs = p.getFileSystem(c);
        return p.makeQualified(fs);
    }

    public static void setRootDir(Configuration c, Path root) throws IOException {
        c.set("hbase.rootdir", root.toString());
    }

    public static boolean rootRegionExists(FileSystem fs, Path rootdir) throws IOException {
        Path rootRegionDir = HRegion.getRegionDir(rootdir, HRegionInfo.ROOT_REGIONINFO);
        return fs.exists(rootRegionDir);
    }

    public static HDFSBlocksDistribution computeHDFSBlocksDistribution(FileSystem fs, FileStatus status, long start, long length) throws IOException {
        BlockLocation[] blockLocations;
        HDFSBlocksDistribution blocksDistribution = new HDFSBlocksDistribution();
        for (BlockLocation bl : blockLocations = fs.getFileBlockLocations(status, start, length)) {
            String[] hosts = bl.getHosts();
            long len = bl.getLength();
            blocksDistribution.addHostsAndBlockWeight(hosts, len);
        }
        return blocksDistribution;
    }

    public static boolean isMajorCompacted(FileSystem fs, Path hbaseRootDir) throws IOException {
        FileStatus[] tableDirs;
        for (FileStatus tableDir : tableDirs = fs.listStatus(hbaseRootDir, (PathFilter)new DirFilter(fs))) {
            FileStatus[] regionDirs;
            Path d = tableDir.getPath();
            if (d.getName().equals(".logs")) continue;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)new DirFilter(fs))) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                if (dd.getName().equals("compaction.dir")) continue;
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)new DirFilter(fs))) {
                    Path family = familyDir.getPath();
                    FileStatus[] familyStatus = fs.listStatus(family);
                    if (familyStatus.length <= 1) continue;
                    LOG.debug((Object)(family.toString() + " has " + familyStatus.length + " files."));
                    return false;
                }
            }
        }
        return true;
    }

    public static int getTotalTableFragmentation(HMaster master) throws IOException {
        Map<String, Integer> map = FSUtils.getTableFragmentation(master);
        return map != null && map.size() > 0 ? map.get("-TOTAL-") : -1;
    }

    public static Map<String, Integer> getTableFragmentation(HMaster master) throws IOException {
        Path path = FSUtils.getRootDir(master.getConfiguration());
        FileSystem fs = path.getFileSystem(master.getConfiguration());
        return FSUtils.getTableFragmentation(fs, path);
    }

    public static Map<String, Integer> getTableFragmentation(FileSystem fs, Path hbaseRootDir) throws IOException {
        FileStatus[] tableDirs;
        HashMap<String, Integer> frags = new HashMap<String, Integer>();
        int cfCountTotal = 0;
        int cfFragTotal = 0;
        DirFilter df = new DirFilter(fs);
        for (FileStatus tableDir : tableDirs = fs.listStatus(hbaseRootDir, (PathFilter)df)) {
            FileStatus[] regionDirs;
            Path d = tableDir.getPath();
            if (d.getName().equals(".logs")) continue;
            int cfCount = 0;
            int cfFrag = 0;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)df)) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                if (dd.getName().equals("compaction.dir")) continue;
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)df)) {
                    ++cfCount;
                    ++cfCountTotal;
                    Path family = familyDir.getPath();
                    FileStatus[] familyStatus = fs.listStatus(family);
                    if (familyStatus.length <= 1) continue;
                    ++cfFrag;
                    ++cfFragTotal;
                }
            }
            frags.put(d.getName(), Math.round((float)cfFrag / (float)cfCount * 100.0f));
        }
        frags.put("-TOTAL-", Math.round((float)cfFragTotal / (float)cfCountTotal * 100.0f));
        return frags;
    }

    public static boolean isPre020FileLayout(FileSystem fs, Path hbaseRootDir) throws IOException {
        Path mapfiles = new Path(new Path(new Path(new Path(hbaseRootDir, "-ROOT-"), "70236052"), "info"), "mapfiles");
        return fs.exists(mapfiles);
    }

    public static boolean isMajorCompactedPre020(FileSystem fs, Path hbaseRootDir) throws IOException {
        FileStatus[] tableDirs;
        for (FileStatus tableDir : tableDirs = fs.listStatus(hbaseRootDir, (PathFilter)new DirFilter(fs))) {
            FileStatus[] regionDirs;
            Path d = tableDir.getPath();
            if (d.getName().equals(".logs")) continue;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)new DirFilter(fs))) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                if (dd.getName().equals("compaction.dir")) continue;
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)new DirFilter(fs))) {
                    Path family = familyDir.getPath();
                    FileStatus[] infoAndMapfile = fs.listStatus(family);
                    if (infoAndMapfile.length != 0 && infoAndMapfile.length != 2) {
                        LOG.debug((Object)(family.toString() + " has more than just info and mapfile: " + infoAndMapfile.length));
                        return false;
                    }
                    for (int ll = 0; ll < 2; ++ll) {
                        if (infoAndMapfile[ll].getPath().getName().equals("info") || infoAndMapfile[ll].getPath().getName().equals("mapfiles")) continue;
                        LOG.debug((Object)("Unexpected directory name: " + infoAndMapfile[ll].getPath()));
                        return false;
                    }
                    FileStatus[] familyStatus = fs.listStatus(new Path(family, "mapfiles"));
                    if (familyStatus.length <= 1) continue;
                    LOG.debug((Object)(family.toString() + " has " + familyStatus.length + " files."));
                    return false;
                }
            }
        }
        return true;
    }

    public static boolean isAppendSupported(Configuration conf) {
        boolean append = conf.getBoolean("dfs.support.append", false);
        if (append) {
            try {
                SequenceFile.Writer.class.getMethod("syncFs", new Class[0]);
                append = true;
            }
            catch (SecurityException e) {
            }
            catch (NoSuchMethodException e) {
                append = false;
            }
        }
        if (!append) {
            try {
                FSDataOutputStream.class.getMethod("hflush", new Class[0]);
                append = true;
            }
            catch (NoSuchMethodException e) {
                append = false;
            }
        }
        return append;
    }

    public static boolean isHDFS(Configuration conf) throws IOException {
        FileSystem fs = FileSystem.get((Configuration)conf);
        String scheme = fs.getUri().getScheme();
        return scheme.equalsIgnoreCase("hdfs");
    }

    public abstract void recoverFileLease(FileSystem var1, Path var2, Configuration var3) throws IOException;

    public static List<Path> getTableDirs(FileSystem fs, Path rootdir) throws IOException {
        FileStatus[] dirs = fs.listStatus(rootdir, (PathFilter)new DirFilter(fs));
        ArrayList<Path> tabledirs = new ArrayList<Path>(dirs.length);
        for (FileStatus dir : dirs) {
            Path p = dir.getPath();
            String tableName = p.getName();
            if (HConstants.HBASE_NON_USER_TABLE_DIRS.contains(tableName)) continue;
            tabledirs.add(p);
        }
        return tabledirs;
    }

    public static Path getTablePath(Path rootdir, byte[] tableName) {
        return FSUtils.getTablePath(rootdir, Bytes.toString(tableName));
    }

    public static Path getTablePath(Path rootdir, String tableName) {
        return new Path(rootdir, tableName);
    }

    public static List<Path> getRegionDirs(FileSystem fs, Path tableDir) throws IOException {
        FileStatus[] rds = fs.listStatus(tableDir, (PathFilter)new RegionDirFilter(fs));
        ArrayList<Path> regionDirs = new ArrayList<Path>(rds.length);
        for (FileStatus rdfs : rds) {
            Path rdPath = rdfs.getPath();
            regionDirs.add(rdPath);
        }
        return regionDirs;
    }

    public static List<Path> getFamilyDirs(FileSystem fs, Path regionDir) throws IOException {
        FileStatus[] fds = fs.listStatus(regionDir, (PathFilter)new FamilyDirFilter(fs));
        ArrayList<Path> familyDirs = new ArrayList<Path>(fds.length);
        for (FileStatus fdfs : fds) {
            Path fdPath = fdfs.getPath();
            familyDirs.add(fdPath);
        }
        return familyDirs;
    }

    public static FileSystem getCurrentFileSystem(Configuration conf) throws IOException {
        return FSUtils.getRootDir(conf).getFileSystem(conf);
    }

    public static Map<String, Path> getTableStoreFilePathMap(FileSystem fs, Path hbaseRootDir) throws IOException {
        FileStatus[] tableDirs;
        HashMap<String, Path> map = new HashMap<String, Path>();
        DirFilter df = new DirFilter(fs);
        for (FileStatus tableDir : tableDirs = fs.listStatus(hbaseRootDir, (PathFilter)df)) {
            FileStatus[] regionDirs;
            Path d = tableDir.getPath();
            if (HConstants.HBASE_NON_TABLE_DIRS.contains(d.getName())) continue;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)df)) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                if (dd.getName().equals("compaction.dir")) continue;
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)df)) {
                    FileStatus[] familyStatus;
                    Path family = familyDir.getPath();
                    for (FileStatus sfStatus : familyStatus = fs.listStatus(family)) {
                        Path sf = sfStatus.getPath();
                        map.put(sf.getName(), sf);
                    }
                }
            }
        }
        return map;
    }

    public static FileStatus[] listStatus(FileSystem fs, Path dir, PathFilter filter) throws IOException {
        FileStatus[] status = null;
        try {
            status = filter == null ? fs.listStatus(dir) : fs.listStatus(dir, filter);
        }
        catch (FileNotFoundException fnfe) {
            LOG.debug((Object)(dir + " doesn't exist"));
        }
        if (status == null || status.length < 1) {
            return null;
        }
        return status;
    }

    public static FileStatus[] listStatus(FileSystem fs, Path dir) throws IOException {
        return FSUtils.listStatus(fs, dir, null);
    }

    public static boolean delete(FileSystem fs, Path path, boolean recursive) throws IOException {
        return fs.delete(path, recursive);
    }

    public static void checkAccess(User user, FileStatus file, FsAction action) throws AccessControlException {
        String username = user.getShortName();
        if (username.equals(file.getOwner()) ? file.getPermission().getUserAction().implies(action) : (FSUtils.contains(user.getGroupNames(), file.getGroup()) ? file.getPermission().getGroupAction().implies(action) : file.getPermission().getOtherAction().implies(action))) {
            return;
        }
        throw new AccessControlException("Permission denied: action=" + action + " path=" + file.getPath() + " user=" + username);
    }

    private static boolean contains(String[] groups, String user) {
        for (String group : groups) {
            if (!group.equals(user)) continue;
            return true;
        }
        return false;
    }

    public static boolean isExists(FileSystem fs, Path path) throws IOException {
        return fs.exists(path);
    }

    public static void logFileSystemState(FileSystem fs, Path root, Log LOG) throws IOException {
        LOG.debug((Object)"Current file system:");
        FSUtils.logFSTree(LOG, fs, root, "|-");
    }

    private static void logFSTree(Log LOG, FileSystem fs, Path root, String prefix) throws IOException {
        FileStatus[] files = FSUtils.listStatus(fs, root, null);
        if (files == null) {
            return;
        }
        for (FileStatus file : files) {
            if (file.isDir()) {
                LOG.debug((Object)(prefix + file.getPath().getName() + "/"));
                FSUtils.logFSTree(LOG, fs, file.getPath(), prefix + "---");
                continue;
            }
            LOG.debug((Object)(prefix + file.getPath().getName()));
        }
    }

    public static String adjustTableNameString(String tableName) {
        return tableName.startsWith(HBASE_URI_PREFIX) ? tableName.substring(tableName.lastIndexOf(47) + 1) : tableName;
    }

    public static byte[] adjustTableName(String tableName) {
        return FSUtils.adjustTableName(Bytes.toBytes(tableName));
    }

    public static byte[] adjustTableName(byte[] tableName) {
        if (Bytes.startsWith(tableName, HBASE_URI_PREFIX_BYTES)) {
            int i;
            for (i = tableName.length - 1; i >= HBASE_URI_PREFIX_BYTES.length && tableName[i] != 47; --i) {
            }
            return Arrays.copyOfRange(tableName, i + 1, tableName.length);
        }
        return (byte[])tableName.clone();
    }

    public static class HFileFilter
    implements PathFilter {
        public static final Pattern hfilePattern = Pattern.compile("^([0-9a-f]+)$");
        final FileSystem fs;

        public HFileFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path rd) {
            if (!hfilePattern.matcher(rd.getName()).matches()) {
                return false;
            }
            try {
                return !this.fs.getFileStatus(rd).isDir();
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Skipping file " + rd + " due to IOException"), (Throwable)ioe);
                return false;
            }
        }
    }

    public static class FamilyDirFilter
    implements PathFilter {
        final FileSystem fs;

        public FamilyDirFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path rd) {
            try {
                HColumnDescriptor.isLegalFamilyName(Bytes.toBytes(rd.getName()));
            }
            catch (IllegalArgumentException iae) {
                return false;
            }
            try {
                return this.fs.getFileStatus(rd).isDir();
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Skipping file " + rd + " due to IOException"), (Throwable)ioe);
                return false;
            }
        }
    }

    public static class RegionDirFilter
    implements PathFilter {
        public static final Pattern regionDirPattern = Pattern.compile("^[0-9a-f]*$");
        final FileSystem fs;

        public RegionDirFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path rd) {
            if (!regionDirPattern.matcher(rd.getName()).matches()) {
                return false;
            }
            try {
                return this.fs.getFileStatus(rd).isDir();
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Skipping file " + rd + " due to IOException"), (Throwable)ioe);
                return false;
            }
        }
    }

    public static class DirFilter
    implements PathFilter {
        private final FileSystem fs;

        public DirFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path p) {
            boolean isValid = false;
            try {
                isValid = HConstants.HBASE_NON_USER_TABLE_DIRS.contains(p) ? false : this.fs.getFileStatus(p).isDir();
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            return isValid;
        }
    }

    static class FileFilter
    implements PathFilter {
        private final FileSystem fs;

        public FileFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path p) {
            try {
                return this.fs.isFile(p);
            }
            catch (IOException e) {
                LOG.debug((Object)("unable to verify if path=" + p + " is a regular file"), (Throwable)e);
                return false;
            }
        }
    }
}

