/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ClockOutOfSyncException;
import org.apache.hadoop.hbase.HMsg;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HServerAddress;
import org.apache.hadoop.hbase.HServerInfo;
import org.apache.hadoop.hbase.HServerLoad;
import org.apache.hadoop.hbase.PleaseHoldException;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.YouAreDeadException;
import org.apache.hadoop.hbase.catalog.CatalogTracker;
import org.apache.hadoop.hbase.client.HConnection;
import org.apache.hadoop.hbase.client.HConnectionManager;
import org.apache.hadoop.hbase.ipc.HRegionInterface;
import org.apache.hadoop.hbase.master.DeadServer;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.handler.MetaServerShutdownHandler;
import org.apache.hadoop.hbase.master.handler.ServerShutdownHandler;
import org.apache.hadoop.hbase.master.metrics.MasterMetrics;

public class ServerManager {
    private static final Log LOG = LogFactory.getLog(ServerManager.class);
    private volatile boolean clusterShutdown = false;
    private final Map<String, HServerInfo> onlineServers = new ConcurrentHashMap<String, HServerInfo>();
    private final Map<String, HRegionInterface> serverConnections = new HashMap<String, HRegionInterface>();
    private final Server master;
    private final MasterServices services;
    private final MasterMetrics metrics;
    private final DeadServer deadservers;
    private final long maxSkew;

    public ServerManager(Server master, MasterServices services, MasterMetrics metrics) {
        this.master = master;
        this.services = services;
        this.metrics = metrics;
        Configuration c = master.getConfiguration();
        this.maxSkew = c.getLong("hbase.master.maxclockskew", 30000L);
        this.deadservers = new DeadServer(c.getInt("hbase.master.maxdeadservers", 100));
    }

    void regionServerStartup(HServerInfo serverInfo, long serverCurrentTime) throws IOException {
        HServerInfo info = new HServerInfo(serverInfo);
        this.checkIsDead(info.getServerName(), "STARTUP");
        this.checkAlreadySameHostPort(info);
        this.checkClockSkew(info, serverCurrentTime);
        this.recordNewServer(info, false, null);
    }

    void checkAlreadySameHostPort(HServerInfo serverInfo) throws PleaseHoldException {
        String hostAndPort = serverInfo.getServerAddress().toString();
        HServerInfo existingServer = this.haveServerWithSameHostAndPortAlready(serverInfo.getHostnamePort());
        if (existingServer != null) {
            String message = "Server start rejected; we already have " + hostAndPort + " registered; existingServer=" + existingServer + ", newServer=" + serverInfo;
            LOG.info((Object)message);
            if (existingServer.getStartCode() < serverInfo.getStartCode()) {
                LOG.info((Object)("Triggering server recovery; existingServer " + existingServer.getServerName() + " looks stale"));
                this.expireServer(existingServer);
            }
            throw new PleaseHoldException(message);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private HServerInfo haveServerWithSameHostAndPortAlready(String hostnamePort) {
        Map<String, HServerInfo> map = this.onlineServers;
        synchronized (map) {
            for (Map.Entry<String, HServerInfo> e : this.onlineServers.entrySet()) {
                if (!e.getValue().getHostnamePort().equals(hostnamePort)) continue;
                return e.getValue();
            }
        }
        return null;
    }

    private void checkClockSkew(HServerInfo serverInfo, long serverCurrentTime) throws ClockOutOfSyncException {
        long skew = System.currentTimeMillis() - serverCurrentTime;
        if (skew > this.maxSkew) {
            String message = "Server " + serverInfo.getServerName() + " has been " + "rejected; Reported time is too far out of sync with master.  " + "Time difference of " + skew + "ms > max allowed of " + this.maxSkew + "ms";
            LOG.warn((Object)message);
            throw new ClockOutOfSyncException(message);
        }
    }

    private void checkIsDead(String serverName, String what) throws YouAreDeadException {
        if (this.deadservers.isDeadServer(serverName)) {
            String message = "Server " + what + " rejected; currently processing " + serverName + " as dead server";
            LOG.debug((Object)message);
            throw new YouAreDeadException(message);
        }
        if (this.deadservers.cleanPreviousInstance(serverName)) {
            LOG.debug((Object)("Server " + serverName + " came back up, removed it from the" + " dead servers list"));
        }
    }

    void recordNewServer(HServerInfo info, boolean useInfoLoad, HRegionInterface hri) {
        HServerLoad load = useInfoLoad ? info.getLoad() : new HServerLoad();
        String serverName = info.getServerName();
        LOG.info((Object)("Registering server=" + serverName + ", regionCount=" + load.getLoad() + ", userLoad=" + useInfoLoad));
        info.setLoad(load);
        this.onlineServers.put(serverName, info);
        if (hri == null) {
            this.serverConnections.remove(serverName);
        } else {
            this.serverConnections.put(serverName, hri);
        }
    }

    HMsg[] regionServerReport(HServerInfo serverInfo, HMsg[] msgs, HRegionInfo[] mostLoadedRegions) throws IOException {
        HServerInfo info = new HServerInfo(serverInfo);
        this.checkIsDead(info.getServerName(), "REPORT");
        HServerInfo storedInfo = this.onlineServers.get(info.getServerName());
        if (storedInfo == null) {
            this.checkAlreadySameHostPort(info);
            this.recordNewServer(info, true, null);
            if (msgs.length > 0) {
                throw new PleaseHoldException("FIX! Putting off message processing because not yet rwady but possible we won't be ready next on next report");
            }
        }
        if (this.raceThatShouldNotHappenAnymore(storedInfo, info)) {
            return HMsg.STOP_REGIONSERVER_ARRAY;
        }
        block3: for (HMsg msg : msgs) {
            LOG.info((Object)("Received " + msg + " from " + serverInfo.getServerName()));
            switch (msg.getType()) {
                case REGION_SPLIT: {
                    this.services.getAssignmentManager().handleSplitReport(serverInfo, msg.getRegionInfo(), msg.getDaughterA(), msg.getDaughterB());
                    continue block3;
                }
                default: {
                    LOG.error((Object)("Unhandled msg type " + msg));
                }
            }
        }
        HMsg[] reply = null;
        int numservers = this.countOfRegionServers();
        if (this.clusterShutdown && numservers <= 2) {
            reply = HMsg.STOP_REGIONSERVER_ARRAY;
        }
        return this.processRegionServerAllsWell(info, mostLoadedRegions, reply);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean raceThatShouldNotHappenAnymore(HServerInfo storedInfo, HServerInfo reportedInfo) {
        if (storedInfo.getStartCode() != reportedInfo.getStartCode()) {
            LOG.warn((Object)("Race condition detected: " + reportedInfo.getServerName()));
            Map<String, HServerInfo> map = this.onlineServers;
            synchronized (map) {
                this.removeServerInfo(reportedInfo.getServerName());
                this.notifyOnlineServers();
            }
            return true;
        }
        return false;
    }

    private HMsg[] processRegionServerAllsWell(HServerInfo serverInfo, HRegionInfo[] mostLoadedRegions, HMsg[] msgs) throws IOException {
        this.onlineServers.put(serverInfo.getServerName(), serverInfo);
        HServerLoad load = serverInfo.getLoad();
        if (load != null && this.metrics != null) {
            this.metrics.incrementRequests(load.getNumberOfRequests());
        }
        return msgs;
    }

    private boolean removeServerInfo(String serverName) {
        HServerInfo info = this.onlineServers.remove(serverName);
        return info != null;
    }

    public double getAverageLoad() {
        int totalLoad = 0;
        int numServers = 0;
        double averageLoad = 0.0;
        for (HServerInfo hsi : this.onlineServers.values()) {
            ++numServers;
            totalLoad += hsi.getLoad().getNumberOfRegions();
        }
        averageLoad = (double)totalLoad / (double)numServers;
        return averageLoad;
    }

    int countOfRegionServers() {
        return this.onlineServers.size();
    }

    public HServerInfo getServerInfo(String name) {
        return this.onlineServers.get(name);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map<String, HServerInfo> getOnlineServers() {
        Map<String, HServerInfo> map = this.onlineServers;
        synchronized (map) {
            return Collections.unmodifiableMap(this.onlineServers);
        }
    }

    public Set<String> getDeadServers() {
        return this.deadservers.clone();
    }

    public boolean areDeadServersInProgress() {
        return this.deadservers.areDeadServersInProgress();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public HServerInfo getHServerInfo(HServerAddress hsa) {
        Map<String, HServerInfo> map = this.onlineServers;
        synchronized (map) {
            for (Map.Entry<String, HServerInfo> e : this.onlineServers.entrySet()) {
                if (!e.getValue().getServerAddress().equals(hsa)) continue;
                return e.getValue();
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void notifyOnlineServers() {
        Map<String, HServerInfo> map = this.onlineServers;
        synchronized (map) {
            this.onlineServers.notifyAll();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    void letRegionServersShutdown() {
        Map<String, HServerInfo> map = this.onlineServers;
        synchronized (map) {
            while (this.onlineServers.size() > 0) {
                StringBuilder sb = new StringBuilder();
                for (String key : this.onlineServers.keySet()) {
                    if (sb.length() > 0) {
                        sb.append(", ");
                    }
                    sb.append(key);
                }
                LOG.info((Object)("Waiting on regionserver(s) to go down " + sb.toString()));
                try {
                    this.onlineServers.wait(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
            return;
        }
    }

    public synchronized void expireServer(HServerInfo hsi) {
        boolean carryingMeta;
        boolean carryingRoot;
        HServerAddress address;
        String serverName = hsi.getServerName();
        HServerInfo info = this.onlineServers.get(serverName);
        if (info == null) {
            LOG.warn((Object)("Received expiration of " + hsi.getServerName() + " but server is not currently online"));
            return;
        }
        if (this.deadservers.contains(serverName)) {
            LOG.warn((Object)("Received expiration of " + hsi.getServerName() + " but server shutdown is already in progress"));
            return;
        }
        this.deadservers.add(serverName);
        this.onlineServers.remove(serverName);
        this.serverConnections.remove(serverName);
        if (this.clusterShutdown) {
            LOG.info((Object)("Cluster shutdown set; " + hsi.getServerName() + " expired; onlineServers=" + this.onlineServers.size()));
            if (this.onlineServers.isEmpty()) {
                this.master.stop("Cluster shutdown set; onlineServer=0");
            }
            return;
        }
        CatalogTracker ct = this.master.getCatalogTracker();
        try {
            address = ct.getRootLocation();
            carryingRoot = address != null && hsi.getServerAddress().equals(address);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.info((Object)"Interrupted");
            return;
        }
        address = ct.getMetaLocation();
        boolean bl = carryingMeta = address != null && hsi.getServerAddress().equals(address);
        if (carryingRoot || carryingMeta) {
            this.services.getExecutorService().submit(new MetaServerShutdownHandler(this.master, this.services, this.deadservers, info, carryingRoot, carryingMeta));
        } else {
            this.services.getExecutorService().submit(new ServerShutdownHandler(this.master, this.services, this.deadservers, info));
        }
        LOG.debug((Object)("Added=" + serverName + " to dead servers, submitted shutdown handler to be executed, root=" + carryingRoot + ", meta=" + carryingMeta));
    }

    public void sendRegionOpen(HServerInfo server, HRegionInfo region) throws IOException {
        HRegionInterface hri = this.getServerConnection(server);
        if (hri == null) {
            LOG.warn((Object)("Attempting to send OPEN RPC to server " + server.getServerName() + " failed because no RPC connection found to this server"));
            return;
        }
        hri.openRegion(region);
    }

    public void sendRegionOpen(HServerInfo server, List<HRegionInfo> regions) throws IOException {
        HRegionInterface hri = this.getServerConnection(server);
        if (hri == null) {
            LOG.warn((Object)("Attempting to send OPEN RPC to server " + server.getServerName() + " failed because no RPC connection found to this server"));
            return;
        }
        hri.openRegions(regions);
    }

    public boolean sendRegionClose(HServerInfo server, HRegionInfo region) throws IOException {
        if (server == null) {
            throw new NullPointerException("Passed server is null");
        }
        HRegionInterface hri = this.getServerConnection(server);
        if (hri == null) {
            throw new IOException("Attempting to send CLOSE RPC to server " + server.getServerName() + " for region " + region.getRegionNameAsString() + " failed because no RPC connection found to this server");
        }
        return hri.closeRegion(region);
    }

    private HRegionInterface getServerConnection(HServerInfo info) throws IOException {
        HConnection connection = HConnectionManager.getConnection(this.master.getConfiguration());
        HRegionInterface hri = this.serverConnections.get(info.getServerName());
        if (hri == null) {
            LOG.debug((Object)("New connection to " + info.getServerName()));
            hri = connection.getHRegionConnection(info.getServerAddress(), false);
            this.serverConnections.put(info.getServerName(), hri);
        }
        return hri;
    }

    public int waitForRegionServers() throws InterruptedException {
        long interval = this.master.getConfiguration().getLong("hbase.master.wait.on.regionservers.interval", 1500L);
        long timeout = this.master.getConfiguration().getLong("hbase.master.wait.on.regionservers.timeout", 4500L);
        int minToStart = this.master.getConfiguration().getInt("hbase.master.wait.on.regionservers.mintostart", 1);
        int maxToStart = this.master.getConfiguration().getInt("hbase.master.wait.on.regionservers.maxtostart", Integer.MAX_VALUE);
        int count = 0;
        long slept = 0L;
        int oldcount = this.countOfRegionServers();
        while (!this.master.isStopped()) {
            Thread.sleep(interval);
            count = this.countOfRegionServers();
            if (count == oldcount && count >= minToStart && (slept += interval) >= timeout) {
                LOG.info((Object)("Finished waiting for regionserver count to settle; count=" + count + ", sleptFor=" + slept));
                break;
            }
            if (count >= maxToStart) {
                LOG.info((Object)("At least the max configured number of regionserver(s) have checked in: " + count));
                break;
            }
            if (count == 0) {
                LOG.info((Object)"Waiting on regionserver(s) to checkin");
            } else {
                LOG.info((Object)("Waiting on regionserver(s) count to settle; currently=" + count));
            }
            oldcount = count;
        }
        int regionCount = 0;
        for (Map.Entry<String, HServerInfo> e : this.onlineServers.entrySet()) {
            HServerLoad load = e.getValue().getLoad();
            if (load == null) continue;
            regionCount += load.getLoad();
        }
        LOG.info((Object)("Exiting wait on regionserver(s) to checkin; count=" + count + ", stopped=" + this.master.isStopped() + ", count of regions out on cluster=" + regionCount));
        return regionCount;
    }

    public List<HServerInfo> getOnlineServersList() {
        return new ArrayList<HServerInfo>(this.onlineServers.values());
    }

    public boolean isServerOnline(String serverName) {
        return this.onlineServers.containsKey(serverName);
    }

    public void shutdownCluster() {
        this.clusterShutdown = true;
        this.master.stop("Cluster shutdown requested");
    }

    public boolean isClusterShutdown() {
        return this.clusterShutdown;
    }

    public void stop() {
    }
}

