/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.common.StorageErrorReporter;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileOutputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogOutputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.JournalManager;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NNStorageRetentionManager;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.hdfs.server.protocol.RemoteEditLog;

@InterfaceAudience.Private
public class FileJournalManager
implements JournalManager {
    private static final Log LOG = LogFactory.getLog(FileJournalManager.class);
    private final Storage.StorageDirectory sd;
    private final StorageErrorReporter errorReporter;
    private int outputBufferCapacity = 524288;
    private static final Pattern EDITS_REGEX = Pattern.compile(NNStorage.NameNodeFile.EDITS.getName() + "_(\\d+)-(\\d+)");
    private static final Pattern EDITS_INPROGRESS_REGEX = Pattern.compile(NNStorage.NameNodeFile.EDITS_INPROGRESS.getName() + "_(\\d+)");
    private File currentInProgress = null;
    @VisibleForTesting
    NNStorageRetentionManager.StoragePurger purger = new NNStorageRetentionManager.DeletionStoragePurger();

    public FileJournalManager(Storage.StorageDirectory sd, StorageErrorReporter errorReporter) {
        this.sd = sd;
        this.errorReporter = errorReporter;
    }

    @Override
    public void close() throws IOException {
    }

    @Override
    public void format(NamespaceInfo ns) throws IOException {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean hasSomeData() {
        throw new UnsupportedOperationException();
    }

    @Override
    public synchronized EditLogOutputStream startLogSegment(long txid) throws IOException {
        try {
            this.currentInProgress = NNStorage.getInProgressEditsFile(this.sd, txid);
            EditLogFileOutputStream stm = new EditLogFileOutputStream(this.currentInProgress, this.outputBufferCapacity);
            ((EditLogOutputStream)stm).create();
            return stm;
        }
        catch (IOException e) {
            LOG.warn((Object)("Unable to start log segment " + txid + " at " + this.currentInProgress + ": " + e.getLocalizedMessage()));
            this.errorReporter.reportErrorOnFile(this.currentInProgress);
            throw e;
        }
    }

    @Override
    public synchronized void finalizeLogSegment(long firstTxId, long lastTxId) throws IOException {
        File inprogressFile = NNStorage.getInProgressEditsFile(this.sd, firstTxId);
        File dstFile = NNStorage.getFinalizedEditsFile(this.sd, firstTxId, lastTxId);
        LOG.info((Object)("Finalizing edits file " + inprogressFile + " -> " + dstFile));
        Preconditions.checkState((!dstFile.exists() ? 1 : 0) != 0, (Object)("Can't finalize edits file " + inprogressFile + " since finalized file " + "already exists"));
        if (!inprogressFile.renameTo(dstFile)) {
            this.errorReporter.reportErrorOnFile(dstFile);
            throw new IllegalStateException("Unable to finalize edits file " + inprogressFile);
        }
        if (inprogressFile.equals(this.currentInProgress)) {
            this.currentInProgress = null;
        }
    }

    @VisibleForTesting
    public Storage.StorageDirectory getStorageDirectory() {
        return this.sd;
    }

    @Override
    public synchronized void setOutputBufferCapacity(int size) {
        this.outputBufferCapacity = size;
    }

    @Override
    public void purgeLogsOlderThan(long minTxIdToKeep) throws IOException {
        LOG.info((Object)("Purging logs older than " + minTxIdToKeep));
        File[] files = FileUtil.listFiles((File)this.sd.getCurrentDir());
        List<EditLogFile> editLogs = FileJournalManager.matchEditLogs(files);
        for (EditLogFile log : editLogs) {
            if (log.getFirstTxId() >= minTxIdToKeep || log.getLastTxId() >= minTxIdToKeep) continue;
            this.purger.purgeLog(log);
        }
    }

    public List<RemoteEditLog> getRemoteEditLogs(long firstTxId, boolean inProgressOk) throws IOException {
        File currentDir = this.sd.getCurrentDir();
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        ArrayList ret = Lists.newArrayListWithCapacity((int)allLogFiles.size());
        for (EditLogFile elf : allLogFiles) {
            if (elf.hasCorruptHeader() || !inProgressOk && elf.isInProgress()) continue;
            if (elf.getFirstTxId() >= firstTxId) {
                ret.add(new RemoteEditLog(elf.firstTxId, elf.lastTxId));
                continue;
            }
            if (elf.getFirstTxId() >= firstTxId || firstTxId > elf.getLastTxId()) continue;
            ret.add(new RemoteEditLog(elf.firstTxId, elf.lastTxId));
        }
        Collections.sort(ret);
        return ret;
    }

    public static List<EditLogFile> matchEditLogs(File logDir) throws IOException {
        return FileJournalManager.matchEditLogs(FileUtil.listFiles((File)logDir));
    }

    static List<EditLogFile> matchEditLogs(File[] filesInStorage) {
        ArrayList ret = Lists.newArrayList();
        for (File f : filesInStorage) {
            Matcher inProgressEditsMatch;
            String name = f.getName();
            Matcher editsMatch = EDITS_REGEX.matcher(name);
            if (editsMatch.matches()) {
                try {
                    long startTxId = Long.valueOf(editsMatch.group(1));
                    long endTxId = Long.valueOf(editsMatch.group(2));
                    ret.add(new EditLogFile(f, startTxId, endTxId));
                }
                catch (NumberFormatException nfe) {
                    LOG.error((Object)("Edits file " + f + " has improperly formatted " + "transaction ID"));
                }
            }
            if (!(inProgressEditsMatch = EDITS_INPROGRESS_REGEX.matcher(name)).matches()) continue;
            try {
                long startTxId = Long.valueOf(inProgressEditsMatch.group(1));
                ret.add(new EditLogFile(f, startTxId, -12345L, true));
            }
            catch (NumberFormatException nfe) {
                LOG.error((Object)("In-progress edits file " + f + " has improperly " + "formatted transaction ID"));
            }
        }
        return ret;
    }

    @Override
    public synchronized void selectInputStreams(Collection<EditLogInputStream> streams, long fromTxId, boolean inProgressOk) throws IOException {
        List<EditLogFile> elfs = FileJournalManager.matchEditLogs(this.sd.getCurrentDir());
        LOG.debug((Object)(this + ": selecting input streams starting at " + fromTxId + (inProgressOk ? " (inProgress ok) " : " (excluding inProgress) ") + "from among " + elfs.size() + " candidate file(s)"));
        FileJournalManager.addStreamsToCollectionFromFiles(elfs, streams, fromTxId, inProgressOk);
    }

    static void addStreamsToCollectionFromFiles(Collection<EditLogFile> elfs, Collection<EditLogInputStream> streams, long fromTxId, boolean inProgressOk) {
        for (EditLogFile elf : elfs) {
            if (elf.isInProgress()) {
                if (!inProgressOk) {
                    LOG.debug((Object)("passing over " + elf + " because it is in progress " + "and we are ignoring in-progress logs."));
                    continue;
                }
                try {
                    elf.validateLog();
                }
                catch (IOException e) {
                    LOG.error((Object)("got IOException while trying to validate header of " + elf + ".  Skipping."), (Throwable)e);
                    continue;
                }
            }
            if (elf.lastTxId < fromTxId) {
                assert (elf.lastTxId != -12345L);
                LOG.debug((Object)("passing over " + elf + " because it ends at " + elf.lastTxId + ", but we only care about transactions " + "as new as " + fromTxId));
                continue;
            }
            EditLogFileInputStream elfis = new EditLogFileInputStream(elf.getFile(), elf.getFirstTxId(), elf.getLastTxId(), elf.isInProgress());
            LOG.debug((Object)("selecting edit log stream " + elf));
            streams.add(elfis);
        }
    }

    @Override
    public synchronized void recoverUnfinalizedSegments() throws IOException {
        File currentDir = this.sd.getCurrentDir();
        LOG.info((Object)("Recovering unfinalized segments in " + currentDir));
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        for (EditLogFile elf : allLogFiles) {
            if (elf.getFile().equals(this.currentInProgress) || !elf.isInProgress()) continue;
            if (elf.getFile().length() == 0L) {
                LOG.info((Object)("Deleting zero-length edit log file " + elf));
                if (elf.getFile().delete()) continue;
                throw new IOException("Unable to delete file " + elf.getFile());
            }
            elf.validateLog();
            if (elf.hasCorruptHeader()) {
                elf.moveAsideCorruptFile();
                throw new JournalManager.CorruptionException("In-progress edit log file is corrupt: " + elf);
            }
            if (elf.getLastTxId() == -12345L) {
                LOG.info((Object)("Moving aside edit log file that seems to have zero transactions " + elf));
                elf.moveAsideEmptyFile();
                continue;
            }
            this.finalizeLogSegment(elf.getFirstTxId(), elf.getLastTxId());
        }
    }

    public List<EditLogFile> getLogFiles(long fromTxId) throws IOException {
        File currentDir = this.sd.getCurrentDir();
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(currentDir);
        ArrayList logFiles = Lists.newArrayList();
        for (EditLogFile elf : allLogFiles) {
            if (fromTxId > elf.getFirstTxId() && !elf.containsTxId(fromTxId)) continue;
            logFiles.add(elf);
        }
        Collections.sort(logFiles, EditLogFile.COMPARE_BY_START_TXID);
        return logFiles;
    }

    public EditLogFile getLogFile(long startTxId) throws IOException {
        return FileJournalManager.getLogFile(this.sd.getCurrentDir(), startTxId);
    }

    public static EditLogFile getLogFile(File dir, long startTxId) throws IOException {
        List<EditLogFile> files = FileJournalManager.matchEditLogs(dir);
        LinkedList ret = Lists.newLinkedList();
        for (EditLogFile elf : files) {
            if (elf.getFirstTxId() != startTxId) continue;
            ret.add(elf);
        }
        if (ret.isEmpty()) {
            return null;
        }
        if (ret.size() == 1) {
            return (EditLogFile)ret.get(0);
        }
        throw new IllegalStateException("More than one log segment in " + dir + " starting at txid " + startTxId + ": " + Joiner.on((String)", ").join((Iterable)ret));
    }

    @Override
    public String toString() {
        return String.format("FileJournalManager(root=%s)", this.sd.getRoot());
    }

    @InterfaceAudience.Private
    public static class EditLogFile {
        private File file;
        private final long firstTxId;
        private long lastTxId;
        private boolean hasCorruptHeader = false;
        private final boolean isInProgress;
        static final Comparator<EditLogFile> COMPARE_BY_START_TXID = new Comparator<EditLogFile>(){

            @Override
            public int compare(EditLogFile a, EditLogFile b) {
                return ComparisonChain.start().compare(a.getFirstTxId(), b.getFirstTxId()).compare(a.getLastTxId(), b.getLastTxId()).result();
            }
        };

        EditLogFile(File file, long firstTxId, long lastTxId) {
            this(file, firstTxId, lastTxId, false);
            assert (lastTxId != -12345L && lastTxId >= firstTxId);
        }

        EditLogFile(File file, long firstTxId, long lastTxId, boolean isInProgress) {
            assert (lastTxId == -12345L && isInProgress || lastTxId != -12345L && lastTxId >= firstTxId);
            assert (firstTxId > 0L || firstTxId == -12345L);
            assert (file != null);
            Preconditions.checkArgument((!isInProgress || lastTxId == -12345L ? 1 : 0) != 0);
            this.firstTxId = firstTxId;
            this.lastTxId = lastTxId;
            this.file = file;
            this.isInProgress = isInProgress;
        }

        public long getFirstTxId() {
            return this.firstTxId;
        }

        public long getLastTxId() {
            return this.lastTxId;
        }

        boolean containsTxId(long txId) {
            return this.firstTxId <= txId && txId <= this.lastTxId;
        }

        public void validateLog() throws IOException {
            FSEditLogLoader.EditLogValidation val = EditLogFileInputStream.validateEditLog(this.file);
            this.lastTxId = val.getEndTxId();
            this.hasCorruptHeader = val.hasCorruptHeader();
        }

        public boolean isInProgress() {
            return this.isInProgress;
        }

        public File getFile() {
            return this.file;
        }

        boolean hasCorruptHeader() {
            return this.hasCorruptHeader;
        }

        void moveAsideCorruptFile() throws IOException {
            assert (this.hasCorruptHeader);
            this.renameSelf(".corrupt");
        }

        public void moveAsideEmptyFile() throws IOException {
            assert (this.lastTxId == -12345L);
            this.renameSelf(".empty");
        }

        private void renameSelf(String newSuffix) throws IOException {
            File src = this.file;
            File dst = new File(src.getParent(), src.getName() + newSuffix);
            boolean success = src.renameTo(dst);
            if (!success) {
                throw new IOException("Couldn't rename log " + src + " to " + dst);
            }
            this.file = dst;
        }

        public String toString() {
            return String.format("EditLogFile(file=%s,first=%019d,last=%019d,inProgress=%b,hasCorruptHeader=%b)", this.file.toString(), this.firstTxId, this.lastTxId, this.isInProgress(), this.hasCorruptHeader);
        }
    }
}

