#!/bin/bash

# *****************************************************************************
#
# AEL Daemon
#
# Copyright (C) 2017 by Hitachi Vantara : http://www.pentaho.com
#
# *****************************************************************************
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with
# the License. You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# *****************************************************************************

# START
# ---------------------------
# If 0 is passed in then the Daemon will run in the forground.
# If 1 is passed in it will run in the backgroud and redirect logs to daemon.log.  A .pid file is created so we can track which process to kill when stop is used.
function start(){

  local root_command="java -cp pdi-websocket-daemon-8.0.0.0-28.jar:lib/* org.pentaho.adaptive.daemon.DaemonMain"
  # If true start as a background process
  if [ $1 -eq 0 ]
  then
    if [ -f $PID_FILE ]; then
      echo "Unable to start AEL Daemon in foreground.  PID file exists indicating there may be an an instance already running."
    else
      $root_command
    fi
  else
    echo Starting PDI Daemon in the background
    $root_command > daemon.log 2>&1 &
    local app_pid=$!
    echo $app_pid > $PID_FILE
    echo Daemon started with process ID: $app_pid
  fi

}

# STATUS
# ---------------------------
# Check to see if a pid file is present and echo the pid if it exists
function status(){
  if [ -f $PID_FILE ]; then
    local pid=`cat .pid`
    echo "AEL Daemon Running (PID: $pid)"
  else
    echo "The .pid file does not exist; assuming daemon is not running."
    exit
  fi
}

# STOP
# ---------------------------
# Read the pid file and kill the daemon
function stop(){
  if [ -f $PID_FILE ]; then
    local pid=`cat $PID_FILE`
    kill $pid
    local kill_status=$?
    echo KILL STAT: $kill_status
    if [ $kill_status -eq $? ]; then
      `kill -9 $pid`
    fi
    rm "$PID_FILE"
    echo "AEL Daemon Stopped (PID: $pid)"
  else
    echo "The .pid file does not exist; unable to find process to stop daemon."
    exit
  fi
}

# HELP
# ---------------------------
# Prints the command usage.
function printHelp() {
  echo "usage: $PROGNAME                     - Runs in the current console"
  echo "       $PROGNAME [start|status|stop] - Runs as a background process"
}

# MAIN PROGRAM
# ---------------------------
# Current working directory
PROGNAME=daemon.sh
PID_FILE=".pid"
pushd $(pwd)

# Directory this script resides
cd $(dirname "$0")

# If no parameters are found; then start daemon in forground
if [ $# -eq 0 ]; then
    start 0
fi

while [[ $# -gt 0 ]] ; do
  opt="$1"; shift;
  case "$opt" in
    "-h"|"--help" )
      printHelp
      exit 0;;
    "start" )
      start 1
      exit 0;;
    "status" )
      status
      exit 0;;
    "stop" )
      stop
      exit 0;;
    * )
      echo "Invalid option \"$opt\"" >&2
      printHelp
      exit 1;;
  esac
done

popd
