/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver.compactions;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.PeekingIterator;
import com.google.common.math.LongMath;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HDFSBlocksDistribution;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.regionserver.RSRpcServices;
import org.apache.hadoop.hbase.regionserver.StoreConfigInformation;
import org.apache.hadoop.hbase.regionserver.StoreFile;
import org.apache.hadoop.hbase.regionserver.StoreUtils;
import org.apache.hadoop.hbase.regionserver.compactions.CompactionRequest;
import org.apache.hadoop.hbase.regionserver.compactions.DateTieredCompactionRequest;
import org.apache.hadoop.hbase.regionserver.compactions.RatioBasedCompactionPolicy;
import org.apache.hadoop.hbase.regionserver.compactions.SortedCompactionPolicy;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.ReflectionUtils;

@InterfaceAudience.LimitedPrivate(value={"Configuration"})
public class DateTieredCompactionPolicy
extends SortedCompactionPolicy {
    private static final Log LOG = LogFactory.getLog(DateTieredCompactionPolicy.class);
    private RatioBasedCompactionPolicy compactionPolicyPerWindow;

    public DateTieredCompactionPolicy(Configuration conf, StoreConfigInformation storeConfigInfo) throws IOException {
        super(conf, storeConfigInfo);
        try {
            this.compactionPolicyPerWindow = (RatioBasedCompactionPolicy)ReflectionUtils.instantiateWithCustomCtor((String)this.comConf.getCompactionPolicyForTieredWindow(), (Class[])new Class[]{Configuration.class, StoreConfigInformation.class}, (Object[])new Object[]{conf, storeConfigInfo});
        }
        catch (Exception e) {
            throw new IOException("Unable to load configured compaction policy '" + this.comConf.getCompactionPolicyForTieredWindow() + "'", e);
        }
    }

    @Override
    @VisibleForTesting
    public boolean needsCompaction(Collection<StoreFile> storeFiles, List<StoreFile> filesCompacting) {
        ArrayList<StoreFile> candidates = new ArrayList<StoreFile>(storeFiles);
        try {
            return this.selectMinorCompaction(candidates, false, true) != null;
        }
        catch (Exception e) {
            LOG.error((Object)"Can not check for compaction: ", (Throwable)e);
            return false;
        }
    }

    @Override
    public boolean shouldPerformMajorCompaction(Collection<StoreFile> filesToCompact) throws IOException {
        long mcTime = this.getNextMajorCompactTime(filesToCompact);
        if (filesToCompact == null || mcTime == 0L) {
            return false;
        }
        long lowTimestamp = StoreUtils.getLowestTimestamp(filesToCompact);
        long now = EnvironmentEdgeManager.currentTime();
        if (lowTimestamp <= 0L || lowTimestamp >= now - mcTime) {
            return false;
        }
        long cfTTL = this.storeConfigInfo.getStoreFileTtl();
        HDFSBlocksDistribution hdfsBlocksDistribution = new HDFSBlocksDistribution();
        long oldestToCompact = DateTieredCompactionPolicy.getOldestToCompact(this.comConf.getMaxStoreFileAgeMillis(), now);
        List<Long> boundaries = this.getCompactBoundariesForMajor(filesToCompact, oldestToCompact, now);
        boolean[] filesInWindow = new boolean[boundaries.size()];
        for (StoreFile file : filesToCompact) {
            long oldest;
            Long minTimestamp = file.getMinimumTimestamp();
            long l = oldest = minTimestamp == null ? Long.valueOf(Long.MIN_VALUE) : now - minTimestamp;
            if (cfTTL != Integer.MAX_VALUE && oldest >= cfTTL) {
                LOG.debug((Object)("Major compaction triggered on store " + this + "; for TTL maintenance"));
                return true;
            }
            if (!file.isMajorCompaction() || file.isBulkLoadResult()) {
                LOG.debug((Object)("Major compaction triggered on store " + this + ", because there are new files and time since last major compaction " + (now - lowTimestamp) + "ms"));
                return true;
            }
            int lowerWindowIndex = Collections.binarySearch(boundaries, minTimestamp == null ? Long.valueOf(Long.MAX_VALUE) : minTimestamp);
            int upperWindowIndex = Collections.binarySearch(boundaries, file.getMaximumTimestamp() == null ? Long.valueOf(Long.MAX_VALUE) : file.getMaximumTimestamp());
            lowerWindowIndex = lowerWindowIndex < 0 ? Math.abs(lowerWindowIndex + 2) : lowerWindowIndex;
            int n = upperWindowIndex = upperWindowIndex < 0 ? Math.abs(upperWindowIndex + 2) : upperWindowIndex;
            if (lowerWindowIndex != upperWindowIndex) {
                LOG.debug((Object)("Major compaction triggered on store " + this + "; because file " + file.getPath() + " has data with timestamps cross window boundaries"));
                return true;
            }
            if (filesInWindow[upperWindowIndex]) {
                LOG.debug((Object)("Major compaction triggered on store " + this + "; because there are more than one file in some windows"));
                return true;
            }
            filesInWindow[upperWindowIndex] = true;
            hdfsBlocksDistribution.add(file.getHDFSBlockDistribution());
        }
        float blockLocalityIndex = hdfsBlocksDistribution.getBlockLocalityIndex(RSRpcServices.getHostname(this.comConf.conf, false));
        if (blockLocalityIndex < this.comConf.getMinLocalityToForceCompact()) {
            LOG.debug((Object)("Major compaction triggered on store " + this + "; to make hdfs blocks local, current blockLocalityIndex is " + blockLocalityIndex + " (min " + this.comConf.getMinLocalityToForceCompact() + ")"));
            return true;
        }
        LOG.debug((Object)("Skipping major compaction of " + this + ", because the files are already major compacted"));
        return false;
    }

    @Override
    protected CompactionRequest createCompactionRequest(ArrayList<StoreFile> candidateSelection, boolean tryingMajor, boolean mayUseOffPeak, boolean mayBeStuck) throws IOException {
        CompactionRequest result = tryingMajor ? this.selectMajorCompaction(candidateSelection) : this.selectMinorCompaction(candidateSelection, mayUseOffPeak, mayBeStuck);
        LOG.debug((Object)("Generated compaction request: " + result));
        return result;
    }

    public CompactionRequest selectMajorCompaction(ArrayList<StoreFile> candidateSelection) {
        long now = EnvironmentEdgeManager.currentTime();
        long oldestToCompact = DateTieredCompactionPolicy.getOldestToCompact(this.comConf.getMaxStoreFileAgeMillis(), now);
        return new DateTieredCompactionRequest(candidateSelection, this.getCompactBoundariesForMajor(candidateSelection, oldestToCompact, now));
    }

    public CompactionRequest selectMinorCompaction(ArrayList<StoreFile> candidateSelection, boolean mayUseOffPeak, boolean mayBeStuck) throws IOException {
        long now = EnvironmentEdgeManager.currentTime();
        long oldestToCompact = DateTieredCompactionPolicy.getOldestToCompact(this.comConf.getMaxStoreFileAgeMillis(), now);
        ArrayList storeFileList = Lists.newArrayList(DateTieredCompactionPolicy.filterOldStoreFiles(candidateSelection, oldestToCompact));
        ArrayList storefileMaxTimestampPairs = Lists.newArrayListWithCapacity((int)Iterables.size((Iterable)storeFileList));
        long maxTimestampSeen = Long.MIN_VALUE;
        for (StoreFile storeFile : storeFileList) {
            maxTimestampSeen = Math.max(maxTimestampSeen, storeFile.getMaximumTimestamp() == null ? Long.MIN_VALUE : storeFile.getMaximumTimestamp());
            storefileMaxTimestampPairs.add(new Pair((Object)storeFile, (Object)maxTimestampSeen));
        }
        Collections.reverse(storefileMaxTimestampPairs);
        Window window = DateTieredCompactionPolicy.getIncomingWindow(now, this.comConf.getBaseWindowMillis());
        int minThreshold = this.comConf.getIncomingWindowMin();
        PeekingIterator it = Iterators.peekingIterator(storefileMaxTimestampPairs.iterator());
        while (it.hasNext()) {
            int compResult = window.compareToTimestamp((Long)((Pair)it.peek()).getSecond());
            if (compResult > 0) {
                window = window.nextWindow(this.comConf.getWindowsPerTier(), oldestToCompact);
                minThreshold = this.comConf.getMinFilesToCompact();
                continue;
            }
            ArrayList fileList = Lists.newArrayList();
            while (it.hasNext() && window.compareToTimestamp((Long)((Pair)it.peek()).getSecond()) <= 0) {
                fileList.add(((Pair)it.next()).getFirst());
            }
            if (fileList.size() < minThreshold) continue;
            LOG.debug((Object)("Processing files: " + fileList + " for window: " + window));
            DateTieredCompactionRequest request = this.generateCompactionRequest(fileList, window, mayUseOffPeak, mayBeStuck, minThreshold);
            if (request == null) continue;
            return request;
        }
        return new CompactionRequest(Collections.emptyList());
    }

    private DateTieredCompactionRequest generateCompactionRequest(ArrayList<StoreFile> storeFiles, Window window, boolean mayUseOffPeak, boolean mayBeStuck, int minThreshold) throws IOException {
        ArrayList<StoreFile> storeFileSelection;
        Collections.reverse(storeFiles);
        this.compactionPolicyPerWindow.setMinThreshold(minThreshold);
        ArrayList<StoreFile> arrayList = storeFileSelection = mayBeStuck ? storeFiles : this.compactionPolicyPerWindow.applyCompactionPolicy(storeFiles, mayUseOffPeak, false);
        if (storeFileSelection != null && !storeFileSelection.isEmpty()) {
            boolean singleOutput = storeFiles.size() != storeFileSelection.size() || this.comConf.useSingleOutputForMinorCompaction();
            List<Long> boundaries = DateTieredCompactionPolicy.getCompactionBoundariesForMinor(window, singleOutput);
            DateTieredCompactionRequest result = new DateTieredCompactionRequest(storeFileSelection, boundaries);
            return result;
        }
        return null;
    }

    private List<Long> getCompactBoundariesForMajor(Collection<StoreFile> filesToCompact, long oldestToCompact, long now) {
        long minTimestamp = Long.MAX_VALUE;
        for (StoreFile file : filesToCompact) {
            minTimestamp = Math.min(minTimestamp, file.getMinimumTimestamp() == null ? Long.MAX_VALUE : file.getMinimumTimestamp());
        }
        ArrayList<Long> boundaries = new ArrayList<Long>();
        Window window = DateTieredCompactionPolicy.getIncomingWindow(now, this.comConf.getBaseWindowMillis());
        while (window.compareToTimestamp(minTimestamp) > 0) {
            boundaries.add(window.startMillis());
            window = window.nextWindow(this.comConf.getWindowsPerTier(), oldestToCompact);
        }
        boundaries.add(Long.MIN_VALUE);
        Collections.reverse(boundaries);
        return boundaries;
    }

    private static List<Long> getCompactionBoundariesForMinor(Window window, boolean singleOutput) {
        ArrayList<Long> boundaries = new ArrayList<Long>();
        boundaries.add(Long.MIN_VALUE);
        if (!singleOutput) {
            boundaries.add(window.startMillis());
        }
        return boundaries;
    }

    private static Iterable<StoreFile> filterOldStoreFiles(List<StoreFile> storeFiles, final long cutoff) {
        return Iterables.filter(storeFiles, (Predicate)new Predicate<StoreFile>(){

            public boolean apply(StoreFile storeFile) {
                if (storeFile == null) {
                    return false;
                }
                Long maxTimestamp = storeFile.getMaximumTimestamp();
                return maxTimestamp == null ? true : maxTimestamp >= cutoff;
            }
        });
    }

    private static Window getIncomingWindow(long now, long baseWindowMillis) {
        return new Window(baseWindowMillis, now / baseWindowMillis);
    }

    private static long getOldestToCompact(long maxAgeMillis, long now) {
        try {
            return LongMath.checkedSubtract((long)now, (long)maxAgeMillis);
        }
        catch (ArithmeticException ae) {
            LOG.warn((Object)("Value for hbase.hstore.compaction.date.tiered.max.storefile.age.millis: " + maxAgeMillis + ". All the files will be eligible for minor compaction."));
            return Long.MIN_VALUE;
        }
    }

    private static final class Window {
        private final long windowMillis;
        private final long divPosition;

        private Window(long baseWindowMillis, long divPosition) {
            this.windowMillis = baseWindowMillis;
            this.divPosition = divPosition;
        }

        public int compareToTimestamp(long timestamp) {
            long pos;
            if (timestamp < 0L) {
                try {
                    timestamp = LongMath.checkedSubtract((long)timestamp, (long)(this.windowMillis - 1L));
                }
                catch (ArithmeticException ae) {
                    timestamp = Long.MIN_VALUE;
                }
            }
            return this.divPosition == (pos = timestamp / this.windowMillis) ? 0 : (this.divPosition < pos ? -1 : 1);
        }

        public Window nextWindow(int windowsPerTier, long oldestToCompact) {
            if (this.divPosition % (long)windowsPerTier > 0L || this.startMillis() - this.windowMillis * (long)windowsPerTier < oldestToCompact) {
                return new Window(this.windowMillis, this.divPosition - 1L);
            }
            return new Window(this.windowMillis * (long)windowsPerTier, this.divPosition / (long)windowsPerTier - 1L);
        }

        public long startMillis() {
            try {
                return LongMath.checkedMultiply((long)this.windowMillis, (long)this.divPosition);
            }
            catch (ArithmeticException ae) {
                return Long.MIN_VALUE;
            }
        }

        public long endMillis() {
            try {
                return LongMath.checkedMultiply((long)this.windowMillis, (long)(this.divPosition + 1L));
            }
            catch (ArithmeticException ae) {
                return Long.MAX_VALUE;
            }
        }

        public String toString() {
            return "[" + this.startMillis() + ", " + this.endMillis() + ")";
        }
    }
}

