/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sqoop.hive;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import org.apache.avro.Schema;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.sqoop.SqoopOptions;
import org.apache.sqoop.avro.AvroUtil;
import org.apache.sqoop.hive.HiveTypes;
import org.apache.sqoop.io.CodecMap;
import org.apache.sqoop.manager.ConnManager;
import org.apache.sqoop.util.FileSystemUtil;

public class TableDefWriter {
    public static final Log LOG = LogFactory.getLog((String)TableDefWriter.class.getName());
    private SqoopOptions options;
    private ConnManager connManager;
    private Configuration configuration;
    private String inputTableName;
    private String outputTableName;
    private boolean commentsEnabled;
    private Schema avroSchema;

    public TableDefWriter(SqoopOptions opts, ConnManager connMgr, String inputTable, String outputTable, Configuration config, boolean withComments) {
        this.options = opts;
        this.connManager = connMgr;
        this.inputTableName = inputTable;
        this.outputTableName = outputTable;
        this.configuration = config;
        this.commentsEnabled = withComments;
    }

    private String[] getColumnNames() {
        if (this.options.getFileLayout() == SqoopOptions.FileLayout.ParquetFile) {
            return this.getColumnNamesFromAvroSchema();
        }
        String[] colNames = this.options.getColumns();
        if (null != colNames) {
            return colNames;
        }
        if (null != this.inputTableName) {
            return this.connManager.getColumnNames(this.inputTableName);
        }
        return this.connManager.getColumnNamesForQuery(this.options.getSqlQuery());
    }

    private String[] getColumnNamesFromAvroSchema() {
        ArrayList<String> result = new ArrayList<String>();
        for (Schema.Field field : this.getAvroSchema().getFields()) {
            result.add(field.name());
        }
        return result.toArray(new String[result.size()]);
    }

    public String getCreateTableStmt() throws IOException {
        this.resetConnManager();
        Properties userMapping = this.options.getMapColumnHive();
        Boolean isHiveExternalTableSet = !StringUtils.isBlank((String)this.options.getHiveExternalTableDir());
        Map<String, Integer> columnTypes = null != this.inputTableName ? this.connManager.getColumnTypes(this.inputTableName) : this.connManager.getColumnTypesForQuery(this.options.getSqlQuery());
        String[] colNames = this.getColumnNames();
        Map<String, Schema.Type> columnNameToAvroType = this.getColumnNameToAvroTypeMapping();
        StringBuilder sb = new StringBuilder();
        if (this.options.doFailIfHiveTableExists()) {
            if (isHiveExternalTableSet.booleanValue()) {
                sb.append("CREATE EXTERNAL TABLE `");
            } else {
                sb.append("CREATE TABLE `");
            }
        } else if (isHiveExternalTableSet.booleanValue()) {
            sb.append("CREATE EXTERNAL TABLE IF NOT EXISTS `");
        } else {
            sb.append("CREATE TABLE IF NOT EXISTS `");
        }
        if (this.options.getHiveDatabaseName() != null) {
            sb.append(this.options.getHiveDatabaseName()).append("`.`");
        }
        sb.append(this.outputTableName).append("` ( ");
        for (Object column : userMapping.keySet()) {
            boolean found = false;
            String[] stringArray = colNames;
            int n = stringArray.length;
            for (int i = 0; i < n; ++i) {
                String c = stringArray[i];
                if (!c.equals(column)) continue;
                found = true;
                break;
            }
            if (found) continue;
            throw new IllegalArgumentException("No column by the name " + column + "found while importing data");
        }
        boolean first = true;
        String partitionKey = this.options.getHivePartitionKey();
        for (String col : colNames) {
            String hiveColType;
            if (col.equals(partitionKey)) {
                throw new IllegalArgumentException("Partition key " + col + " cannot be a column to import.");
            }
            if (!first) {
                sb.append(", ");
            }
            first = false;
            if (this.options.getFileLayout() == SqoopOptions.FileLayout.TextFile) {
                Integer colType = columnTypes.get(col);
                hiveColType = this.getHiveColumnTypeForTextTable(userMapping, col, colType);
            } else if (this.options.getFileLayout() == SqoopOptions.FileLayout.ParquetFile) {
                hiveColType = HiveTypes.toHiveType(columnNameToAvroType.get(col));
            } else {
                throw new RuntimeException("File format is not supported for Hive tables.");
            }
            sb.append('`').append(col).append("` ").append(hiveColType);
        }
        sb.append(") ");
        if (this.commentsEnabled) {
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
            String curDateStr = dateFormat.format(new Date());
            sb.append("COMMENT 'Imported by sqoop on " + curDateStr + "' ");
        }
        if (partitionKey != null) {
            sb.append("PARTITIONED BY (").append(partitionKey).append(" STRING) ");
        }
        if (SqoopOptions.FileLayout.ParquetFile.equals((Object)this.options.getFileLayout())) {
            sb.append("STORED AS PARQUET");
        } else {
            sb.append("ROW FORMAT DELIMITED FIELDS TERMINATED BY '");
            sb.append(TableDefWriter.getHiveOctalCharCode(this.options.getOutputFieldDelim()));
            sb.append("' LINES TERMINATED BY '");
            sb.append(TableDefWriter.getHiveOctalCharCode(this.options.getOutputRecordDelim()));
            String codec = this.options.getCompressionCodec();
            if (codec != null && (codec.equals("lzop") || codec.equals(CodecMap.getCodecClassName("lzop")))) {
                sb.append("' STORED AS INPUTFORMAT 'com.hadoop.mapred.DeprecatedLzoTextInputFormat'");
                sb.append(" OUTPUTFORMAT 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat'");
            } else {
                sb.append("' STORED AS TEXTFILE");
            }
        }
        if (isHiveExternalTableSet.booleanValue()) {
            sb.append(" LOCATION '" + this.options.getHiveExternalTableDir() + "'");
        }
        LOG.debug((Object)("Create statement: " + sb.toString()));
        return sb.toString();
    }

    private Map<String, Schema.Type> getColumnNameToAvroTypeMapping() {
        if (this.options.getFileLayout() != SqoopOptions.FileLayout.ParquetFile) {
            return Collections.emptyMap();
        }
        HashMap<String, Schema.Type> result = new HashMap<String, Schema.Type>();
        Schema avroSchema = this.getAvroSchema();
        for (Schema.Field field : avroSchema.getFields()) {
            result.put(field.name(), this.getNonNullAvroType(field.schema()));
        }
        return result;
    }

    private Schema.Type getNonNullAvroType(Schema schema) {
        if (schema.getType() != Schema.Type.UNION) {
            return schema.getType();
        }
        for (Schema subSchema : schema.getTypes()) {
            if (subSchema.getType() == Schema.Type.NULL) continue;
            return subSchema.getType();
        }
        return null;
    }

    private String getHiveColumnTypeForTextTable(Properties userMapping, String columnName, Integer columnType) throws IOException {
        String hiveColType = userMapping.getProperty(columnName);
        if (hiveColType == null) {
            hiveColType = this.connManager.toHiveType(this.inputTableName, columnName, columnType);
        }
        if (null == hiveColType) {
            throw new IOException("Hive does not support the SQL type for column " + columnName);
        }
        if (HiveTypes.isHiveTypeImprovised(columnType)) {
            LOG.warn((Object)("Column " + columnName + " had to be cast to a less precise type in Hive"));
        }
        return hiveColType;
    }

    public String getLoadDataStmt() throws IOException {
        Path finalPath = this.getFinalPath();
        StringBuilder sb = new StringBuilder();
        sb.append("LOAD DATA INPATH '");
        sb.append(finalPath.toString() + "'");
        if (this.options.doOverwriteHiveTable()) {
            sb.append(" OVERWRITE");
        }
        sb.append(" INTO TABLE `");
        if (this.options.getHiveDatabaseName() != null) {
            sb.append(this.options.getHiveDatabaseName()).append("`.`");
        }
        sb.append(this.outputTableName);
        sb.append('`');
        if (this.options.getHivePartitionKey() != null) {
            sb.append(" PARTITION (").append(this.options.getHivePartitionKey()).append("='").append(this.options.getHivePartitionValue()).append("')");
        }
        LOG.debug((Object)("Load statement: " + sb.toString()));
        return sb.toString();
    }

    public Path getFinalPath() throws IOException {
        String warehouseDir = this.options.getWarehouseDir();
        if (null == warehouseDir) {
            warehouseDir = "";
        } else if (!warehouseDir.endsWith(File.separator)) {
            warehouseDir = warehouseDir + File.separator;
        }
        String tablePath = null;
        String targetDir = this.options.getTargetDir();
        tablePath = null != targetDir ? warehouseDir + targetDir : warehouseDir + this.inputTableName;
        return FileSystemUtil.makeQualified(new Path(tablePath), this.configuration);
    }

    public static String getHiveOctalCharCode(int charNum) {
        if (charNum > 127) {
            throw new IllegalArgumentException("Character " + charNum + " is an out-of-range delimiter");
        }
        return String.format("\\%03o", charNum);
    }

    private void resetConnManager() {
        this.connManager.discardConnection(true);
    }

    SqoopOptions getOptions() {
        return this.options;
    }

    ConnManager getConnManager() {
        return this.connManager;
    }

    Configuration getConfiguration() {
        return this.configuration;
    }

    String getInputTableName() {
        return this.inputTableName;
    }

    String getOutputTableName() {
        return this.outputTableName;
    }

    boolean isCommentsEnabled() {
        return this.commentsEnabled;
    }

    Schema getAvroSchema() {
        if (this.avroSchema == null) {
            String schemaString = this.options.getConf().get("parquetjob.avro.schema");
            this.avroSchema = AvroUtil.parseAvroSchema(schemaString);
        }
        return this.avroSchema;
    }
}

