/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.balancer;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URI;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Formatter;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.StorageType;
import org.apache.hadoop.hdfs.server.balancer.BalancingPolicy;
import org.apache.hadoop.hdfs.server.balancer.Dispatcher;
import org.apache.hadoop.hdfs.server.balancer.ExitStatus;
import org.apache.hadoop.hdfs.server.balancer.Matcher;
import org.apache.hadoop.hdfs.server.balancer.NameNodeConnector;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockPlacementPolicy;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockPlacementPolicyDefault;
import org.apache.hadoop.hdfs.server.namenode.UnsupportedActionException;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorageReport;
import org.apache.hadoop.hdfs.server.protocol.StorageReport;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

@InterfaceAudience.Private
public class Balancer {
    static final Log LOG = LogFactory.getLog(Balancer.class);
    private static final Path BALANCER_ID_PATH = new Path("/system/balancer.id");
    private static final long GB = 0x40000000L;
    private static final long MAX_SIZE_TO_MOVE = 0x280000000L;
    private static final String USAGE = "Usage: java " + Balancer.class.getSimpleName() + "\n\t[-policy <policy>]\tthe balancing policy: " + BalancingPolicy.Node.INSTANCE.getName() + " or " + BalancingPolicy.Pool.INSTANCE.getName() + "\n\t[-threshold <threshold>]\tPercentage of disk capacity" + "\n\t[-exclude [-f <hosts-file> | comma-sperated list of hosts]]" + "\tExcludes the specified datanodes." + "\n\t[-include [-f <hosts-file> | comma-sperated list of hosts]]" + "\tIncludes only the specified datanodes.";
    private final Dispatcher dispatcher;
    private final BalancingPolicy policy;
    private final double threshold;
    private final Collection<Dispatcher.Source> overUtilized = new LinkedList<Dispatcher.Source>();
    private final Collection<Dispatcher.Source> aboveAvgUtilized = new LinkedList<Dispatcher.Source>();
    private final Collection<Dispatcher.DDatanode.StorageGroup> belowAvgUtilized = new LinkedList<Dispatcher.DDatanode.StorageGroup>();
    private final Collection<Dispatcher.DDatanode.StorageGroup> underUtilized = new LinkedList<Dispatcher.DDatanode.StorageGroup>();

    private static void checkReplicationPolicyCompatibility(Configuration conf) throws UnsupportedActionException {
        if (!(BlockPlacementPolicy.getInstance(conf, null, null, null) instanceof BlockPlacementPolicyDefault)) {
            throw new UnsupportedActionException("Balancer without BlockPlacementPolicyDefault");
        }
    }

    Balancer(NameNodeConnector theblockpool, Parameters p, Configuration conf) {
        long movedWinWidth = conf.getLong("dfs.balancer.movedWinWidth", 5400000L);
        int moverThreads = conf.getInt("dfs.balancer.moverThreads", 1000);
        int dispatcherThreads = conf.getInt("dfs.balancer.dispatcherThreads", 200);
        int maxConcurrentMovesPerNode = conf.getInt("dfs.datanode.balance.max.concurrent.moves", 5);
        this.dispatcher = new Dispatcher(theblockpool, p.nodesToBeIncluded, p.nodesToBeExcluded, movedWinWidth, moverThreads, dispatcherThreads, maxConcurrentMovesPerNode, conf);
        this.threshold = p.threshold;
        this.policy = p.policy;
    }

    private static long getCapacity(DatanodeStorageReport report, StorageType t) {
        long capacity = 0L;
        for (StorageReport r : report.getStorageReports()) {
            if (r.getStorage().getStorageType() != t) continue;
            capacity += r.getCapacity();
        }
        return capacity;
    }

    private static long getRemaining(DatanodeStorageReport report, StorageType t) {
        long remaining = 0L;
        for (StorageReport r : report.getStorageReports()) {
            if (r.getStorage().getStorageType() != t) continue;
            remaining += r.getRemaining();
        }
        return remaining;
    }

    private long init(List<DatanodeStorageReport> reports) {
        for (DatanodeStorageReport r : reports) {
            this.policy.accumulateSpaces(r);
        }
        this.policy.initAvgUtilization();
        long overLoadedBytes = 0L;
        long underLoadedBytes = 0L;
        for (DatanodeStorageReport r : reports) {
            Dispatcher.DDatanode dn = this.dispatcher.newDatanode(r);
            for (StorageType t : StorageType.asList()) {
                Dispatcher.DDatanode.StorageGroup g;
                Double utilization = this.policy.getUtilization(r, t);
                if (utilization == null) continue;
                long capacity = Balancer.getCapacity(r, t);
                double utilizationDiff = utilization - this.policy.getAvgUtilization(t);
                double thresholdDiff = Math.abs(utilizationDiff) - this.threshold;
                long maxSize2Move = Balancer.computeMaxSize2Move(capacity, Balancer.getRemaining(r, t), utilizationDiff, this.threshold);
                if (utilizationDiff > 0.0) {
                    Dispatcher.Source s = dn.addSource(t, maxSize2Move, this.dispatcher);
                    if (thresholdDiff <= 0.0) {
                        this.aboveAvgUtilized.add(s);
                    } else {
                        overLoadedBytes += Balancer.precentage2bytes(thresholdDiff, capacity);
                        this.overUtilized.add(s);
                    }
                    g = s;
                } else {
                    g = dn.addStorageGroup(t, maxSize2Move);
                    if (thresholdDiff <= 0.0) {
                        this.belowAvgUtilized.add(g);
                    } else {
                        underLoadedBytes += Balancer.precentage2bytes(thresholdDiff, capacity);
                        this.underUtilized.add(g);
                    }
                }
                this.dispatcher.getStorageGroupMap().put(g);
            }
        }
        this.logUtilizationCollections();
        Preconditions.checkState((this.dispatcher.getStorageGroupMap().size() == this.overUtilized.size() + this.underUtilized.size() + this.aboveAvgUtilized.size() + this.belowAvgUtilized.size() ? 1 : 0) != 0, (Object)"Mismatched number of storage groups");
        return Math.max(overLoadedBytes, underLoadedBytes);
    }

    private static long computeMaxSize2Move(long capacity, long remaining, double utilizationDiff, double threshold) {
        double diff = Math.min(threshold, Math.abs(utilizationDiff));
        long maxSizeToMove = Balancer.precentage2bytes(diff, capacity);
        if (utilizationDiff < 0.0) {
            maxSizeToMove = Math.min(remaining, maxSizeToMove);
        }
        return Math.min(0x280000000L, maxSizeToMove);
    }

    private static long precentage2bytes(double precentage, long capacity) {
        Preconditions.checkArgument((precentage >= 0.0 ? 1 : 0) != 0, (Object)("precentage = " + precentage + " < 0"));
        return (long)(precentage * (double)capacity / 100.0);
    }

    private void logUtilizationCollections() {
        Balancer.logUtilizationCollection("over-utilized", this.overUtilized);
        if (LOG.isTraceEnabled()) {
            Balancer.logUtilizationCollection("above-average", this.aboveAvgUtilized);
            Balancer.logUtilizationCollection("below-average", this.belowAvgUtilized);
        }
        Balancer.logUtilizationCollection("underutilized", this.underUtilized);
    }

    private static <T extends Dispatcher.DDatanode.StorageGroup> void logUtilizationCollection(String name, Collection<T> items) {
        LOG.info((Object)(items.size() + " " + name + ": " + items));
    }

    private long chooseStorageGroups() {
        if (this.dispatcher.getCluster().isNodeGroupAware()) {
            this.chooseStorageGroups(Matcher.SAME_NODE_GROUP);
        }
        this.chooseStorageGroups(Matcher.SAME_RACK);
        this.chooseStorageGroups(Matcher.ANY_OTHER);
        return this.dispatcher.bytesToMove();
    }

    private void chooseStorageGroups(Matcher matcher) {
        this.chooseStorageGroups(this.overUtilized, this.underUtilized, matcher);
        this.chooseStorageGroups(this.overUtilized, this.belowAvgUtilized, matcher);
        this.chooseStorageGroups(this.underUtilized, this.aboveAvgUtilized, matcher);
    }

    private <G extends Dispatcher.DDatanode.StorageGroup, C extends Dispatcher.DDatanode.StorageGroup> void chooseStorageGroups(Collection<G> groups, Collection<C> candidates, Matcher matcher) {
        Iterator<G> i = groups.iterator();
        while (i.hasNext()) {
            Dispatcher.DDatanode.StorageGroup g = (Dispatcher.DDatanode.StorageGroup)i.next();
            while (this.choose4One(g, candidates, matcher)) {
            }
            if (g.hasSpaceForScheduling()) continue;
            i.remove();
        }
    }

    private <C extends Dispatcher.DDatanode.StorageGroup> boolean choose4One(Dispatcher.DDatanode.StorageGroup g, Collection<C> candidates, Matcher matcher) {
        Iterator<C> i = candidates.iterator();
        C chosen = this.chooseCandidate(g, i, matcher);
        if (chosen == null) {
            return false;
        }
        if (g instanceof Dispatcher.Source) {
            this.matchSourceWithTargetToMove((Dispatcher.Source)g, (Dispatcher.DDatanode.StorageGroup)chosen);
        } else {
            this.matchSourceWithTargetToMove((Dispatcher.Source)chosen, g);
        }
        if (!((Dispatcher.DDatanode.StorageGroup)chosen).hasSpaceForScheduling()) {
            i.remove();
        }
        return true;
    }

    private void matchSourceWithTargetToMove(Dispatcher.Source source, Dispatcher.DDatanode.StorageGroup target) {
        long size = Math.min(source.availableSizeToMove(), target.availableSizeToMove());
        Dispatcher.Task task = new Dispatcher.Task(target, size);
        source.addTask(task);
        target.incScheduledSize(task.getSize());
        this.dispatcher.add(source, target);
        LOG.info((Object)("Decided to move " + StringUtils.byteDesc((long)size) + " bytes from " + source.getDisplayName() + " to " + target.getDisplayName()));
    }

    private <G extends Dispatcher.DDatanode.StorageGroup, C extends Dispatcher.DDatanode.StorageGroup> C chooseCandidate(G g, Iterator<C> candidates, Matcher matcher) {
        if (g.hasSpaceForScheduling()) {
            while (candidates.hasNext()) {
                Dispatcher.DDatanode.StorageGroup c = (Dispatcher.DDatanode.StorageGroup)candidates.next();
                if (!c.hasSpaceForScheduling()) {
                    candidates.remove();
                    continue;
                }
                if (!matcher.match(this.dispatcher.getCluster(), g.getDatanodeInfo(), c.getDatanodeInfo())) continue;
                return (C)c;
            }
        }
        return null;
    }

    private void resetData(Configuration conf) {
        this.overUtilized.clear();
        this.aboveAvgUtilized.clear();
        this.belowAvgUtilized.clear();
        this.underUtilized.clear();
        this.policy.reset();
        this.dispatcher.reset(conf);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ExitStatus run(int iteration, Formatter formatter, Configuration conf) {
        try {
            List<DatanodeStorageReport> reports = this.dispatcher.init();
            long bytesLeftToMove = this.init(reports);
            if (bytesLeftToMove == 0L) {
                System.out.println("The cluster is balanced. Exiting...");
                ExitStatus exitStatus = ExitStatus.SUCCESS;
                return exitStatus;
            }
            LOG.info((Object)("Need to move " + StringUtils.byteDesc((long)bytesLeftToMove) + " to make the cluster balanced."));
            long bytesToMove = this.chooseStorageGroups();
            if (bytesToMove == 0L) {
                System.out.println("No block can be moved. Exiting...");
                ExitStatus exitStatus = ExitStatus.NO_MOVE_BLOCK;
                return exitStatus;
            }
            LOG.info((Object)("Will move " + StringUtils.byteDesc((long)bytesToMove) + " in this iteration"));
            formatter.format("%-24s %10d  %19s  %18s  %17s%n", DateFormat.getDateTimeInstance().format(new Date()), iteration, StringUtils.byteDesc((long)this.dispatcher.getBytesMoved()), StringUtils.byteDesc((long)bytesLeftToMove), StringUtils.byteDesc((long)bytesToMove));
            if (!this.dispatcher.dispatchAndCheckContinue()) {
                ExitStatus exitStatus = ExitStatus.NO_MOVE_PROGRESS;
                return exitStatus;
            }
            ExitStatus exitStatus = ExitStatus.IN_PROGRESS;
            return exitStatus;
        }
        catch (IllegalArgumentException e) {
            System.out.println(e + ".  Exiting ...");
            ExitStatus exitStatus = ExitStatus.ILLEGAL_ARGUMENTS;
            return exitStatus;
        }
        catch (IOException e) {
            System.out.println(e + ".  Exiting ...");
            ExitStatus exitStatus = ExitStatus.IO_EXCEPTION;
            return exitStatus;
        }
        catch (InterruptedException e) {
            System.out.println(e + ".  Exiting ...");
            ExitStatus exitStatus = ExitStatus.INTERRUPTED;
            return exitStatus;
        }
        finally {
            this.dispatcher.shutdownNow();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static int run(Collection<URI> namenodes, Parameters p, Configuration conf) throws IOException, InterruptedException {
        long sleeptime = 2000L * conf.getLong("dfs.heartbeat.interval", 3L);
        LOG.info((Object)("namenodes  = " + namenodes));
        LOG.info((Object)("parameters = " + p));
        Formatter formatter = new Formatter(System.out);
        System.out.println("Time Stamp               Iteration#  Bytes Already Moved  Bytes Left To Move  Bytes Being Moved");
        ArrayList<NameNodeConnector> connectors = new ArrayList<NameNodeConnector>(namenodes.size());
        try {
            for (URI uri : namenodes) {
                NameNodeConnector nnc = new NameNodeConnector(Balancer.class.getSimpleName(), uri, BALANCER_ID_PATH, conf);
                nnc.getKeyManager().startBlockKeyUpdater();
                connectors.add(nnc);
            }
            boolean done = false;
            int iteration = 0;
            while (!done) {
                done = true;
                Collections.shuffle(connectors);
                for (NameNodeConnector nnc : connectors) {
                    Balancer b = new Balancer(nnc, p, conf);
                    ExitStatus r = b.run(iteration, formatter, conf);
                    b.resetData(conf);
                    if (r == ExitStatus.IN_PROGRESS) {
                        done = false;
                        continue;
                    }
                    if (r == ExitStatus.SUCCESS) continue;
                    int n = r.getExitCode();
                    return n;
                }
                if (!done) {
                    Thread.sleep(sleeptime);
                }
                ++iteration;
            }
        }
        finally {
            for (NameNodeConnector nnc : connectors) {
                nnc.close();
            }
        }
        return ExitStatus.SUCCESS.getExitCode();
    }

    private static String time2Str(long elapsedTime) {
        String unit;
        double time = elapsedTime;
        if (elapsedTime < 1000L) {
            unit = "milliseconds";
        } else if (elapsedTime < 60000L) {
            unit = "seconds";
            time /= 1000.0;
        } else if (elapsedTime < 3600000L) {
            unit = "minutes";
            time /= 60000.0;
        } else {
            unit = "hours";
            time /= 3600000.0;
        }
        return time + " " + unit;
    }

    public static void main(String[] args) {
        if (DFSUtil.parseHelpArgument(args, USAGE, System.out, true)) {
            System.exit(0);
        }
        try {
            System.exit(ToolRunner.run((Configuration)new HdfsConfiguration(), (Tool)new Cli(), (String[])args));
        }
        catch (Throwable e) {
            LOG.error((Object)"Exiting balancer due an exception", e);
            System.exit(-1);
        }
    }

    static class Cli
    extends Configured
    implements Tool {
        Cli() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Loose catch block
         */
        public int run(String[] args) {
            int n;
            long startTime = Time.now();
            Configuration conf = this.getConf();
            try {
                Balancer.checkReplicationPolicyCompatibility(conf);
                Collection<URI> namenodes = DFSUtil.getNsServiceRpcUris(conf);
                n = Balancer.run(namenodes, Cli.parse(args), conf);
            }
            catch (IOException e) {
                System.out.println(e + ".  Exiting ...");
                int n2 = ExitStatus.IO_EXCEPTION.getExitCode();
                System.out.format("%-24s ", DateFormat.getDateTimeInstance().format(new Date()));
                System.out.println("Balancing took " + Balancer.time2Str(Time.now() - startTime));
                return n2;
            }
            catch (InterruptedException e2) {
                System.out.println(e2 + ".  Exiting ...");
                int n3 = ExitStatus.INTERRUPTED.getExitCode();
                {
                    catch (Throwable throwable) {
                        System.out.format("%-24s ", DateFormat.getDateTimeInstance().format(new Date()));
                        System.out.println("Balancing took " + Balancer.time2Str(Time.now() - startTime));
                        throw throwable;
                    }
                }
                System.out.format("%-24s ", DateFormat.getDateTimeInstance().format(new Date()));
                System.out.println("Balancing took " + Balancer.time2Str(Time.now() - startTime));
                return n3;
            }
            System.out.format("%-24s ", DateFormat.getDateTimeInstance().format(new Date()));
            System.out.println("Balancing took " + Balancer.time2Str(Time.now() - startTime));
            return n;
        }

        static Parameters parse(String[] args) {
            BalancingPolicy policy = Parameters.DEFAULT.policy;
            double threshold = Parameters.DEFAULT.threshold;
            Set<String> nodesTobeExcluded = Parameters.DEFAULT.nodesToBeExcluded;
            Set<String> nodesTobeIncluded = Parameters.DEFAULT.nodesToBeIncluded;
            if (args != null) {
                try {
                    for (int i = 0; i < args.length; ++i) {
                        if ("-threshold".equalsIgnoreCase(args[i])) {
                            Preconditions.checkArgument((++i < args.length ? 1 : 0) != 0, (Object)("Threshold value is missing: args = " + Arrays.toString(args)));
                            try {
                                threshold = Double.parseDouble(args[i]);
                                if (threshold < 1.0 || threshold > 100.0) {
                                    throw new IllegalArgumentException("Number out of range: threshold = " + threshold);
                                }
                                LOG.info((Object)("Using a threshold of " + threshold));
                                continue;
                            }
                            catch (IllegalArgumentException e) {
                                System.err.println("Expecting a number in the range of [1.0, 100.0]: " + args[i]);
                                throw e;
                            }
                        }
                        if ("-policy".equalsIgnoreCase(args[i])) {
                            Preconditions.checkArgument((++i < args.length ? 1 : 0) != 0, (Object)("Policy value is missing: args = " + Arrays.toString(args)));
                            try {
                                policy = BalancingPolicy.parse(args[i]);
                                continue;
                            }
                            catch (IllegalArgumentException e) {
                                System.err.println("Illegal policy name: " + args[i]);
                                throw e;
                            }
                        }
                        if ("-exclude".equalsIgnoreCase(args[i])) {
                            Preconditions.checkArgument((++i < args.length ? 1 : 0) != 0, (Object)("List of nodes to exclude | -f <filename> is missing: args = " + Arrays.toString(args)));
                            if ("-f".equalsIgnoreCase(args[i])) {
                                Preconditions.checkArgument((++i < args.length ? 1 : 0) != 0, (Object)("File containing nodes to exclude is not specified: args = " + Arrays.toString(args)));
                                nodesTobeExcluded = Dispatcher.Util.getHostListFromFile(args[i], "exclude");
                                continue;
                            }
                            nodesTobeExcluded = Dispatcher.Util.parseHostList(args[i]);
                            continue;
                        }
                        if ("-include".equalsIgnoreCase(args[i])) {
                            Preconditions.checkArgument((++i < args.length ? 1 : 0) != 0, (Object)("List of nodes to include | -f <filename> is missing: args = " + Arrays.toString(args)));
                            if ("-f".equalsIgnoreCase(args[i])) {
                                Preconditions.checkArgument((++i < args.length ? 1 : 0) != 0, (Object)("File containing nodes to include is not specified: args = " + Arrays.toString(args)));
                                nodesTobeIncluded = Dispatcher.Util.getHostListFromFile(args[i], "include");
                                continue;
                            }
                            nodesTobeIncluded = Dispatcher.Util.parseHostList(args[i]);
                            continue;
                        }
                        throw new IllegalArgumentException("args = " + Arrays.toString(args));
                    }
                    Preconditions.checkArgument((nodesTobeExcluded.isEmpty() || nodesTobeIncluded.isEmpty() ? 1 : 0) != 0, (Object)"-exclude and -include options cannot be specified together.");
                }
                catch (RuntimeException e) {
                    Cli.printUsage(System.err);
                    throw e;
                }
            }
            return new Parameters(policy, threshold, nodesTobeExcluded, nodesTobeIncluded);
        }

        private static void printUsage(PrintStream out) {
            out.println(USAGE + "\n");
        }
    }

    static class Parameters {
        static final Parameters DEFAULT = new Parameters(BalancingPolicy.Node.INSTANCE, 10.0, Collections.<String>emptySet(), Collections.<String>emptySet());
        final BalancingPolicy policy;
        final double threshold;
        Set<String> nodesToBeExcluded;
        Set<String> nodesToBeIncluded;

        Parameters(BalancingPolicy policy, double threshold, Set<String> nodesToBeExcluded, Set<String> nodesToBeIncluded) {
            this.policy = policy;
            this.threshold = threshold;
            this.nodesToBeExcluded = nodesToBeExcluded;
            this.nodesToBeIncluded = nodesToBeIncluded;
        }

        public String toString() {
            return Balancer.class.getSimpleName() + "." + this.getClass().getSimpleName() + "[" + this.policy + ", threshold=" + this.threshold + ", number of nodes to be excluded = " + this.nodesToBeExcluded.size() + ", number of nodes to be included = " + this.nodesToBeIncluded.size() + "]";
        }
    }
}

