/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.mob;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.mob.CachedMobFile;
import org.apache.hadoop.hbase.mob.MobCacheConfig;
import org.apache.hadoop.hbase.mob.MobFile;
import org.apache.hadoop.hbase.util.IdLock;

@InterfaceAudience.Private
public class MobFileCache {
    private static final Log LOG = LogFactory.getLog(MobFileCache.class);
    private Map<String, CachedMobFile> map = null;
    private final AtomicLong count;
    private long lastAccess;
    private final AtomicLong miss;
    private long lastMiss;
    private final ReentrantLock evictionLock = new ReentrantLock(true);
    private final IdLock keyLock = new IdLock();
    private final ScheduledExecutorService scheduleThreadPool = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("MobFileCache #%d").setDaemon(true).build());
    private final Configuration conf;
    private final int mobFileMaxCacheSize;
    private final boolean isCacheEnabled;
    private float evictRemainRatio;

    public MobFileCache(Configuration conf) {
        this.conf = conf;
        this.mobFileMaxCacheSize = conf.getInt("hbase.mob.file.cache.size", 1000);
        this.isCacheEnabled = this.mobFileMaxCacheSize > 0;
        this.map = new ConcurrentHashMap<String, CachedMobFile>(this.mobFileMaxCacheSize);
        this.count = new AtomicLong(0L);
        this.miss = new AtomicLong(0L);
        this.lastAccess = 0L;
        this.lastMiss = 0L;
        if (this.isCacheEnabled) {
            long period = conf.getLong("hbase.mob.cache.evict.period", 3600L);
            this.evictRemainRatio = conf.getFloat("hbase.mob.cache.evict.remain.ratio", 0.5f);
            if ((double)this.evictRemainRatio < 0.0) {
                this.evictRemainRatio = 0.0f;
                LOG.warn((Object)"hbase.mob.cache.evict.remain.ratio is less than 0.0, 0.0 is used.");
            } else if ((double)this.evictRemainRatio > 1.0) {
                this.evictRemainRatio = 1.0f;
                LOG.warn((Object)"hbase.mob.cache.evict.remain.ratio is larger than 1.0, 1.0 is used.");
            }
            this.scheduleThreadPool.scheduleAtFixedRate(new EvictionThread(this), period, period, TimeUnit.SECONDS);
        }
        LOG.info((Object)("MobFileCache is initialized, and the cache size is " + this.mobFileMaxCacheSize));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void evict() {
        if (this.isCacheEnabled) {
            if (!this.evictionLock.tryLock()) {
                return;
            }
            this.printStatistics();
            ArrayList<CachedMobFile> evictedFiles = new ArrayList<CachedMobFile>();
            try {
                if (this.map.size() <= this.mobFileMaxCacheSize) {
                    return;
                }
                ArrayList<CachedMobFile> files = new ArrayList<CachedMobFile>(this.map.values());
                Collections.sort(files);
                int start = (int)((float)this.mobFileMaxCacheSize * this.evictRemainRatio);
                if (start >= 0) {
                    for (int i = start; i < files.size(); ++i) {
                        String name = ((CachedMobFile)files.get(i)).getFileName();
                        CachedMobFile evictedFile = this.map.remove(name);
                        if (evictedFile == null) continue;
                        evictedFiles.add(evictedFile);
                    }
                }
            }
            finally {
                this.evictionLock.unlock();
            }
            for (CachedMobFile evictedFile : evictedFiles) {
                this.closeFile(evictedFile);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void evictFile(String fileName) {
        if (this.isCacheEnabled) {
            IdLock.Entry lockEntry = null;
            try {
                lockEntry = this.keyLock.getLockEntry(fileName.hashCode());
                CachedMobFile evictedFile = this.map.remove(fileName);
                if (evictedFile != null) {
                    evictedFile.close();
                }
            }
            catch (IOException e) {
                LOG.error((Object)("Fail to evict the file " + fileName), (Throwable)e);
            }
            finally {
                if (lockEntry != null) {
                    this.keyLock.releaseLockEntry(lockEntry);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MobFile openFile(FileSystem fs, Path path, MobCacheConfig cacheConf) throws IOException {
        if (!this.isCacheEnabled) {
            return MobFile.create(fs, path, this.conf, cacheConf);
        }
        String fileName = path.getName();
        CachedMobFile cached = this.map.get(fileName);
        IdLock.Entry lockEntry = this.keyLock.getLockEntry(fileName.hashCode());
        try {
            if (cached == null && (cached = this.map.get(fileName)) == null) {
                if (this.map.size() > this.mobFileMaxCacheSize) {
                    this.evict();
                }
                cached = CachedMobFile.create(fs, path, this.conf, cacheConf);
                cached.open();
                this.map.put(fileName, cached);
                this.miss.incrementAndGet();
            }
            cached.open();
            cached.access(this.count.incrementAndGet());
        }
        finally {
            this.keyLock.releaseLockEntry(lockEntry);
        }
        return cached;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void closeFile(MobFile file) {
        IdLock.Entry lockEntry = null;
        try {
            if (!this.isCacheEnabled) {
                file.close();
            } else {
                lockEntry = this.keyLock.getLockEntry(file.getFileName().hashCode());
                file.close();
            }
            if (lockEntry != null) {
                this.keyLock.releaseLockEntry(lockEntry);
            }
        }
        catch (IOException e) {
            try {
                LOG.error((Object)("MobFileCache, Exception happen during close " + file.getFileName()), (Throwable)e);
                if (lockEntry != null) {
                    this.keyLock.releaseLockEntry(lockEntry);
                }
            }
            catch (Throwable throwable) {
                if (lockEntry != null) {
                    this.keyLock.releaseLockEntry(lockEntry);
                }
                throw throwable;
            }
        }
    }

    public int getCacheSize() {
        return this.map == null ? 0 : this.map.size();
    }

    public void printStatistics() {
        long access = this.count.get() - this.lastAccess;
        long missed = this.miss.get() - this.lastMiss;
        long hitRate = (access - missed) * 100L / access;
        LOG.info((Object)("MobFileCache Statistics, access: " + access + ", miss: " + missed + ", hit: " + (access - missed) + ", hit rate: " + (access == 0L ? 0L : hitRate) + "%"));
        this.lastAccess += access;
        this.lastMiss += missed;
    }

    static class EvictionThread
    extends Thread {
        MobFileCache lru;

        public EvictionThread(MobFileCache lru) {
            super("MobFileCache.EvictionThread");
            this.setDaemon(true);
            this.lru = lru;
        }

        @Override
        public void run() {
            this.lru.evict();
        }
    }
}

